<?php
require_once dirname(__DIR__) . '/config.php';

$message = '';
$error = '';

// Get filter parameters
$year = $_GET['year'] ?? date('Y');

// Calculate year boundaries
$start_date = $year . '-01-01';
$end_date = $year . '-12-31';
$prev_year = $year - 1;
$prev_start_date = $prev_year . '-01-01';
$prev_end_date = $prev_year . '-12-31';

// Initialize variables
$current_year_data = [];
$previous_year_data = [];
$changes = [];

try {
    // Get current year balances (end of year)
    $stmt = $pdo->prepare("
        SELECT
            coa.account_type,
            coa.account_code,
            coa.account_name,
            COALESCE(
                SUM(
                    CASE
                        WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                        ELSE jel.credit_amount - jel.debit_amount
                    END
                ), 0
            ) as balance
        FROM chart_of_accounts coa
        LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
        LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date <= ?
        WHERE coa.is_active = TRUE
        GROUP BY coa.account_type, coa.account_code, coa.account_name
        ORDER BY coa.account_code
    ");
    $stmt->execute([$end_date]);
    $current_year_raw = $stmt->fetchAll();

    // Get previous year balances (end of previous year)
    $stmt->execute([$prev_end_date]);
    $previous_year_raw = $stmt->fetchAll();

    // Aggregate by account type
    $current_year_data = [
        'assets' => 0,
        'liabilities' => 0,
        'equity' => 0,
        'revenue' => 0,
        'expense' => 0
    ];

    $previous_year_data = [
        'assets' => 0,
        'liabilities' => 0,
        'equity' => 0,
        'revenue' => 0,
        'expense' => 0
    ];

    foreach ($current_year_raw as $account) {
        $type = strtolower($account['account_type']);
        if ($type === 'asset') $type = 'assets';
        elseif ($type === 'liability') $type = 'liabilities';
        if (isset($current_year_data[$type])) {
            $current_year_data[$type] += $account['balance'];
        }
    }

    foreach ($previous_year_raw as $account) {
        $type = strtolower($account['account_type']);
        if ($type === 'asset') $type = 'assets';
        elseif ($type === 'liability') $type = 'liabilities';
        if (isset($previous_year_data[$type])) {
            $previous_year_data[$type] += $account['balance'];
        }
    }

    // Calculate net income for current year
    $current_net_income = $current_year_data['revenue'] - $current_year_data['expense'];

    // Calculate net income for previous year
    $previous_net_income = $previous_year_data['revenue'] - $previous_year_data['expense'];

    // Adjust equity for net income
    $current_year_data['equity'] += $current_net_income;
    $previous_year_data['equity'] += $previous_net_income;

    // Calculate changes
    $changes = [
        'assets' => $current_year_data['assets'] - $previous_year_data['assets'],
        'liabilities' => $current_year_data['liabilities'] - $previous_year_data['liabilities'],
        'equity' => $current_year_data['equity'] - $previous_year_data['equity'],
        'net_income' => $current_net_income - $previous_net_income
    ];

    // Beginning balances for the current year are the ending balances of the previous year
    $beginning_balances = $previous_year_data;

} catch (PDOException $e) {
    $error = "Error fetching year-end summary data: " . $e->getMessage();
}

// Fetch school settings for print functionality
$school_settings = $pdo->query("SELECT school_name, address, phone, email, logo_path FROM school_settings LIMIT 1")->fetch(PDO::FETCH_ASSOC);

include 'includes/accounting_header.php';
?>

<div class="year-end-summary-container">
    <div class="report-header">
        <h1>📊 Year-End Summary Report</h1>
        <div class="report-filters">
            <form method="get" class="filter-form">
                <div class="filter-group">
                    <label for="year">Year:</label>
                    <select name="year" id="year" onchange="this.form.submit()">
                        <?php for ($y = date('Y'); $y >= date('Y') - 5; $y--): ?>
                            <option value="<?php echo $y; ?>" <?php echo $year == $y ? 'selected' : ''; ?>><?php echo $y; ?></option>
                        <?php endfor; ?>
                    </select>
                </div>

                <a href="export_year_end_summary.php?<?php echo http_build_query($_GET); ?>" class="btn btn-secondary">Export</a>
                <button type="button" onclick="printReport()" class="btn btn-primary">Print Report</button>
            </form>
        </div>
    </div>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Year-End Summary -->
    <div class="report-section">
        <h2>Year-End Summary: <?php echo $year; ?> vs <?php echo $prev_year; ?></h2>

        <div class="summary-grid">
            <!-- Beginning Balances -->
            <div class="summary-card">
                <h3>Beginning Balances (<?php echo $year; ?>)</h3>
                <div class="balance-item">
                    <span class="label">Assets:</span>
                    <span class="amount">GHC <?php echo number_format($beginning_balances['assets'], 2); ?></span>
                </div>
                <div class="balance-item">
                    <span class="label">Liabilities:</span>
                    <span class="amount">GHC <?php echo number_format($beginning_balances['liabilities'], 2); ?></span>
                </div>
                <div class="balance-item">
                    <span class="label">Equity:</span>
                    <span class="amount">GHC <?php echo number_format($beginning_balances['equity'], 2); ?></span>
                </div>
                <hr>
                <div class="balance-item total">
                    <span class="label">Total:</span>
                    <span class="amount">GHC <?php echo number_format($beginning_balances['assets'] - $beginning_balances['liabilities'] - $beginning_balances['equity'], 2); ?></span>
                </div>
            </div>

            <!-- Changes During Year -->
            <div class="summary-card">
                <h3>Changes During <?php echo $year; ?></h3>
                <div class="balance-item">
                    <span class="label">Assets:</span>
                    <span class="amount <?php echo $changes['assets'] >= 0 ? 'positive' : 'negative'; ?>">
                        GHC <?php echo number_format($changes['assets'], 2); ?>
                    </span>
                </div>
                <div class="balance-item">
                    <span class="label">Liabilities:</span>
                    <span class="amount <?php echo $changes['liabilities'] >= 0 ? 'positive' : 'negative'; ?>">
                        GHC <?php echo number_format($changes['liabilities'], 2); ?>
                    </span>
                </div>
                <div class="balance-item">
                    <span class="label">Equity (Net Income):</span>
                    <span class="amount <?php echo $changes['equity'] >= 0 ? 'positive' : 'negative'; ?>">
                        GHC <?php echo number_format($changes['equity'], 2); ?>
                    </span>
                </div>
                <hr>
                <div class="balance-item total">
                    <span class="label">Net Change:</span>
                    <span class="amount <?php echo ($changes['assets'] - $changes['liabilities'] - $changes['equity']) >= 0 ? 'positive' : 'negative'; ?>">
            </div>

            <!-- Ending Balances -->
            <div class="summary-card">
                <h3>Ending Balances (<?php echo $year; ?>)</h3>
                <div class="balance-item">
                    <span class="label">Assets:</span>
                    <span class="amount">GHC <?php echo number_format($current_year_data['assets'], 2); ?></span>
                </div>
                <div class="balance-item">
                    <span class="label">Liabilities:</span>
                    <span class="amount">GHC <?php echo number_format($current_year_data['liabilities'], 2); ?></span>
                </div>
                <div class="balance-item">
                    <span class="label">Equity:</span>
                    <span class="amount">GHC <?php echo number_format($current_year_data['equity'], 2); ?></span>
                </div>
                <hr>
                <div class="balance-item total">
                    <span class="label">Total:</span>
                    <span class="amount">GHC <?php echo number_format($current_year_data['assets'] - $current_year_data['liabilities'] - $current_year_data['equity'], 2); ?></span>
                </div>
            </div>
        </div>
    </div>

    <!-- Year-over-Year Comparison -->
    <div class="report-section">
        <h2>Year-over-Year Comparison</h2>
        <div class="comparison-table">
            <table class="bs-table">
                <thead>
                    <tr>
                        <th>Category</th>
                        <th><?php echo $prev_year; ?> Ending</th>
                        <th><?php echo $year; ?> Ending</th>
                        <th>Change</th>
                        <th>% Change</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><strong>Assets</strong></td>
                        <td>GHC <?php echo number_format($previous_year_data['assets'], 2); ?></td>
                        <td>GHC <?php echo number_format($current_year_data['assets'], 2); ?></td>
                        <td class="<?php echo $changes['assets'] >= 0 ? 'positive' : 'negative'; ?>">
                            GHC <?php echo number_format($changes['assets'], 2); ?>
                        </td>
                        <td class="<?php echo $previous_year_data['assets'] != 0 && $changes['assets'] >= 0 ? 'positive' : 'negative'; ?>">
                            <?php echo $previous_year_data['assets'] != 0 ? number_format(($changes['assets'] / $previous_year_data['assets']) * 100, 1) . '%' : 'N/A'; ?>
                        </td>
                    </tr>
                    <tr>
                        <td><strong>Liabilities</strong></td>
                        <td>GHC <?php echo number_format($previous_year_data['liabilities'], 2); ?></td>
                        <td>GHC <?php echo number_format($current_year_data['liabilities'], 2); ?></td>
                        <td class="<?php echo $changes['liabilities'] >= 0 ? 'positive' : 'negative'; ?>">
                            GHC <?php echo number_format($changes['liabilities'], 2); ?>
                        </td>
                        <td class="<?php echo $previous_year_data['liabilities'] != 0 && $changes['liabilities'] >= 0 ? 'positive' : 'negative'; ?>">
                            <?php echo $previous_year_data['liabilities'] != 0 ? number_format(($changes['liabilities'] / $previous_year_data['liabilities']) * 100, 1) . '%' : 'N/A'; ?>
                        </td>
                    </tr>
                    <tr>
                        <td><strong>Equity</strong></td>
                        <td>GHC <?php echo number_format($previous_year_data['equity'], 2); ?></td>
                        <td>GHC <?php echo number_format($current_year_data['equity'], 2); ?></td>
                        <td class="<?php echo $changes['equity'] >= 0 ? 'positive' : 'negative'; ?>">
                            GHC <?php echo number_format($changes['equity'], 2); ?>
                        </td>
                        <td class="<?php echo $previous_year_data['equity'] != 0 && $changes['equity'] >= 0 ? 'positive' : 'negative'; ?>">
                            <?php echo $previous_year_data['equity'] != 0 ? number_format(($changes['equity'] / $previous_year_data['equity']) * 100, 1) . '%' : 'N/A'; ?>
                        </td>
                    </tr>
                    <tr class="total-row">
                        <td><strong>Net Assets</strong></td>
                        <td>GHC <?php echo number_format($previous_year_data['assets'] - $previous_year_data['liabilities'] - $previous_year_data['equity'], 2); ?></td>
                        <td>GHC <?php echo number_format($current_year_data['assets'] - $current_year_data['liabilities'] - $current_year_data['equity'], 2); ?></td>
                        <td class="<?php echo ($changes['assets'] - $changes['liabilities'] - $changes['equity']) >= 0 ? 'positive' : 'negative'; ?>">
                            GHC <?php echo number_format($changes['assets'] - $changes['liabilities'] - $changes['equity'], 2); ?>
                        </td>
                        <td>N/A</td>
                    </tr>
                    <tr>
                        <td><strong>Net Income</strong></td>
                        <td>GHC <?php echo number_format($previous_net_income, 2); ?></td>
                        <td>GHC <?php echo number_format($current_net_income, 2); ?></td>
                        <td class="<?php echo $changes['net_income'] >= 0 ? 'positive' : 'negative'; ?>">
                            GHC <?php echo number_format($changes['net_income'], 2); ?>
                        </td>
                        <td class="<?php echo $previous_net_income != 0 && $changes['net_income'] >= 0 ? 'positive' : 'negative'; ?>">
                            <?php echo $previous_net_income != 0 ? number_format(($changes['net_income'] / $previous_net_income) * 100, 1) . '%' : 'N/A'; ?>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Financial Health Trends -->
    <div class="report-section">
        <h2>📈 Financial Health Trends</h2>
        <div class="analysis-grid">
            <div class="analysis-card">
                <h4>Growth Rate</h4>
                <p class="<?php echo $changes['assets'] >= 0 ? 'positive' : 'negative'; ?>">
                    <?php echo $previous_year_data['assets'] != 0 ? number_format(($changes['assets'] / $previous_year_data['assets']) * 100, 1) . '%' : 'N/A'; ?>
                </p>
                <small>Asset growth from <?php echo $prev_year; ?> to <?php echo $year; ?></small>
            </div>

            <div class="analysis-card">
                <h4>Profitability</h4>
                <p class="<?php echo $current_net_income >= 0 ? 'positive' : 'negative'; ?>">
                    <?php echo $current_year_data['assets'] != 0 ? number_format(($current_net_income / $current_year_data['assets']) * 100, 1) . '%' : 'N/A'; ?>
                </p>
                <small>Return on assets (ROA) for <?php echo $year; ?></small>
            </div>

            <div class="analysis-card">
                <h4>Leverage</h4>
                <p class="<?php echo $current_year_data['assets'] > 0 ? 'neutral' : 'warning'; ?>">
                    <?php echo $current_year_data['assets'] > 0 ? number_format(($current_year_data['liabilities'] / $current_year_data['assets']) * 100, 1) . '%' : 'N/A'; ?>
                </p>
                <small>Debt-to-asset ratio for <?php echo $year; ?></small>
            </div>

            <div class="analysis-card">
                <h4>Equity Growth</h4>
                <p class="<?php echo $changes['equity'] >= 0 ? 'positive' : 'negative'; ?>">
                    <?php echo $previous_year_data['equity'] != 0 ? number_format(($changes['equity'] / $previous_year_data['equity']) * 100, 1) . '%' : 'N/A'; ?>
                </p>
                <small>Equity growth from <?php echo $prev_year; ?> to <?php echo $year; ?></small>
            </div>
        </div>
    </div>
</div>

<style>
.year-end-summary-container {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.report-header {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.report-header h1 {
    color: #2c3e50;
    margin: 0 0 20px 0;
}

.filter-form {
    display: flex;
    gap: 20px;
    align-items: end;
    flex-wrap: wrap;
}

.filter-group {
    display: flex;
    flex-direction: column;
    min-width: 150px;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.filter-group select {
    padding: 8px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.report-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.report-section h2 {
    color: #2c3e50;
    margin-bottom: 20px;
    border-bottom: 2px solid #ecf0f1;
    padding-bottom: 10px;
}

.summary-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.summary-card {
    background: #f8f9fa;
    padding: 20px;
    border-radius: 8px;
    border: 1px solid #dee2e6;
}

.summary-card h3 {
    color: #2c3e50;
    margin-bottom: 15px;
    font-size: 1.1em;
    text-align: center;
}

.balance-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 8px 0;
    border-bottom: 1px solid #e9ecef;
}

.balance-item:last-child {
    border-bottom: none;
}

.balance-item.total {
    border-top: 2px solid #6c757d;
    font-weight: bold;
    margin-top: 10px;
    padding-top: 15px;
}

.balance-item .label {
    font-weight: 500;
    color: #495057;
}

.balance-item .amount {
    font-weight: 600;
    color: #2c3e50;
}

.positive {
    color: #27ae60;
}

.negative {
    color: #e74c3c;
}

.comparison-table {
    overflow-x: auto;
}

.bs-table {
    width: 100%;
    border-collapse: collapse;
    margin-bottom: 20px;
}

.bs-table th,
.bs-table td {
    padding: 12px;
    text-align: left;
    border-bottom: 1px solid #ecf0f1;
}

.bs-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.total-row {
    background: #f8f9fa;
    font-weight: 600;
}

.analysis-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
}

.analysis-card {
    background: white;
    padding: 20px;
    border-radius: 8px;
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    border-left: 4px solid #3498db;
    text-align: center;
}

.analysis-card h4 {
    color: #2c3e50;
    margin: 0 0 10px 0;
    font-size: 1.1em;
}

.analysis-card p {
    font-size: 1.4em;
    font-weight: 600;
    margin: 10px 0;
}

.analysis-card small {
    color: #7f8c8d;
    font-size: 0.9em;
}

.neutral { color: #2c3e50; }
.warning { color: #f39c12; }

.btn {
    display: inline-block;
    padding: 10px 20px;
    background: #3498db;
    color: white;
    text-decoration: none;
    border-radius: 5px;
    transition: background 0.3s;
}

.btn:hover {
    background: #2980b9;
}

.btn-secondary {
    background: #95a5a6;
}

.btn-secondary:hover {
    background: #7f8c8d;
}

.error-message {
    background: #f8d7da;
    color: #721c24;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #f5c6cb;
}

@media (max-width: 768px) {
    .summary-grid,
    .analysis-grid {
        grid-template-columns: 1fr;
    }

    .filter-form {
        flex-direction: column;
        align-items: stretch;
    }
}
</style>

<script>
// Print Report Function
function printReport() {
    // Create a new window for printing
    var printWindow = window.open('', '_blank');

    // Build the print content
    var printContent = `
        <!DOCTYPE html>
        <html>
        <head>
            <title>Year-End Summary Report</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    margin: 20px;
                    line-height: 1.4;
                }
                .header {
                    text-align: center;
                    border-bottom: 2px solid #333;
                    padding-bottom: 20px;
                    margin-bottom: 30px;
                }
                .school-info {
                    margin-bottom: 20px;
                }
                .report-title {
                    font-size: 24px;
                    font-weight: bold;
                    margin: 20px 0;
                    color: #2c3e50;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 20px;
                    font-size: 12px;
                }
                th, td {
                    border: 1px solid #ddd;
                    padding: 8px 12px;
                    text-align: left;
                }
                th {
                    background-color: #f8f9fa;
                    font-weight: bold;
                    color: #2c3e50;
                }
                .positive {
                    color: #27ae60;
                    font-weight: bold;
                }
                .negative {
                    color: #e74c3c;
                    font-weight: bold;
                }
                .total-row {
                    background-color: #e9ecef;
                    font-weight: bold;
                }
                .footer {
                    margin-top: 40px;
                    text-align: center;
                    font-size: 12px;
                    color: #6c757d;
                    border-top: 1px solid #dee2e6;
                    padding-top: 20px;
                }
                @media print {
                    body { margin: 0; }
                    .no-print { display: none; }
                }
            </style>
        </head>
        <body>
            <div class="header">
                <div class="school-info">
                    <h1><?php echo htmlspecialchars($school_settings['school_name'] ?? 'School Name'); ?></h1>
                    <p><?php echo htmlspecialchars($school_settings['address'] ?? 'School Address'); ?></p>
                    <p>Phone: <?php echo htmlspecialchars($school_settings['phone'] ?? 'School Phone'); ?> | Email: <?php echo htmlspecialchars($school_settings['email'] ?? 'School Email'); ?></p>
                </div>
                <h2 class="report-title">Year-End Summary Report</h2>
                <p>Year: <?php echo $year; ?> vs <?php echo $prev_year; ?></p>
                <p>Generated on: <?php echo date('M d, Y'); ?> <?php echo date('H:i:s'); ?></p>
            </div>

            <h3>Year-End Summary</h3>
            <table>
                <thead>
                    <tr>
                        <th>Category</th>
                        <th>Beginning <?php echo $year; ?></th>
                        <th>Changes <?php echo $year; ?></th>
                        <th>Ending <?php echo $year; ?></th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><strong>Assets</strong></td>
                        <td><?php echo number_format($beginning_balances['assets'], 2); ?></td>
                        <td class="<?php echo $changes['assets'] >= 0 ? 'positive' : 'negative'; ?>"><?php echo number_format($changes['assets'], 2); ?></td>
                        <td><?php echo number_format($current_year_data['assets'], 2); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Liabilities</strong></td>
                        <td><?php echo number_format($beginning_balances['liabilities'], 2); ?></td>
                        <td class="<?php echo $changes['liabilities'] >= 0 ? 'positive' : 'negative'; ?>"><?php echo number_format($changes['liabilities'], 2); ?></td>
                        <td><?php echo number_format($current_year_data['liabilities'], 2); ?></td>
                    </tr>
                    <tr>
                        <td><strong>Equity</strong></td>
                        <td><?php echo number_format($beginning_balances['equity'], 2); ?></td>
                        <td class="<?php echo $changes['net_income'] >= 0 ? 'positive' : 'negative'; ?>"><?php echo number_format($changes['net_income'], 2); ?></td>
                        <td><?php echo number_format($current_year_data['equity'], 2); ?></td>
                    </tr>
                    <tr class="total-row">
                        <td><strong>Total</strong></td>
                        <td><?php echo number_format($beginning_balances['assets'] - $beginning_balances['liabilities'] - $beginning_balances['equity'], 2); ?></td>
                        <td><?php echo number_format($changes['assets'] - $changes['liabilities'] - $changes['net_income'], 2); ?></td>
                        <td><?php echo number_format($current_year_data['assets'] - $current_year_data['liabilities'] - $current_year_data['equity'], 2); ?></td>
                    </tr>
                </tbody>
            </table>

            <h3>Year-over-Year Comparison</h3>
            <table>
                <thead>
                    <tr>
                        <th>Category</th>
                        <th><?php echo $prev_year; ?> Ending</th>
                        <th><?php echo $year; ?> Ending</th>
                        <th>Change</th>
                        <th>% Change</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><strong>Assets</strong></td>
                        <td><?php echo number_format($previous_year_data['assets'], 2); ?></td>
                        <td><?php echo number_format($current_year_data['assets'], 2); ?></td>
                        <td class="<?php echo $changes['assets'] >= 0 ? 'positive' : 'negative'; ?>"><?php echo number_format($changes['assets'], 2); ?></td>
                        <td><?php echo $previous_year_data['assets'] != 0 ? number_format(($changes['assets'] / $previous_year_data['assets']) * 100, 1) . '%' : 'N/A'; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Liabilities</strong></td>
                        <td><?php echo number_format($previous_year_data['liabilities'], 2); ?></td>
                        <td><?php echo number_format($current_year_data['liabilities'], 2); ?></td>
                        <td class="<?php echo $changes['liabilities'] >= 0 ? 'positive' : 'negative'; ?>"><?php echo number_format($changes['liabilities'], 2); ?></td>
                        <td><?php echo $previous_year_data['liabilities'] != 0 ? number_format(($changes['liabilities'] / $previous_year_data['liabilities']) * 100, 1) . '%' : 'N/A'; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Equity</strong></td>
                        <td><?php echo number_format($previous_year_data['equity'], 2); ?></td>
                        <td><?php echo number_format($current_year_data['equity'], 2); ?></td>
                        <td class="<?php echo $changes['equity'] >= 0 ? 'positive' : 'negative'; ?>"><?php echo number_format($changes['equity'], 2); ?></td>
                        <td><?php echo $previous_year_data['equity'] != 0 ? number_format(($changes['equity'] / $previous_year_data['equity']) * 100, 1) . '%' : 'N/A'; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Net Income</strong></td>
                        <td><?php echo number_format($previous_net_income, 2); ?></td>
                        <td><?php echo number_format($current_net_income, 2); ?></td>
                        <td class="<?php echo $changes['net_income'] >= 0 ? 'positive' : 'negative'; ?>"><?php echo number_format($changes['net_income'], 2); ?></td>
                        <td><?php echo $previous_net_income != 0 ? number_format(($changes['net_income'] / $previous_net_income) * 100, 1) . '%' : 'N/A'; ?></td>
                    </tr>
                </tbody>
            </table>

            <div class="footer">
                <p>This report was generated by the School Management System</p>
                <p>Report Date: <?php echo date('M d, Y'); ?></p>
            </div>
        </body>
        </html>`;

    // Write content to the new window
    printWindow.document.write(printContent);
    printWindow.document.close();

    // Wait for content to load then print
    printWindow.onload = function() {
        printWindow.print();
        printWindow.close();
    };
}
</script>

<?php include 'includes/accounting_footer.php'; ?>
