<?php
require_once '../config.php';

$message = '';
$message_type = '';

// Get payroll ID from URL
$payroll_id = isset($_GET['payroll_id']) ? (int)$_GET['payroll_id'] : null;

// Get payroll details
if ($payroll_id) {
    $stmt = $pdo->prepare("SELECT * FROM payroll_runs WHERE id = ?");
    $stmt->execute([$payroll_id]);
    $payroll = $stmt->fetch();

    if (!$payroll) {
        $message = "Payroll not found.";
        $message_type = 'error';
    }
} else {
    // Get latest payroll if no specific ID
    $stmt = $pdo->query("SELECT * FROM payroll_runs ORDER BY created_at DESC LIMIT 1");
    $payroll = $stmt->fetch();
    if ($payroll) {
        $payroll_id = $payroll['id'];
    }
}

// Get payslips for this payroll
$payslips = [];
if ($payroll_id) {
    $stmt = $pdo->prepare("
        SELECT p.*,
               CONCAT(p.first_name, ' ', p.last_name) as staff_name,
               s.staff_id as staff_id_display
        FROM payslips p
        LEFT JOIN staff s ON p.staff_id = s.id
        WHERE p.payroll_run_id = ?
        ORDER BY staff_name
    ");
    $stmt->execute([$payroll_id]);
    $payslips = $stmt->fetchAll();
}

// Handle PDF generation
if (isset($_GET['generate_pdf']) && isset($_GET['payslip_id'])) {
    $payslip_id = (int)$_GET['payslip_id'];

    // Get payslip data
    $stmt = $pdo->prepare("
        SELECT p.*, pr.payroll_month,
               CONCAT(p.first_name, ' ', p.last_name) as teacher_name,
               s.staff_id as staff_id_display,
               pr.processed_at
        FROM payslips p
        JOIN payroll_runs pr ON p.payroll_run_id = pr.id
        LEFT JOIN staff s ON p.staff_id = s.id
        WHERE p.id = ?
    ");
    $stmt->execute([$payslip_id]);
    $payslip_data = $stmt->fetch();

    if ($payslip_data) {
        generatePayslipPDF($payslip_data);
        exit;
    }
}

function generatePayslipPDF($data) {
    require_once '../vendor/autoload.php';

    // Fetch school settings
    global $pdo;
    $schoolSettings = $pdo->query("SELECT school_name, address, phone, email, logo_path FROM school_settings LIMIT 1")->fetch(PDO::FETCH_ASSOC) ?? [];

    // Create new PDF document
    $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

    // Set document information
    $pdf->SetCreator($schoolSettings['school_name'] ?? 'School Management System');
    $pdf->SetAuthor('Accounts Department');
    $pdf->SetTitle('Employee Payslip - ' . $data['teacher_name']);
    $pdf->SetSubject('Payslip for ' . date('F Y', strtotime($data['payroll_month'])));

    // Remove default header/footer
    $pdf->setPrintHeader(false);
    $pdf->setPrintFooter(false);

    // Set margins
    $pdf->SetMargins(15, 15, 15);

    // Set auto page breaks
    $pdf->SetAutoPageBreak(TRUE, 15);

    // Add a page
    $pdf->AddPage();

    // Parse payslip data
    $payslipData = json_decode($data['payslip_data'], true) ?? [];

    // Fetch actual allowances from database
    $allowances = [];
    if ($data['staff_id']) {
        $stmt = $pdo->prepare("
            SELECT at.name, sa.amount
            FROM staff_allowances sa
            JOIN allowance_types at ON sa.allowance_type_id = at.id
            JOIN salary_structures ss ON sa.salary_structure_id = ss.id
            WHERE ss.staff_id = ? AND ss.is_active = 1 AND at.is_active = 1
        ");
        $stmt->execute([$data['staff_id']]);
        $allowanceRows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($allowanceRows as $row) {
            $type = strtolower($row['name']);
            if (strpos($type, 'housing') !== false) {
                $allowances['housing'] = $row['amount'];
            } elseif (strpos($type, 'transport') !== false) {
                $allowances['transport'] = $row['amount'];
            } elseif (strpos($type, 'utility') !== false) {
                $allowances['utility'] = $row['amount'];
            } else {
                $allowances['other'] = ($allowances['other'] ?? 0) + $row['amount'];
            }
        }
    }

    // Use database allowances if available, otherwise fall back to JSON
    $allowanceData = $allowances ?: ($payslipData['allowances'] ?? []);

    // Set font
    $pdf->SetFont('helvetica', '', 10);

    // School Header with Logo
    if (!empty($schoolSettings['logo_path']) && file_exists('../' . $schoolSettings['logo_path'])) {
        $pdf->Image('../' . $schoolSettings['logo_path'], 15, 15, 30, 0, '', '', '', false, 300, '', false, false, 0, false, false, false);
        $pdf->SetXY(50, 15);
    }

    // School Information
    $pdf->SetFont('helvetica', 'B', 16);
    $pdf->Cell(0, 8, $schoolSettings['school_name'] ?? 'School Management System', 0, 1, 'C');
    $pdf->SetFont('helvetica', '', 10);

    if (!empty($schoolSettings['address'])) {
        $pdf->Cell(0, 5, $schoolSettings['address'], 0, 1, 'C');
    }

    $contactInfo = [];
    if (!empty($schoolSettings['phone'])) $contactInfo[] = 'Phone: ' . $schoolSettings['phone'];
    if (!empty($schoolSettings['email'])) $contactInfo[] = 'Email: ' . $schoolSettings['email'];

    if (!empty($contactInfo)) {
        $pdf->Cell(0, 5, implode(' | ', $contactInfo), 0, 1, 'C');
    }

    $pdf->Ln(5);

    // Document Title
    $pdf->SetFont('helvetica', 'B', 14);
    $pdf->Cell(0, 10, 'EMPLOYEE PAYSLIP', 0, 1, 'C');
    $pdf->Ln(5);

    // Employee details
    $pdf->SetFont('helvetica', '', 10);
    $pdf->Cell(50, 8, 'Employee:', 0, 0);
    $pdf->Cell(0, 8, $data['teacher_name'], 0, 1);
    $pdf->Cell(50, 8, 'Staff ID:', 0, 0);
    $pdf->Cell(0, 8, $data['staff_id_display'], 0, 1);
    $pdf->Cell(50, 8, 'Pay Period:', 0, 0);
    $pdf->Cell(0, 8, date('F Y', strtotime($data['payroll_month'])), 0, 1);
    $pdf->Cell(50, 8, 'Generated:', 0, 0);
    $pdf->Cell(0, 8, date('d/m/Y H:i', strtotime($data['processed_at'])), 0, 1);
    $pdf->Ln(5);

    // Earnings table
    $pdf->SetFont('helvetica', 'B', 11);
    $pdf->SetFillColor(240, 240, 240);
    $pdf->Cell(0, 8, 'Earnings', 1, 1, 'L', true);

    $pdf->SetFont('helvetica', '', 10);
    $pdf->Cell(100, 7, 'Basic Salary', 1, 0);
    $pdf->Cell(0, 7, 'GHC ' . number_format($data['basic_salary'], 2), 1, 1, 'R');

    $pdf->Cell(100, 7, 'Housing Allowance', 1, 0);
    $pdf->Cell(0, 7, 'GHC ' . number_format($allowanceData['housing'] ?? 0, 2), 1, 1, 'R');

    $pdf->Cell(100, 7, 'Transport Allowance', 1, 0);
    $pdf->Cell(0, 7, 'GHC ' . number_format($allowanceData['transport'] ?? 0, 2), 1, 1, 'R');

    $pdf->Cell(100, 7, 'Utility Allowance', 1, 0);
    $pdf->Cell(0, 7, 'GHC ' . number_format($allowanceData['utility'] ?? 0, 2), 1, 1, 'R');

    $pdf->Cell(100, 7, 'Other Allowances', 1, 0);
    $pdf->Cell(0, 7, 'GHC ' . number_format($allowanceData['other'] ?? 0, 2), 1, 1, 'R');

    $pdf->Cell(100, 7, 'Overtime', 1, 0);
    $pdf->Cell(0, 7, 'GHC ' . number_format($data['overtime_amount'], 2), 1, 1, 'R');

    $pdf->SetFont('helvetica', 'B', 10);
    $pdf->Cell(100, 7, 'Gross Salary', 1, 0);
    $pdf->Cell(0, 7, 'GHC ' . number_format($data['gross_salary'] + $data['overtime_amount'], 2), 1, 1, 'R');
    $pdf->Ln(5);

    // Deductions table
    $pdf->SetFont('helvetica', 'B', 11);
    $pdf->Cell(0, 8, 'Deductions', 1, 1, 'L', true);

    $pdf->SetFont('helvetica', '', 10);
    $pdf->Cell(100, 7, 'Income Tax (PAYE)', 1, 0);
    $pdf->Cell(0, 7, 'GHC ' . number_format($data['tax_amount'], 2), 1, 1, 'R');

    $pdf->Cell(100, 7, 'SSNIT (5.5%)', 1, 0);
    $pdf->Cell(0, 7, 'GHC ' . number_format($data['ssnit_employee'], 2), 1, 1, 'R');

    $pdf->Cell(100, 7, 'Leave Deductions', 1, 0);
    $pdf->Cell(0, 7, 'GHC ' . number_format($data['leave_deductions'], 2), 1, 1, 'R');

    $pdf->SetFont('helvetica', 'B', 10);
    $pdf->Cell(100, 7, 'Total Deductions', 1, 0);
    $pdf->Cell(0, 7, 'GHC ' . number_format($data['total_deductions'], 2), 1, 1, 'R');
    $pdf->Ln(5);

    // Net Salary
    $pdf->SetFont('helvetica', 'B', 12);
    $pdf->SetFillColor(220, 220, 220);
    $pdf->Cell(100, 8, 'Net Salary', 1, 0, 'L', true);
    $pdf->Cell(0, 8, 'GHC ' . number_format($data['net_salary'], 2), 1, 1, 'R', true);
    $pdf->Ln(10);

    // Footer
    $pdf->SetFont('helvetica', 'I', 8);
    $pdf->Cell(0, 5, 'This is a computer-generated payslip and does not require a signature.', 0, 1, 'C');
    $pdf->Cell(0, 5, 'For any queries, please contact the Accounts Department.', 0, 1, 'C');

    // Output PDF
    $filename = 'payslip_' . preg_replace('/[^a-zA-Z0-9]/', '_', $data['teacher_name']) . '_' . date('M_Y', strtotime($data['payroll_month'])) . '.pdf';
    $pdf->Output($filename, 'D');
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>School Management System - View Payslips</title>
    <link rel="stylesheet" href="../css/style.css">
    <link rel="stylesheet" href="../css/admin_custom.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css">
</head>
<body>
    <?php include 'includes/accounting_sidebar.php'; ?>
    <div class="admin-main">
        <header class="admin-header">
            <div class="header-content">
                <button class="mobile-menu-toggle" id="mobile-menu-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <div class="header-info">
                    <h1><i class="fas fa-calculator"></i> School Management System</h1>
                <div class="header-meta">
                    <span class="admin-badge"><i class="fas fa-user-shield"></i> Accounts Panel</span>
                    <span class="current-date"><i class="fas fa-calendar"></i> <?php echo date('M d, Y'); ?></span>
                </div>
                </div>
            </div>
        </header>
        <main>
<style>
.payroll-header {
    background: white;
    border-radius: 12px;
    padding: 25px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    margin-bottom: 30px;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.payroll-info {
    display: flex;
    gap: 30px;
}

.payroll-stat {
    text-align: center;
}

.payroll-stat .value {
    font-size: 2em;
    font-weight: bold;
    color: #3498db;
}

.payroll-stat .label {
    color: #7f8c8d;
    font-size: 0.9em;
}

.payslips-table {
    background: white;
    border-radius: 12px;
    overflow: hidden;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.payslips-table table {
    width: 100%;
    border-collapse: collapse;
}

.payslips-table th,
.payslips-table td {
    padding: 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.payslips-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #2c3e50;
}

.payslips-table tr:hover {
    background: #f8f9fa;
}

.amount {
    font-weight: 600;
    color: #27ae60;
}

.deduction {
    color: #e74c3c;
}

.net-amount {
    font-weight: bold;
    font-size: 1.1em;
    color: #2c3e50;
}

.payroll-selector {
    background: #f8f9fa;
    padding: 15px;
    border-radius: 8px;
    margin-bottom: 20px;
}

.payroll-selector select {
    padding: 8px 12px;
    border: 1px solid #ddd;
    border-radius: 4px;
    margin-left: 10px;
}

.action-buttons {
    display: flex;
    gap: 10px;
}

.btn {
    display: inline-block;
    padding: 8px 16px;
    margin: 0;
    font-size: 14px;
    font-weight: 400;
    line-height: 1.42857143;
    text-align: center;
    white-space: nowrap;
    vertical-align: middle;
    cursor: pointer;
    border: 1px solid transparent;
    border-radius: 4px;
    text-decoration: none;
    background-color: #007bff;
    color: #fff;
    border-color: #007bff;
}

.btn:hover {
    background-color: #0056b3;
    border-color: #004085;
}

.btn-small {
    padding: 6px 12px;
    font-size: 0.9em;
}
</style>

<h2>📄 View Payslips</h2>

<?php if ($message): ?>
    <div class="message <?php echo $message_type; ?>"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>

<!-- Payroll Selector -->
<div class="payroll-selector">
    <form method="GET" style="display: inline;">
        <label for="payroll_id">Select Payroll Period:</label>
        <select name="payroll_id" id="payroll_id" onchange="this.form.submit()">
            <option value="">-- Select Payroll --</option>
            <?php
            $stmt = $pdo->query("SELECT id, payroll_month, status FROM payroll_runs ORDER BY payroll_month DESC");
            while ($row = $stmt->fetch()) {
                $selected = ($row['id'] == $payroll_id) ? 'selected' : '';
                echo "<option value=\"{$row['id']}\" $selected>" . date('F Y', strtotime($row['payroll_month'])) . " ({$row['status']})</option>";
            }
            ?>
        </select>
    </form>
</div>

<?php if ($payroll): ?>
    <!-- Payroll Header -->
    <div class="payroll-header">
        <div>
            <h3>Payroll for <?php echo date('F Y', strtotime($payroll['payroll_month'])); ?></h3>
            <p>Status: <span style="color: <?php echo $payroll['status'] == 'processed' ? '#27ae60' : '#f39c12'; ?>; font-weight: bold;"><?php echo ucfirst($payroll['status']); ?></span></p>
            <p>Processed: <?php echo $payroll['processed_at'] ? date('d/m/Y H:i', strtotime($payroll['processed_at'])) : 'Not processed'; ?></p>
        </div>


    </div>

    <!-- Payslips Table -->
    <div class="payslips-table">
        <table>
            <thead>
                <tr>
                    <th>Employee</th>
                    <th>Staff ID</th>
                    <th>Basic Salary</th>
                    <th>Allowances</th>
                    <th>Gross Salary</th>
                    <th>Deductions</th>
                    <th>Net Salary</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($payslips as $payslip): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($payslip['staff_name']); ?></td>
                        <td><?php echo htmlspecialchars($payslip['staff_id_display']); ?></td>
                        <td class="amount">GHC <?php echo number_format($payslip['basic_salary'], 2); ?></td>
                        <td class="amount">GHC <?php echo number_format($payslip['total_allowances'], 2); ?></td>
                        <td class="amount">GHC <?php echo number_format($payslip['gross_salary'], 2); ?></td>
                        <td class="deduction">GHC <?php echo number_format($payslip['total_deductions'], 2); ?></td>
                        <td class="net-amount">GHC <?php echo number_format($payslip['net_salary'], 2); ?></td>
                        <td>
                            <div class="action-buttons">
                                <a href="?payroll_id=<?php echo $payroll_id; ?>&generate_pdf=1&payslip_id=<?php echo $payslip['id']; ?>"
                                   class="btn btn-small" target="_blank">📄 PDF</a>
                                <button onclick="viewDetails(<?php echo $payslip['id']; ?>)" class="btn btn-small">👁️ Details</button>
                            </div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <!-- Payslip Details Modal -->
    <div id="payslipModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: white; padding: 30px; border-radius: 12px; max-width: 600px; width: 90%; max-height: 80%; overflow-y: auto;">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                <h3>Payslip Details</h3>
                <button onclick="closeModal()" style="background: none; border: none; font-size: 1.5em; cursor: pointer;">&times;</button>
            </div>
            <div id="payslipContent"></div>
        </div>
    </div>

<?php else: ?>
    <div style="text-align: center; padding: 60px; background: white; border-radius: 12px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
        <h3>No Payroll Data Found</h3>
        <p>No payroll has been processed yet. Please process payroll first.</p>
        <a href="process_payroll.php" class="btn btn-primary">Process Payroll</a>
    </div>
<?php endif; ?>

<script>
function viewDetails(payslipId) {
    // Fetch payslip details via AJAX
    fetch(`/school/api/get_payslip_details.php?id=${payslipId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                document.getElementById('payslipContent').innerHTML = generatePayslipHTML(data.payslip);
                document.getElementById('payslipModal').style.display = 'block';
            } else {
                alert('Error loading payslip details');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading payslip details');
        });
}

function generatePayslipHTML(payslip) {
    let data = {};
    try {
        data = payslip.payslip_data ? JSON.parse(payslip.payslip_data) : {};
    } catch (e) {
        console.error('Error parsing payslip_data JSON:', e);
        data = {};
    }

    return `
        <div style="font-family: Arial, sans-serif;">
            <div style="text-align: center; margin-bottom: 20px;">
                <h4>${payslip.teacher_name}</h4>
                <p>Pay Period: ${new Date(payslip.payroll_month).toLocaleDateString('en-US', { year: 'numeric', month: 'long' })}</p>
            </div>

            <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
                <tr style="background: #f8f9fa;"><th colspan="2" style="padding: 10px; text-align: left;">Earnings</th></tr>
                <tr><td style="padding: 8px; border: 1px solid #ddd;">Basic Salary</td><td style="padding: 8px; border: 1px solid #ddd; text-align: right;">GHC ${parseFloat(payslip.basic_salary).toFixed(2)}</td></tr>
                <tr><td style="padding: 8px; border: 1px solid #ddd;">Housing Allowance</td><td style="padding: 8px; border: 1px solid #ddd; text-align: right;">GHC ${(data.allowances?.housing || 0).toFixed(2)}</td></tr>
                <tr><td style="padding: 8px; border: 1px solid #ddd;">Transport Allowance</td><td style="padding: 8px; border: 1px solid #ddd; text-align: right;">GHC ${(data.allowances?.transport || 0).toFixed(2)}</td></tr>
                <tr><td style="padding: 8px; border: 1px solid #ddd;">Utility Allowance</td><td style="padding: 8px; border: 1px solid #ddd; text-align: right;">GHC ${(data.allowances?.utility || 0).toFixed(2)}</td></tr>
                <tr><td style="padding: 8px; border: 1px solid #ddd;">Other Allowances</td><td style="padding: 8px; border: 1px solid #ddd; text-align: right;">GHC ${(data.allowances?.other || 0).toFixed(2)}</td></tr>
                <tr><td style="padding: 8px; border: 1px solid #ddd;">Overtime</td><td style="padding: 8px; border: 1px solid #ddd; text-align: right;">GHC ${parseFloat(payslip.overtime_amount).toFixed(2)}</td></tr>
                <tr style="background: #e8f5e8; font-weight: bold;"><td style="padding: 8px; border: 1px solid #ddd;">Gross Salary</td><td style="padding: 8px; border: 1px solid #ddd; text-align: right;">GHC ${(parseFloat(payslip.gross_salary) + parseFloat(payslip.overtime_amount)).toFixed(2)}</td></tr>
            </table>

            <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
                <tr style="background: #f8f9fa;"><th colspan="2" style="padding: 10px; text-align: left;">Deductions</th></tr>
                <tr><td style="padding: 8px; border: 1px solid #ddd;">Income Tax (PAYE)</td><td style="padding: 8px; border: 1px solid #ddd; text-align: right;">GHC ${parseFloat(payslip.tax_amount).toFixed(2)}</td></tr>
                <tr><td style="padding: 8px; border: 1px solid #ddd;">SSNIT (5.5%)</td><td style="padding: 8px; border: 1px solid #ddd; text-align: right;">GHC ${parseFloat(payslip.ssnit_employee).toFixed(2)}</td></tr>
                <tr><td style="padding: 8px; border: 1px solid #ddd;">Leave Deductions</td><td style="padding: 8px; border: 1px solid #ddd; text-align: right;">GHC ${parseFloat(payslip.leave_deductions).toFixed(2)}</td></tr>
                <tr style="background: #ffeaea; font-weight: bold;"><td style="padding: 8px; border: 1px solid #ddd;">Total Deductions</td><td style="padding: 8px; border: 1px solid #ddd; text-align: right;">GHC ${parseFloat(payslip.total_deductions).toFixed(2)}</td></tr>
            </table>

            <table style="width: 100%; border-collapse: collapse;">
                <tr style="background: #e8f0ff; font-weight: bold; font-size: 1.1em;">
                    <td style="padding: 12px; border: 1px solid #ddd;">Net Salary</td>
                    <td style="padding: 12px; border: 1px solid #ddd; text-align: right;">GHC ${parseFloat(payslip.net_salary).toFixed(2)}</td>
                </tr>
            </table>

            <div style="margin-top: 20px; padding: 15px; background: #f8f9fa; border-radius: 8px;">
                <h5>Attendance Summary</h5>
                <p>Working Days: ${payslip.working_days} | Absent Days: ${payslip.absent_days} | Leave Days: ${payslip.leave_days} | Overtime Hours: ${payslip.overtime_hours}</p>
            </div>
        </div>
    `;
}

function closeModal() {
    document.getElementById('payslipModal').style.display = 'none';
}

// Close modal when clicking outside
document.getElementById('payslipModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeModal();
    }
});
</script>

<?php
include '../includes/admin_footer.php';
?>
