<?php
require_once '../config.php';

$message = '';
$error = '';

// Fetch cash accounts for dropdowns
try {
    $stmt = $pdo->query("SELECT id, account_name, current_balance FROM cash_accounts ORDER BY account_name ASC");
    $cash_accounts = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading cash accounts: " . $e->getMessage();
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $source_account_id = intval($_POST['source_account']);
    $destination_account_id = intval($_POST['destination_account']);
    $amount = floatval($_POST['amount']);
    $description = trim($_POST['description']);

    // Validation
    if ($source_account_id === $destination_account_id) {
        $error = "Source and destination accounts must be different.";
    } elseif ($amount <= 0) {
        $error = "Transfer amount must be greater than zero.";
    } else {
        try {
            // Start transaction
            $pdo->beginTransaction();

            // Get source account balance
            $stmt = $pdo->prepare("SELECT current_balance FROM cash_accounts WHERE id = ?");
            $stmt->execute([$source_account_id]);
            $source_account = $stmt->fetch();

            if (!$source_account) {
                throw new Exception("Source account not found.");
            }

            if ($source_account['current_balance'] < $amount) {
                throw new Exception("Insufficient funds in source account. Current balance: GHC " . number_format($source_account['current_balance'], 2));
            }

            // Get destination account to verify existence
            $stmt = $pdo->prepare("SELECT current_balance FROM cash_accounts WHERE id = ?");
            $stmt->execute([$destination_account_id]);
            $destination_account = $stmt->fetch();

            if (!$destination_account) {
                throw new Exception("Destination account not found.");
            }

            // Debit source account
            $new_source_balance = $source_account['current_balance'] - $amount;
            $stmt = $pdo->prepare("UPDATE cash_accounts SET current_balance = ? WHERE id = ?");
            $stmt->execute([$new_source_balance, $source_account_id]);

            // Credit destination account
            $new_destination_balance = $destination_account['current_balance'] + $amount;
            $stmt = $pdo->prepare("UPDATE cash_accounts SET current_balance = ? WHERE id = ?");
            $stmt->execute([$new_destination_balance, $destination_account_id]);

            // Log transactions
            $user_id = $_SESSION['user_id'] ?? null;

            // Log debit transaction
            $stmt = $pdo->prepare("INSERT INTO cash_account_transactions (account_id, transaction_type, amount, description, created_by) VALUES (?, 'decrease', ?, ?, ?)");
            $stmt->execute([$source_account_id, $amount, $description ?: 'Transfer to account ID ' . $destination_account_id, $user_id]);

            // Log credit transaction
            $stmt = $pdo->prepare("INSERT INTO cash_account_transactions (account_id, transaction_type, amount, description, created_by) VALUES (?, 'increase', ?, ?, ?)");
            $stmt->execute([$destination_account_id, $amount, $description ?: 'Transfer from account ID ' . $source_account_id, $user_id]);

            $pdo->commit();

            $message = "Funds transferred successfully!";
            header("Location: " . $_SERVER['PHP_SELF'] . "?success=" . urlencode($message));
            exit();

        } catch (Exception $e) {
            $pdo->rollBack();
            $error = "Error during transfer: " . $e->getMessage();
        }
    }
}

include 'includes/accounting_header.php';
?>

<div class="transfer-funds-container">
    <div class="page-header">
        <h1>💸 Transfer Funds Between Cash Accounts</h1>
        <p>Transfer funds securely between your cash accounts</p>
    </div>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <form method="post" class="transfer-form">
        <div class="form-group">
            <label for="source_account">Source Account *</label>
            <select id="source_account" name="source_account" required>
                <option value="">Select Source Account</option>
                <?php foreach ($cash_accounts as $account): ?>
                    <option value="<?php echo $account['id']; ?>"><?php echo htmlspecialchars($account['account_name']) . ' (Balance: GHC ' . number_format($account['current_balance'], 2) . ')'; ?></option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group">
            <label for="destination_account">Destination Account *</label>
            <select id="destination_account" name="destination_account" required>
                <option value="">Select Destination Account</option>
                <?php foreach ($cash_accounts as $account): ?>
                    <option value="<?php echo $account['id']; ?>"><?php echo htmlspecialchars($account['account_name']) . ' (Balance: GHC ' . number_format($account['current_balance'], 2) . ')'; ?></option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group">
            <label for="amount">Amount (GHC) *</label>
            <input type="number" id="amount" name="amount" step="0.01" min="0.01" required>
        </div>

        <div class="form-group">
            <label for="description">Description (optional)</label>
            <textarea id="description" name="description" rows="3" placeholder="Enter description or reason for transfer"></textarea>
        </div>

        <button type="submit" class="btn btn-primary">Transfer Funds</button>
    </form>
</div>

<style>
.transfer-funds-container {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.page-header {
    background: white;
    padding: 30px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.page-header h1 {
    color: #2c3e50;
    margin: 0 0 10px 0;
}

.page-header p {
    color: #7f8c8d;
    margin: 0;
}

.transfer-form {
    max-width: 500px;
    margin: 0 auto;
    background: white;
    padding: 30px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.form-group {
    margin-bottom: 20px;
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 8px;
    color: #495057;
}

.form-group select,
.form-group input,
.form-group textarea {
    padding: 12px 15px;
    border: 1px solid #ced4da;
    border-radius: 6px;
    font-size: 16px;
    transition: border-color 0.3s ease;
}

.form-group select:focus,
.form-group input:focus,
.form-group textarea:focus {
    border-color: #4CAF50;
    outline: none;
}

.btn-primary {
    background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);
    color: white;
    font-weight: 700;
    padding: 14px 20px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    transition: background-color 0.3s ease;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #45a049 0%, #3d8b40 100%);
}
.error-message {
    max-width: 500px;
    margin: 0 auto 20px auto;
    padding: 15px 20px;
    background: #f8d7da;
    color: #721c24;
    border-radius: 8px;
    border: 1px solid #f5c6cb;
    font-weight: 600;
}
</style>

<?php include '../includes/admin_footer.php'; ?>
