<?php
require_once '../config.php';
require_once '../includes/access_control.php';

// Check if user is accounts
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'accounts') {
    header('Location: ../index.php');
    exit;
}

$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['record_sale'])) {
        $student_id = intval($_POST['student_id']);
        $items = $_POST['items'] ?? [];
        $notes = trim($_POST['notes']);

        if (empty($items)) {
            $message = "Please add at least one item to the sale.";
            $message_type = "error";
        } else {
            try {
                // Generate unique sale_id
                $sale_id = 'SALE_' . date('Ymd_His') . '_' . rand(1000, 9999);

                $pdo->beginTransaction();

                foreach ($items as $item) {
                    $item_id = intval($item['item_id']);
                    $quantity = intval($item['quantity']);
                    $unit_price = floatval($item['unit_price']);

                    // Insert sale item
                    $stmt = $pdo->prepare("INSERT INTO student_sales (sale_id, student_id, item_id, quantity, unit_price, recorded_by, notes) VALUES (?, ?, ?, ?, ?, ?, ?)");
                    $stmt->execute([$sale_id, $student_id, $item_id, $quantity, $unit_price, $_SESSION['user_id'], $notes]);

                    // Update inventory stock
                    $pdo->prepare("UPDATE inventory_items SET quantity = quantity - ? WHERE id = ?")
                         ->execute([$quantity, $item_id]);
                }

                $pdo->commit();
                $message = "Sale recorded successfully!";
                $message_type = "success";
            } catch (PDOException $e) {
                $pdo->rollBack();
                $message = "Error recording sale: " . $e->getMessage();
                $message_type = "error";
            }
        }
    }

    if (isset($_POST['record_payment'])) {
        $sale_id = trim($_POST['sale_id']);
        $payment_amount = floatval($_POST['payment_amount']);

        try {
            $pdo->beginTransaction();

            // Get all items for this sale
            $sale_stmt = $pdo->prepare("SELECT id, actual_received, expected_amount FROM student_sales WHERE sale_id = ?");
            $sale_stmt->execute([$sale_id]);
            $sale_items = $sale_stmt->fetchAll(PDO::FETCH_ASSOC);

            if (!empty($sale_items)) {
                $total_expected = array_sum(array_column($sale_items, 'expected_amount'));
                $total_received = array_sum(array_column($sale_items, 'actual_received'));
                $new_total_received = $total_received + $payment_amount;

                // Distribute payment proportionally across items
                $remaining_payment = $payment_amount;
                foreach ($sale_items as $item) {
                    if ($remaining_payment <= 0) break;

                    $item_outstanding = $item['expected_amount'] - $item['actual_received'];
                    $payment_for_item = min($remaining_payment, $item_outstanding);

                    $new_received = $item['actual_received'] + $payment_for_item;
                    $status = ($new_received >= $item['expected_amount']) ? 'paid' : 'partial';

                    $update_stmt = $pdo->prepare("UPDATE student_sales SET actual_received = ?, status = ? WHERE id = ?");
                    $update_stmt->execute([$new_received, $status, $item['id']]);

                    $remaining_payment -= $payment_for_item;
                }

                $pdo->commit();

                // Get sale details for receipt
                $receipt_stmt = $pdo->prepare("
                    SELECT
                        ss.sale_id,
                        s.name as student_name,
                        GROUP_CONCAT(CONCAT(ii.name, ' (', ss.quantity, ' x ', ss.unit_price, ')') SEPARATOR ', ') as items,
                        SUM(ss.expected_amount) as total_expected,
                        SUM(ss.actual_received) as total_received,
                        SUM(ss.outstanding) as total_outstanding,
                        ss.created_at as sale_date
                    FROM student_sales ss
                    JOIN students s ON ss.student_id = s.id
                    JOIN inventory_items ii ON ss.item_id = ii.id
                    WHERE ss.sale_id = ?
                    GROUP BY ss.sale_id
                ");
                $receipt_stmt->execute([$sale_id]);
                $receipt_data = $receipt_stmt->fetch(PDO::FETCH_ASSOC);

                // Get accountant name
                $accountant_stmt = $pdo->prepare("SELECT name FROM users WHERE id = ?");
                $accountant_stmt->execute([$_SESSION['user_id']]);
                $accountant = $accountant_stmt->fetch(PDO::FETCH_ASSOC);

                // Store receipt data in session for printing
                $_SESSION['receipt_data'] = [
                    'sale_id' => $receipt_data['sale_id'],
                    'student_name' => $receipt_data['student_name'],
                    'items' => $receipt_data['items'],
                    'total_expected' => $receipt_data['total_expected'],
                    'payment_amount' => $payment_amount,
                    'total_received' => $receipt_data['total_received'],
                    'total_outstanding' => $receipt_data['total_outstanding'],
                    'sale_date' => $receipt_data['sale_date'],
                    'accountant_name' => $accountant['name'] ?? 'Unknown',
                    'payment_date' => date('Y-m-d H:i:s')
                ];

                $message = "Payment recorded successfully! Receipt will be printed.";
                $message_type = "success";
                $print_receipt = true;
            }
        } catch (PDOException $e) {
            $pdo->rollBack();
            $message = "Error recording payment: " . $e->getMessage();
            $message_type = "error";
        }
    }
}

// Get all students
$students_stmt = $pdo->query("SELECT id, CONCAT(first_name, ' ', COALESCE(middle_name, ''), ' ', last_name) as name FROM students ORDER BY first_name, last_name");
$students = $students_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get all inventory items
$items_stmt = $pdo->query("SELECT ii.id, ii.name, ii.selling_price, ii.quantity as stock FROM inventory_items ii WHERE ii.status = 'active' ORDER BY ii.name");
$items = $items_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get sales records grouped by sale_id
$sales_stmt = $pdo->prepare("
    SELECT
        ss.sale_id,
        GROUP_CONCAT(DISTINCT s.name) as student_name,
        GROUP_CONCAT(CONCAT(ii.name, ' (', ss.quantity, ')') SEPARATOR ', ') as items,
        SUM(ss.expected_amount) as total_expected,
        SUM(ss.actual_received) as total_received,
        SUM(ss.outstanding) as total_outstanding,
        CASE
            WHEN SUM(ss.outstanding) = 0 THEN 'paid'
            WHEN SUM(ss.actual_received) > 0 THEN 'partial'
            ELSE 'pending'
        END as overall_status,
        MAX(ss.created_at) as sale_date,
        GROUP_CONCAT(ss.id) as item_ids
    FROM student_sales ss
    JOIN students s ON ss.student_id = s.id
    JOIN inventory_items ii ON ss.item_id = ii.id
    GROUP BY ss.sale_id
    ORDER BY MAX(ss.created_at) DESC
    LIMIT 50
");
$sales_stmt->execute();
$sales = $sales_stmt->fetchAll(PDO::FETCH_ASSOC);

$page_title = "Student Sales Management";
include 'includes/accounting_header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2><i class="fas fa-shopping-cart"></i> Student Sales Management</h2>
                <button class="btn btn-facilities" data-toggle="modal" data-target="#recordSaleModal">
                    <i class="fas fa-plus"></i> Record New Sale
                </button>
            </div>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="close" data-dismiss="alert">
                        <span>&times;</span>
                    </button>
                </div>
            <?php endif; ?>

            <?php if (isset($print_receipt) && $print_receipt && isset($_SESSION['receipt_data'])): ?>
                <script>
                    function printReceipt() {
                        var printWindow = window.open('print_receipt.php', '_blank', 'width=800,height=600');
                        printWindow.focus();
                    }
                    window.onload = function() {
                        setTimeout(printReceipt, 1000); // Delay to ensure page loads
                    };
                </script>
            <?php endif; ?>

            <!-- Sales Records -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5><i class="fas fa-list"></i> Recent Sales Records</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover facilities-table">
                            <thead class="thead-dark">
                                <tr>
                                    <th>Sale ID</th>
                                    <th>Student</th>
                                    <th>Items</th>
                                    <th>Total Expected</th>
                                    <th>Received</th>
                                    <th>Outstanding</th>
                                    <th>Status</th>
                                    <th>Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($sales as $sale): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($sale['sale_id']); ?></td>
                                        <td><?php echo htmlspecialchars($sale['student_name']); ?></td>
                                        <td><?php echo htmlspecialchars($sale['items']); ?></td>
                                        <td><?php echo number_format($sale['total_expected'], 2); ?></td>
                                        <td><?php echo number_format($sale['total_received'], 2); ?></td>
                                        <td><?php echo number_format($sale['total_outstanding'], 2); ?></td>
                                        <td>
                                            <span class="badge badge-<?php
                                                echo $sale['overall_status'] === 'paid' ? 'success' :
                                                     ($sale['overall_status'] === 'partial' ? 'warning' : 'danger');
                                            ?>">
                                                <?php echo ucfirst($sale['overall_status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo date('M d, Y', strtotime($sale['sale_date'])); ?></td>
                                        <td>
                                            <?php if ($sale['total_outstanding'] > 0): ?>
                                                <button class="btn btn-sm btn-primary" onclick="recordPayment('<?php echo $sale['sale_id']; ?>', '<?php echo htmlspecialchars($sale['student_name']); ?>', <?php echo $sale['total_outstanding']; ?>)">
                                                    <i class="fas fa-money-bill"></i> Pay
                                                </button>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                <?php if (empty($sales)): ?>
                                    <tr>
                                        <td colspan="9" class="text-center">No sales records found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Record Sale Modal -->
<div class="modal fade" id="recordSaleModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-xl" role="document">
        <div class="modal-content">
            <form method="POST" id="saleForm">
                <div class="modal-header">
                    <h5 class="modal-title">Record New Sale</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="form-group">
                        <label for="student_id">Student *</label>
                        <select class="form-control" id="student_id" name="student_id" required>
                            <option value="">Select Student</option>
                            <?php foreach ($students as $student): ?>
                                <option value="<?php echo $student['id']; ?>"><?php echo htmlspecialchars($student['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label>Items *</label>
                        <div id="itemsContainer">
                            <div class="item-row border p-3 mb-3">
                                <div class="row">
                                    <div class="col-md-4">
                                        <label>Item</label>
                                        <select class="form-control item-select" name="items[0][item_id]" required onchange="updatePrice(this)">
                                            <option value="">Select Item</option>
                                            <?php foreach ($items as $item): ?>
                                                <option value="<?php echo $item['id']; ?>" data-price="<?php echo $item['selling_price']; ?>" data-stock="<?php echo $item['stock']; ?>">
                                                    <?php echo htmlspecialchars($item['name']); ?> (Stock: <?php echo $item['stock']; ?>)
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="col-md-2">
                                        <label>Quantity</label>
                                        <input type="number" class="form-control quantity-input" name="items[0][quantity]" min="1" required onchange="calculateRowTotal(this)">
                                    </div>
                                    <div class="col-md-2">
                                        <label>Unit Price</label>
                                        <input type="number" step="0.01" class="form-control unit-price-input" name="items[0][unit_price]" min="0" required onchange="calculateRowTotal(this)">
                                    </div>
                                    <div class="col-md-2">
                                        <label>Total</label>
                                        <input type="text" class="form-control row-total" readonly>
                                    </div>
                                    <div class="col-md-2">
                                        <label>&nbsp;</label>
                                        <button type="button" class="btn btn-danger btn-block" onclick="removeItem(this)">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <button type="button" class="btn btn-secondary" onclick="addItem()">
                            <i class="fas fa-plus"></i> Add Another Item
                        </button>
                    </div>

                    <div class="form-group">
                        <label for="notes">Notes</label>
                        <textarea class="form-control" id="notes" name="notes" rows="3"></textarea>
                    </div>

                    <div class="form-group">
                        <h5>Grand Total: <span id="grandTotal">0.00</span></h5>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="record_sale" class="btn btn-primary">Record Sale</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Record Payment Modal -->
<div class="modal fade" id="recordPaymentModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title">Record Payment</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <input type="hidden" id="payment_sale_id" name="sale_id">
                    <div class="form-group">
                        <label for="payment_student">Student</label>
                        <input type="text" class="form-control" id="payment_student" readonly>
                    </div>
                    <div class="form-group">
                        <label for="outstanding_amount">Outstanding Amount</label>
                        <input type="text" class="form-control" id="outstanding_amount" readonly>
                    </div>
                    <div class="form-group">
                        <label for="payment_amount">Payment Amount *</label>
                        <input type="number" step="0.01" class="form-control" id="payment_amount" name="payment_amount" min="0.01" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="record_payment" class="btn btn-primary">Record Payment</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
let itemIndex = 1;

function addItem() {
    const container = document.getElementById('itemsContainer');
    const itemRow = document.createElement('div');
    itemRow.className = 'item-row border p-3 mb-3';
    itemRow.innerHTML = `
        <div class="row">
            <div class="col-md-4">
                <label>Item</label>
                <select class="form-control item-select" name="items[${itemIndex}][item_id]" required onchange="updatePrice(this)">
                    <option value="">Select Item</option>
                    <?php foreach ($items as $item): ?>
                        <option value="<?php echo $item['id']; ?>" data-price="<?php echo $item['selling_price']; ?>" data-stock="<?php echo $item['stock']; ?>">
                            <?php echo htmlspecialchars($item['name']); ?> (Stock: <?php echo $item['stock']; ?>)
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-2">
                <label>Quantity</label>
                <input type="number" class="form-control quantity-input" name="items[${itemIndex}][quantity]" min="1" required onchange="calculateRowTotal(this)">
            </div>
            <div class="col-md-2">
                <label>Unit Price</label>
                <input type="number" step="0.01" class="form-control unit-price-input" name="items[${itemIndex}][unit_price]" min="0" required onchange="calculateRowTotal(this)">
            </div>
            <div class="col-md-2">
                <label>Total</label>
                <input type="text" class="form-control row-total" readonly>
            </div>
            <div class="col-md-2">
                <label>&nbsp;</label>
                <button type="button" class="btn btn-danger btn-block" onclick="removeItem(this)">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
        </div>
    `;
    container.appendChild(itemRow);
    itemIndex++;
    updateGrandTotal();
}

function removeItem(button) {
    const itemRow = button.closest('.item-row');
    itemRow.remove();
    updateGrandTotal();
}

function updatePrice(select) {
    const selectedOption = select.options[select.selectedIndex];
    const price = selectedOption.getAttribute('data-price') || 0;
    const row = select.closest('.item-row');
    const priceInput = row.querySelector('.unit-price-input');
    priceInput.value = price;
    calculateRowTotal(priceInput);
}

function calculateRowTotal(input) {
    const row = input.closest('.item-row');
    const quantity = parseFloat(row.querySelector('.quantity-input').value) || 0;
    const unitPrice = parseFloat(row.querySelector('.unit-price-input').value) || 0;
    const total = quantity * unitPrice;
    row.querySelector('.row-total').value = total.toFixed(2);
    updateGrandTotal();
}

function updateGrandTotal() {
    const rowTotals = document.querySelectorAll('.row-total');
    let grandTotal = 0;
    rowTotals.forEach(total => {
        grandTotal += parseFloat(total.value) || 0;
    });
    document.getElementById('grandTotal').textContent = grandTotal.toFixed(2);
}

function recordPayment(saleId, studentName, outstanding) {
    document.getElementById('payment_sale_id').value = saleId;
    document.getElementById('payment_student').value = studentName;
    document.getElementById('outstanding_amount').value = outstanding.toFixed(2);
    document.getElementById('payment_amount').value = '';
    $('#recordPaymentModal').modal('show');
}

$(document).ready(function() {
    // Initialize calculations for existing items
    updateGrandTotal();
});
</script>

<?php include '../includes/admin_footer.php'; ?>
