<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';
require_once 'accounting_functions.php';
require_once 'fee_payment_accounting.php';

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['record_payment'])) {
        $student_fee_id = $_POST['student_fee_id'];
        $payment_amount = floatval($_POST['payment_amount']);
        $payment_method = $_POST['payment_method'];
        $payment_date = $_POST['payment_date'];
        $receipt_number = 'RCP-' . date('Ymd') . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);

        try {
            // Get student fee details
            $stmt = $pdo->prepare("SELECT sf.*, CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as student_name, f.name as fee_name, f.amount as fee_amount FROM student_fees sf JOIN students s ON sf.student_id = s.id JOIN fees f ON sf.fee_id = f.id WHERE sf.id = ?");
            $stmt->execute([$student_fee_id]);
            $student_fee = $stmt->fetch();

            if ($student_fee) {
                $total_fee_amount = floatval($student_fee['fee_amount']);
                $current_paid_amount = floatval($student_fee['paid_amount'] ?? 0);
                $new_paid_amount = $current_paid_amount + $payment_amount;

                // Determine payment status
                if ($new_paid_amount >= $total_fee_amount) {
                    $paid_status = TRUE;
                    $remaining_balance = 0;
                    $status_text = 'Fully Paid';
                } else {
                    $paid_status = FALSE;
                    $remaining_balance = $total_fee_amount - $new_paid_amount;
                    $status_text = 'Partially Paid';
                }

                // Update payment status and amounts
                $stmt = $pdo->prepare("UPDATE student_fees SET paid = ?, paid_at = ?, payment_method = ?, receipt_number = ?, paid_amount = ?, processed_by = ? WHERE id = ?");
                $stmt->execute([$paid_status, $payment_date, $payment_method, $receipt_number, $new_paid_amount, $_SESSION['user_id'] ?? null, $student_fee_id]);

                // Log the payment activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'fee_payment_recorded',
                    "Recorded payment of GHC {$payment_amount} for {$student_fee['student_name']} - {$student_fee['fee_name']} (Receipt: {$receipt_number})",
                    $payment_amount
                );

                // Get accountant name
                $accountant_stmt = $pdo->prepare("SELECT COALESCE(CONCAT(sa.first_name, ' ', sa.last_name), sa.username, 'Unknown') as name FROM staff_accounts sa WHERE sa.id = ?");
                $accountant_stmt->execute([$_SESSION['user_id']]);
                $accountant = $accountant_stmt->fetch();

                // Process fee payment accounting using the new module
                $student_fee_data = [
                    'student_name' => $student_fee['student_name'],
                    'fee_name' => $student_fee['fee_name'],
                    'fee_id' => $student_fee['fee_id'],
                    'receipt_number' => $receipt_number
                ];

                $accounting_success = processFeePaymentAccounting(
                    $pdo,
                    $student_fee_id,
                    $payment_amount,
                    $payment_method,
                    $payment_date,
                    $_SESSION['user_id'] ?? null,
                    $student_fee_data
                );

                if (!$accounting_success) {
                    error_log("WARNING: Fee payment accounting processing failed, but payment was recorded in student_fees table");
                }

                // Store receipt data in session for printing
                $_SESSION['receipt_data'] = [
                    'receipt_number' => $receipt_number,
                    'student_name' => $student_fee['student_name'],
                    'fee_name' => $student_fee['fee_name'],
                    'amount' => $payment_amount,
                    'total_paid' => $new_paid_amount,
                    'remaining_balance' => $remaining_balance,
                    'payment_date' => $payment_date,
                    'payment_method' => $payment_method,
                    'status' => $status_text,
                    'accountant_name' => $accountant['name'] ?? 'Unknown'
                ];

                // Redirect to prevent form resubmission (PRG pattern)
                header("Location: record_payments.php?success=1");
                exit;
            } else {
                $error = 'Student fee record not found.';
            }
        } catch (PDOException $e) {
            $error = 'Error recording payment: ' . $e->getMessage();
        }
    }
}

// Handle success message from redirect
if (isset($_GET['success']) && isset($_SESSION['receipt_data'])) {
    $receipt = $_SESSION['receipt_data'];
    $message = "Payment recorded successfully! Receipt: {$receipt['receipt_number']} - Status: {$receipt['status']}";

    if ($receipt['remaining_balance'] > 0) {
        $message .= " - Remaining Balance: GHC " . number_format($receipt['remaining_balance'], 2);
    }

    // Don't clear receipt data here - let print_receipt.php handle it
}

$filter_search = $_GET['search'] ?? '';
$filter_class = $_GET['class'] ?? '';
$filter_academic_year = $_GET['academic_year'] ?? '';
$filter_fee_type = $_GET['fee_type'] ?? '';
$filter_status = $_GET['status'] ?? 'unpaid';

$query = "SELECT sf.*, CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as student_name, s.class, s.academic_year, f.name as fee_name, f.amount as fee_amount FROM student_fees sf JOIN students s ON sf.student_id = s.id JOIN fees f ON sf.fee_id = f.id WHERE 1=1";
$params = [];

if ($filter_search) {
    $query .= " AND CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) LIKE ?";
    $params[] = '%' . $filter_search . '%';
}
if ($filter_class) {
    $query .= " AND s.class = ?";
    $params[] = $filter_class;
}
if ($filter_academic_year) {
    $query .= " AND s.academic_year = ?";
    $params[] = $filter_academic_year;
}
if ($filter_fee_type) {
    $query .= " AND f.name = ?";
    $params[] = $filter_fee_type;
}

if ($filter_status === 'paid') {
    $query .= " AND sf.paid = TRUE";
} elseif ($filter_status === 'unpaid') {
    $query .= " AND sf.paid = FALSE";
} elseif ($filter_status === 'partial') {
    $query .= " AND sf.paid = FALSE AND sf.paid_amount > 0";
}

$query .= " ORDER BY sf.due_date ASC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$student_fees = $stmt->fetchAll();

$classes_stmt = $pdo->query("SELECT name FROM classes ORDER BY name");
$classes = $classes_stmt->fetchAll(PDO::FETCH_COLUMN);

$years_stmt = $pdo->query("SELECT year FROM academic_years ORDER BY year DESC");
$academic_years = $years_stmt->fetchAll(PDO::FETCH_COLUMN);

// No longer need primary income account checks or fallback account selection

include 'includes/accounting_header.php';
?>

<div class="page-container">
    <h2>Record Fee Payments</h2>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?>
            <?php if (isset($_SESSION['receipt_data'])): ?>
                <a href="print_receipt.php?PHPSESSID=<?php echo session_id(); ?>" class="btn btn-small" target="_blank">Print Receipt</a>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <form method="get" action="record_payments.php" class="filter-form">
        <div class="form-group">
            <label for="search">Search Student:</label>
            <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($_GET['search'] ?? ''); ?>" placeholder="Enter student name">
        </div>

        <div class="form-group">
            <label for="class">Filter by Class:</label>
            <select id="class" name="class">
                <option value="">All Classes</option>
                <?php foreach ($classes as $class): ?>
                    <option value="<?php echo htmlspecialchars($class); ?>" <?php if ($class == $filter_class) echo 'selected'; ?>>
                        <?php echo htmlspecialchars($class); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group">
            <label for="academic_year">Filter by Academic Year:</label>
            <select id="academic_year" name="academic_year">
                <option value="">All Years</option>
                <?php foreach ($academic_years as $year): ?>
                    <option value="<?php echo htmlspecialchars($year); ?>" <?php if ($year == $filter_academic_year) echo 'selected'; ?>>
                        <?php echo htmlspecialchars($year); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group">
            <label for="fee_type">Filter by Fee Type:</label>
            <select id="fee_type" name="fee_type">
                <option value="">All Fee Types</option>
                <?php
                $fee_types_stmt = $pdo->query("SELECT DISTINCT name FROM fees ORDER BY name");
                $fee_types = $fee_types_stmt->fetchAll(PDO::FETCH_COLUMN);
                $filter_fee_type = $_GET['fee_type'] ?? '';
                foreach ($fee_types as $fee_type):
                ?>
                    <option value="<?php echo htmlspecialchars($fee_type); ?>" <?php if ($fee_type == $filter_fee_type) echo 'selected'; ?>>
                        <?php echo htmlspecialchars($fee_type); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group">
            <label for="status">Payment Status:</label>
            <select id="status" name="status">
                <option value="unpaid" <?php if ($filter_status == 'unpaid') echo 'selected'; ?>>Unpaid</option>
                <option value="partial" <?php if ($filter_status == 'partial') echo 'selected'; ?>>Partially Paid</option>
                <option value="paid" <?php if ($filter_status == 'paid') echo 'selected'; ?>>Paid</option>
                <option value="all" <?php if ($filter_status == 'all') echo 'selected'; ?>>All</option>
            </select>
        </div>

        <div class="form-actions">
            <button type="submit" class="btn btn-secondary">Filter</button>
            <a href="record_payments.php" class="btn btn-secondary">Clear Filters</a>
        </div>
    </form>

    <div class="table-container">
        <h3>Student Fee Records</h3>
        <div class="table-wrapper">
            <table class="payments-table">
                <thead>
                    <tr>
                        <th>Student Name</th>
                        <th>Class</th>
                        <th>Academic Year</th>
                        <th>Fee Name</th>
                        <th>Amount</th>
                        <th>Balance</th>
                        <th>Due Date</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($student_fees as $fee): ?>
                        <?php
                            $remaining_balance = $fee['fee_amount'] - ($fee['paid_amount'] ?? 0);
                        ?>
                        <tr>
                            <td><?php echo htmlspecialchars($fee['student_name']); ?></td>
                            <td><?php echo htmlspecialchars($fee['class']); ?></td>
                            <td><?php echo htmlspecialchars($fee['academic_year']); ?></td>
                            <td><?php echo htmlspecialchars($fee['fee_name']); ?></td>
                            <td>GHC <?php echo number_format($fee['fee_amount'], 2); ?></td>
                            <td>GHC <?php echo number_format($remaining_balance, 2); ?></td>
                            <td><?php echo htmlspecialchars($fee['due_date']); ?></td>
                            <td>
                                <?php
                                    if ($fee['paid']) {
                                        $status_class = 'paid';
                                        $status_text = 'Paid';
                                    } elseif ($fee['paid_amount'] > 0) {
                                        $status_class = 'partial';
                                        $status_text = 'Partially Paid';
                                    } else {
                                        $status_class = 'unpaid';
                                        $status_text = 'Unpaid';
                                    }
                                ?>
                                <span class="status-badge <?php echo $status_class; ?>">
                                    <?php echo $status_text; ?>
                                </span>
                            </td>
                            <td>
                                <?php if (!$fee['paid']): ?>
                                    <button type="button" class="btn btn-small btn-primary" onclick="openPaymentModal(<?php echo $fee['id']; ?>, '<?php echo htmlspecialchars($fee['student_name']); ?>', '<?php echo htmlspecialchars($fee['fee_name']); ?>', <?php echo $remaining_balance; ?>)">
                                        Record Payment
                                    </button>
                                <?php else: ?>
                                    <span class="text-muted">Paid on <?php echo htmlspecialchars($fee['paid_at']); ?></span>
                                    <?php if ($fee['receipt_number']): ?>
                                        <br><small>Receipt: <?php echo htmlspecialchars($fee['receipt_number']); ?></small>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Payment Modal -->
    <div id="paymentModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closePaymentModal()">&times;</span>
            <h3>Record Payment</h3>
            <form method="post" action="record_payments.php" id="paymentForm" class="modal-form">
                <input type="hidden" name="student_fee_id" id="student_fee_id">

                <div class="form-group">
                    <label>Student: <span id="modal_student_name"></span></label>
                </div>

                <div class="form-group">
                    <label>Fee: <span id="modal_fee_name"></span></label>
                </div>

                <div class="form-group">
                    <label for="payment_amount">Payment Amount (GHC):</label>
                    <input type="number" id="payment_amount" name="payment_amount" step="0.01" required>
                </div>

                <div class="form-group">
                    <label for="payment_method">Payment Method:</label>
                    <select id="payment_method" name="payment_method" required>
                        <option value="">Select Method</option>
                        <option value="Cash">Cash</option>
                        <option value="Bank Transfer">Bank Transfer</option>
                        <option value="Mobile Money">Mobile Money</option>
                        <option value="Cheque">Cheque</option>
                    </select>
                </div>

                <div class="form-group">
                    <label for="payment_date">Payment Date:</label>
                    <input type="date" id="payment_date" name="payment_date" value="<?php echo date('Y-m-d'); ?>" required>
                    <button type="submit" name="record_payment" class="btn btn-primary">Record Payment</button>
                    <button type="button" class="btn btn-secondary" onclick="closePaymentModal()">Cancel</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function openPaymentModal(feeId, studentName, feeName, amount) {
    document.getElementById('student_fee_id').value = feeId;
    document.getElementById('modal_student_name').textContent = studentName;
    document.getElementById('modal_fee_name').textContent = feeName;
    document.getElementById('payment_amount').value = amount;
    document.getElementById('paymentModal').style.display = 'block';
}

function closePaymentModal() {
    document.getElementById('paymentModal').style.display = 'none';
    document.getElementById('paymentForm').reset();
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('paymentModal');
    if (event.target == modal) {
        closePaymentModal();
    }
}

// Auto-open receipt in new tab after payment processing
<?php if (isset($_GET['success']) && isset($_SESSION['receipt_data'])): ?>
window.onload = function() {
    // Add a small delay to avoid popup blockers
    setTimeout(function() {
        // Open receipt in new tab
        const receiptWindow = window.open('print_receipt.php?PHPSESSID=<?php echo session_id(); ?>', '_blank');
        if (!receiptWindow) {
            alert('Please allow popups for this site to automatically open the receipt. You can also click the "Print Receipt" button above.');
        }
    }, 500);
};
<?php endif; ?>
</script>

<style>
.page-container {
    background: white;
    border-radius: 10px;
    padding: 25px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    margin-bottom: 20px;
}

.page-container h2 {
    color: #2c3e50;
    margin-bottom: 20px;
    font-size: 1.8em;
    font-weight: 600;
}

.page-container h3 {
    color: #2c3e50;
    margin: 20px 0 15px 0;
    font-size: 1.4em;
    font-weight: 500;
}

.message {
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border-color: #c3e6cb;
}

.error-message {
    background: #f8d7da;
    color: #721c24;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #f5c6cb;
}

.filter-form {
    display: flex;
    gap: 20px;
    align-items: end;
    margin-bottom: 30px;
    padding: 20px;
    background: #f8f9fa;
    border-radius: 10px;
    border: 1px solid #dee2e6;
    flex-wrap: wrap;
}

.filter-form .form-group {
    flex: 1;
    min-width: 200px;
}

.filter-form label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #2c3e50;
}

.filter-form input[type="text"],
.filter-form select {
    width: 100%;
    padding: 10px 15px;
    border: 2px solid #2E7D32;
    border-radius: 5px;
    background: #fff;
    color: #2c3e50;
    font-size: 0.9em;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.3s ease;
}

.filter-form input[type="text"]:focus,
.filter-form select:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.filter-form input[type="text"]:hover,
.filter-form select:hover {
    border-color: #4CAF50;
}

.form-actions {
    display: flex;
    gap: 10px;
    align-self: end;
}

.table-container {
    background: white;
    padding: 20px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    margin-bottom: 20px;
}

.table-wrapper {
    overflow-x: auto;
}

.payments-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 10px;
}

.payments-table th {
    background: #fff;
    color: #000;
    padding: 12px;
    text-align: left;
    border-bottom: 2px solid #dee2e6;
    font-weight: 600;
}

.payments-table td {
    padding: 12px;
    border-bottom: 1px solid #dee2e6;
}

.payments-table tr:hover {
    background: #f8f9fa;
}

.status-badge {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 0.8em;
    font-weight: bold;
}

.status-badge.paid { background: #d4edda; color: #155724; }
.status-badge.partial { background: #fff3cd; color: #856404; }
.status-badge.unpaid { background: #f8d7da; color: #721c24; }

.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
    backdrop-filter: blur(5px);
}

.modal-content {
    background: white;
    margin: 10% auto;
    padding: 25px;
    border-radius: 10px;
    width: 90%;
    max-width: 500px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.3);
    position: relative;
    max-height: 80vh;
    overflow-y: auto;
}

.modal-content h3 {
    color: #2c3e50;
    margin-bottom: 20px;
    text-align: center;
}

.close {
    position: absolute;
    right: 15px;
    top: 15px;
    color: #aaa;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
    transition: color 0.3s;
}

.close:hover {
    color: #2E7D32;
}

.modal-form .form-group {
    margin-bottom: 20px;
}

.modal-form label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #2c3e50;
}

.modal-form input[type="number"],
.modal-form input[type="date"],
.modal-form select {
    width: 100%;
    padding: 10px 15px;
    border: 2px solid #2E7D32;
    border-radius: 5px;
    background: #fff;
    color: #2c3e50;
    font-size: 0.9em;
    transition: all 0.3s ease;
}

.modal-form input[type="number"]:focus,
.modal-form input[type="date"]:focus,
.modal-form select:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.modal-form .form-actions {
    display: flex;
    gap: 10px;
    justify-content: flex-end;
    margin-top: 20px;
}

.text-muted {
    color: #7f8c8d;
    font-size: 0.9em;
}

.btn {
    padding: 10px 20px;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    font-size: 0.9em;
    font-weight: 500;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    background: linear-gradient(135deg, #6c757d 0%, #5a6268 100%);
    color: #fff;
}

.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: #fff;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.btn-secondary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: #fff;
}

.btn-secondary:hover {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.btn-small {
    padding: 6px 12px;
    font-size: 0.8em;
}

/* Responsive */
@media (max-width: 768px) {
    .filter-form {
        flex-direction: column;
        gap: 15px;
    }

    .filter-form .form-group {
        min-width: auto;
    }

    .form-actions {
        justify-content: center;
    }

    .payments-table {
        font-size: 0.9em;
    }

    .payments-table th,
    .payments-table td {
        padding: 8px 6px;
    }

    .modal-content {
        margin: 5% auto;
        width: 95%;
        padding: 20px;
    }

    .modal-form .form-actions {
        flex-direction: column;
    }
}
</style>

<?php
include 'includes/accounting_footer.php';
?>
