<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';
require_once 'purchase_order_functions.php';

// Check permissions - only admins and accounts users can manage receipts
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    header('Location: ../login.php');
    exit;
}

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['create_receipt'])) {
        // Create new receipt
        $po_id = $_POST['po_id'];
        $receipt_date = $_POST['receipt_date'];
        $notes = trim($_POST['notes']);
        $items = $_POST['items'] ?? [];

        if (empty($po_id) || empty($receipt_date)) {
            $error = "Purchase Order and Receipt Date are required.";
        } elseif (empty($items)) {
            $error = "At least one item must be received.";
        } else {
            try {
                $pdo->beginTransaction();

                // Generate receipt number
                $year = date('Y');
                $stmt = $pdo->prepare("SELECT MAX(CAST(SUBSTRING(receipt_number, 10) AS UNSIGNED)) as max_num FROM purchase_order_receipts WHERE receipt_number LIKE ?");
                $stmt->execute(["REC-$year-%"]);
                $result = $stmt->fetch();
                $next_num = ($result['max_num'] ?? 0) + 1;
                $receipt_number = "REC-$year-" . str_pad($next_num, 4, '0', STR_PAD_LEFT);

                // Calculate total received amount
                $total_received = 0;
                foreach ($items as $item) {
                    $total_received += floatval($item['quantity']) * floatval($item['unit_price']);
                }

                // Insert receipt
                $stmt = $pdo->prepare("
                    INSERT INTO purchase_order_receipts
                    (purchase_order_id, receipt_number, receipt_date, received_by, notes, total_received_amount)
                    VALUES (?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$po_id, $receipt_number, $receipt_date, $_SESSION['user_id'], $notes, $total_received]);
                $receipt_id = $pdo->lastInsertId();

                // Insert receipt items and update PO items
                foreach ($items as $item) {
                    $po_item_id = $item['po_item_id'];
                    $quantity_received = floatval($item['quantity']);
                    $unit_price = floatval($item['unit_price']);
                    $total_price = $quantity_received * $unit_price;

                    // Insert receipt item
                    $stmt = $pdo->prepare("
                        INSERT INTO purchase_order_receipt_items
                        (receipt_id, purchase_order_item_id, quantity_received, unit_price, total_price, notes)
                        VALUES (?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$receipt_id, $po_item_id, $quantity_received, $unit_price, $total_price, $item['notes'] ?? '']);

                    // Update PO item received quantity
                    $stmt = $pdo->prepare("
                        UPDATE purchase_order_items
                        SET quantity_received = quantity_received + ?
                        WHERE id = ?
                    ");
                    $stmt->execute([$quantity_received, $po_item_id]);
                }

                // Create accounting journal entry
                PurchaseOrderAccounting::createReceivingJournalEntry($pdo, $receipt_id, $_SESSION['user_id']);

                // Update PO status
                PurchaseOrderAccounting::updatePurchaseOrderStatus($pdo, $po_id);

                $pdo->commit();

                // Log activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'po_receipt_created',
                    "Created receipt $receipt_number for PO ID: $po_id"
                );

                $message = "Receipt created successfully!";

            } catch (Exception $e) {
                $pdo->rollBack();
                $error = "Error creating receipt: " . $e->getMessage();
            }
        }
    }
}

// Get all POs that can be received (approved, ordered, or partially received with remaining items)
try {
    $stmt = $pdo->query("
        SELECT po.id, po.po_number, po.supplier_id, po.order_date, po.expected_delivery_date, po.total_amount, po.status, po.notes,
               s.name AS supplier_name,
               (
                   SELECT COUNT(*) FROM purchase_order_items poi WHERE poi.purchase_order_id = po.id
               ) as total_items,
               (
                   SELECT SUM(poi.quantity_received) FROM purchase_order_items poi WHERE poi.purchase_order_id = po.id
               ) as total_received
        FROM purchase_orders po
        LEFT JOIN suppliers s ON po.supplier_id = s.id
        WHERE po.status IN ('approved', 'ordered', 'partially_received')
        AND EXISTS (
            SELECT 1 FROM purchase_order_items poi
            WHERE poi.purchase_order_id = po.id
            AND (poi.quantity_ordered - poi.quantity_received) > 0
        )
        ORDER BY po.order_date DESC
    ");
    $receivable_pos = $stmt->fetchAll();
} catch (PDOException $e) {
    $receivable_pos = [];
}

// Get PO items for AJAX
if (isset($_GET['get_po_items'])) {
    $po_id = $_GET['po_id'];
    try {
        $stmt = $pdo->prepare("
            SELECT poi.id, poi.item_name, poi.description, poi.quantity_ordered as quantity, poi.quantity_received,
                   poi.unit_price, poi.total_price,
                   (poi.quantity_ordered - poi.quantity_received) as remaining_quantity
            FROM purchase_order_items poi
            WHERE poi.purchase_order_id = ? AND (poi.quantity_ordered - poi.quantity_received) > 0
        ");
        $stmt->execute([$po_id]);
        $items = $stmt->fetchAll();

        header('Content-Type: application/json');
        echo json_encode($items);
        exit;
    } catch (PDOException $e) {
        header('Content-Type: application/json');
        echo json_encode(['error' => $e->getMessage()]);
        exit;
    }
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .receipt-container {
        max-width: 1400px;
        margin: 0 auto;
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .receipt-header {
        background: white;
        padding: 25px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .receipt-content {
        display: grid;
        grid-template-columns: 1fr 2fr;
        gap: 30px;
    }

    .create-receipt-section,
    .receipts-list-section {
        background: white;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .section-title {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
    }

    .form-group {
        margin-bottom: 20px;
    }

    .form-group label {
        display: block;
        font-weight: 600;
        margin-bottom: 5px;
        color: #495057;
    }

    .form-group input,
    .form-group select,
    .form-group textarea {
        width: 100%;
        padding: 10px 12px;
        border: 1px solid #ced4da;
        border-radius: 5px;
        font-size: 14px;
    }

    .form-group textarea {
        resize: vertical;
        min-height: 80px;
    }

    .btn {
        padding: 10px 20px;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        display: inline-block;
        margin-right: 10px;
    }

    .btn-primary {
        background: #007bff;
        color: white;
    }

    .btn-success {
        background: #28a745;
        color: white;
    }

    .btn-secondary {
        background: #6c757d;
        color: white;
    }

    .message {
        padding: 15px;
        border-radius: 5px;
        margin-bottom: 20px;
    }

    .success-message {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .error-message {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .receipts-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    .receipts-table th,
    .receipts-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .receipts-table th {
        background: #f8f9fa;
        font-weight: 600;
        color: #495057;
    }

    .po-number {
        font-weight: bold;
        color: #007bff;
    }

    .status-badge {
        display: inline-block;
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: 600;
    }

    .status-approved { background: #d4edda; color: #155724; }
    .status-ordered { background: #cce5ff; color: #004085; }
    .status-partially_received { background: #fff3cd; color: #856404; }

    .items-section {
        margin-top: 20px;
        padding: 20px;
        background: #f8f9fa;
        border-radius: 5px;
    }

    .item-row {
        display: grid;
        grid-template-columns: 2fr 1fr 1fr 1fr 2fr;
        gap: 10px;
        margin-bottom: 10px;
        align-items: center;
    }

    .item-row input {
        padding: 8px;
        border: 1px solid #ced4da;
        border-radius: 3px;
    }

    .item-header {
        font-weight: 600;
        color: #495057;
        margin-bottom: 10px;
    }

    @media (max-width: 768px) {
        .receipt-content {
            grid-template-columns: 1fr;
        }

        .item-row {
            grid-template-columns: 1fr;
        }
    }
</style>

<div class="receipt-container">
    <div class="receipt-header">
        <h1>📦 Purchase Order Receiving</h1>
        <p>Record receipt of goods and services from purchase orders</p>
    </div>

    <?php if ($message): ?>
        <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="receipt-content">
        <div class="create-receipt-section">
            <h2 class="section-title">➕ Create New Receipt</h2>
            <form method="post" id="receiptForm">
                <div class="form-group">
                    <label for="po_id">Purchase Order *</label>
                    <select name="po_id" id="po_id" required onchange="loadPOItems()">
                        <option value="">Select Purchase Order</option>
                        <?php foreach ($receivable_pos as $po): ?>
                            <option value="<?php echo $po['id']; ?>">
                                <?php echo htmlspecialchars($po['po_number']); ?> -
                                <?php echo htmlspecialchars($po['supplier_name']); ?> -
                                <?php echo number_format($po['total_amount'], 2); ?> GHS
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="receipt_date">Receipt Date *</label>
                    <input type="date" name="receipt_date" id="receipt_date" required>
                </div>

                <div class="form-group">
                    <label for="notes">Notes</label>
                    <textarea name="notes" id="notes" placeholder="Details about the receipt..."></textarea>
                </div>

                <div id="itemsSection" class="items-section" style="display: none;">
                    <h3 class="item-header">Items to Receive</h3>
                    <div id="itemsContainer"></div>
                </div>

                <button type="submit" name="create_receipt" class="btn btn-primary">Create Receipt</button>
            </form>
        </div>

        <div class="receipts-list-section">
            <h2 class="section-title">📋 Receivable Purchase Orders</h2>
            <table class="receipts-table">
                <thead>
                    <tr>
                        <th>PO Number</th>
                        <th>Supplier</th>
                        <th>Date</th>
                        <th>Total (GHS)</th>
                        <th>Status</th>
                        <th>Progress</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($receivable_pos as $po): ?>
                        <?php
                        // Fetch items for this PO
                        $items = [];
                        try {
                            $stmt = $pdo->prepare("SELECT item_name, quantity_ordered as quantity, quantity_received, unit, unit_price, total_price FROM purchase_order_items WHERE purchase_order_id = ?");
                            $stmt->execute([$po['id']]);
                            $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
                        } catch (PDOException $e) {
                            $items = [];
                        }
                        ?>
                        <tr>
                            <td class="po-number"><?php echo htmlspecialchars($po['po_number']); ?></td>
                            <td><?php echo htmlspecialchars($po['supplier_name'] ?? 'N/A'); ?></td>
                            <td><?php echo date('M j, Y', strtotime($po['order_date'])); ?></td>
                            <td><?php echo number_format($po['total_amount'], 2); ?></td>
                            <td>
                                <span class="status-badge status-<?php echo strtolower(str_replace('_', '-', $po['status'])); ?>">
                                    <?php echo ucfirst(str_replace('_', ' ', $po['status'])); ?>
                                </span>
                            </td>
                            <td>
                                <?php
                                $received = $po['total_received'] ?? 0;
                                $total = $po['total_items'] ?? 1;
                                $percentage = $total > 0 ? round(($received / $total) * 100) : 0;
                                echo "$received / $total items ($percentage%)";
                                ?>
                            </td>
                            <td>
                                <button class="btn btn-info btn-sm" onclick="viewPO(this)" data-po-id="<?php echo $po['id']; ?>" data-po-number="<?php echo htmlspecialchars($po['po_number']); ?>" data-supplier-name="<?php echo htmlspecialchars($po['supplier_name'] ?? 'N/A'); ?>" data-order-date="<?php echo htmlspecialchars($po['order_date']); ?>" data-total-amount="<?php echo $po['total_amount']; ?>" data-status="<?php echo htmlspecialchars($po['status']); ?>" data-items='<?php echo htmlspecialchars(json_encode($items)); ?>'>👁️ View</button>
                                <button class="btn btn-secondary btn-sm" onclick="selectPO(<?php echo $po['id']; ?>)">Receive</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal for viewing PO details -->
<div id="viewModal" class="modal" style="display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.4);">
    <div class="modal-content" style="background-color: #fefefe; margin: 5% auto; padding: 20px; border: 1px solid #888; width: 90%; max-width: 800px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2);">
        <span class="close" onclick="closeModal()" style="color: #aaa; float: right; font-size: 28px; font-weight: bold; cursor: pointer;">&times;</span>
        <h2 id="modalTitle" style="color: #2c3e50; margin-bottom: 20px;">Purchase Order Details</h2>
        <div id="modalBody">
            <!-- PO details will be populated here -->
        </div>
    </div>
</div>

<script>
function loadPOItems() {
    const poId = document.getElementById('po_id').value;
    if (!poId) {
        document.getElementById('itemsSection').style.display = 'none';
        return;
    }

    fetch(`?get_po_items=1&po_id=${poId}`)
        .then(response => response.json())
        .then(data => {
            if (data.error) {
                alert('Error loading items: ' + data.error);
                return;
            }

            const container = document.getElementById('itemsContainer');
            container.innerHTML = '';

            if (data.length === 0) {
                container.innerHTML = '<p>No items available for receipt.</p>';
                document.getElementById('itemsSection').style.display = 'none';
                return;
            }

            data.forEach(item => {
                const itemDiv = document.createElement('div');
                itemDiv.className = 'item-row';
                itemDiv.innerHTML = `
                    <div>
                        <strong>${item.item_name}</strong><br>
                        <small>${item.description || 'No description'}</small>
                    </div>
                    <div>
                        <label>Ordered: ${item.quantity}</label><br>
                        <label>Remaining: ${item.remaining_quantity}</label>
                    </div>
                    <div>
                        <input type="number" name="items[${item.id}][quantity]" placeholder="Qty to receive"
                               max="${item.remaining_quantity}" step="0.01" min="0" required>
                        <input type="hidden" name="items[${item.id}][po_item_id]" value="${item.id}">
                    </div>
                    <div>
                        <input type="number" name="items[${item.id}][unit_price]" placeholder="Unit Price"
                               step="0.01" min="0" value="${item.unit_price}" required>
                    </div>
                    <div>
                        <textarea name="items[${item.id}][notes]" placeholder="Notes" rows="2"></textarea>
                    </div>
                `;
                container.appendChild(itemDiv);
            });

            document.getElementById('itemsSection').style.display = 'block';
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading purchase order items');
        });
}

function selectPO(poId) {
    document.getElementById('po_id').value = poId;
    loadPOItems();
    document.getElementById('po_id').scrollIntoView();
}

// Set today's date as default for receipt date
document.addEventListener('DOMContentLoaded', function() {
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('receipt_date').value = today;
});

function viewPO(button) {
    const poId = button.getAttribute('data-po-id');
    const poNumber = button.getAttribute('data-po-number');
    const supplierName = button.getAttribute('data-supplier-name');
    const orderDate = button.getAttribute('data-order-date');
    const totalAmount = parseFloat(button.getAttribute('data-total-amount'));
    const status = button.getAttribute('data-status');
    const items = JSON.parse(button.getAttribute('data-items'));

    const modal = document.getElementById('viewModal');
    const modalTitle = document.getElementById('modalTitle');
    const modalBody = document.getElementById('modalBody');

    modalTitle.textContent = `Purchase Order: ${poNumber}`;

    let html = `
        <div style="margin-bottom: 20px;">
            <strong>Supplier:</strong> ${supplierName}<br>
            <strong>Order Date:</strong> ${new Date(orderDate).toLocaleDateString()}<br>
            <strong>Total Amount:</strong> ${totalAmount.toFixed(2)} GHS<br>
            <strong>Status:</strong> <span class="status-badge status-${status.toLowerCase().replace('_', '-')}">${status.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}</span>
        </div>
        <h3>Items</h3>
        <table style="width: 100%; border-collapse: collapse;">
            <thead>
                <tr style="background: #f8f9fa;">
                    <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">Item Name</th>
                    <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">Quantity</th>
                    <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">Received</th>
                    <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">Unit</th>
                    <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">Unit Price</th>
                    <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">Total</th>
                </tr>
            </thead>
            <tbody>
    `;

    items.forEach(item => {
        const remaining = item.quantity - item.quantity_received;
        const unitPrice = parseFloat(item.unit_price) || 0;
        const totalPrice = parseFloat(item.total_price) || 0;
        html += `
            <tr>
                <td style="padding: 8px; border: 1px solid #ddd;">${item.item_name}</td>
                <td style="padding: 8px; border: 1px solid #ddd;">${item.quantity}</td>
                <td style="padding: 8px; border: 1px solid #ddd;">${item.quantity_received} (${remaining} remaining)</td>
                <td style="padding: 8px; border: 1px solid #ddd;">${item.unit || 'N/A'}</td>
                <td style="padding: 8px; border: 1px solid #ddd;">${unitPrice.toFixed(2)} GHS</td>
                <td style="padding: 8px; border: 1px solid #ddd;">${totalPrice.toFixed(2)} GHS</td>
            </tr>
        `;
    });

    html += `
            </tbody>
        </table>
    `;

    modalBody.innerHTML = html;
    modal.style.display = 'block';
}

function closeModal() {
    document.getElementById('viewModal').style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('viewModal');
    if (event.target == modal) {
        modal.style.display = 'none';
    }
}
</script>

<?php include '../includes/admin_footer.php'; ?>
