<?php
require_once '../config.php';

echo "🔄 Recalculating Account Balances from Journal Entries\n";
echo "==================================================\n\n";

try {
    // Get all accounts
    $stmt = $pdo->query("SELECT id, account_code, account_name, account_type, normal_balance FROM chart_of_accounts ORDER BY account_code");
    $accounts = $stmt->fetchAll();

    echo "📊 Found " . count($accounts) . " accounts to process\n\n";

    $updated_count = 0;
    $total_difference = 0;

    foreach ($accounts as $account) {
        // Calculate balance from journal entries based on normal balance
        if ($account['normal_balance'] === 'debit') {
            // For debit normal accounts: balance = debits - credits
            $stmt = $pdo->prepare("
                SELECT COALESCE(SUM(
                    CASE
                        WHEN jel.debit_amount IS NOT NULL THEN jel.debit_amount
                        WHEN jel.credit_amount IS NOT NULL THEN -jel.credit_amount
                        ELSE 0
                    END
                ), 0) as calculated_balance
                FROM journal_entry_lines jel
                JOIN journal_entries je ON jel.journal_entry_id = je.id
                WHERE jel.account_id = ? AND je.status = 'posted'
            ");
        } else {
            // For credit normal accounts: balance = credits - debits
            $stmt = $pdo->prepare("
                SELECT COALESCE(SUM(
                    CASE
                        WHEN jel.debit_amount IS NOT NULL THEN -jel.debit_amount
                        WHEN jel.credit_amount IS NOT NULL THEN jel.credit_amount
                        ELSE 0
                    END
                ), 0) as calculated_balance
                FROM journal_entry_lines jel
                JOIN journal_entries je ON jel.journal_entry_id = je.id
                WHERE jel.account_id = ? AND je.status = 'posted'
            ");
        }
        $stmt->execute([$account['id']]);
        $result = $stmt->fetch();
        $calculated_balance = floatval($result['calculated_balance']);

        // Get current stored balance
        $stmt = $pdo->prepare("SELECT current_balance FROM chart_of_accounts WHERE id = ?");
        $stmt->execute([$account['id']]);
        $current_balance = floatval($stmt->fetch()['current_balance']);

        $difference = $calculated_balance - $current_balance;

        if (abs($difference) > 0.01) {
            // Update the balance
            $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = ? WHERE id = ?");
            $stmt->execute([$calculated_balance, $account['id']]);

            echo "✅ Updated {$account['account_code']} ({$account['account_name']}): " . number_format($current_balance, 2) . " → " . number_format($calculated_balance, 2) . " (Diff: " . number_format($difference, 2) . ")\n";

            $updated_count++;
            $total_difference += $difference;
        } else {
            echo "✓ {$account['account_code']} ({$account['account_name']}): " . number_format($current_balance, 2) . " (No change)\n";
        }
    }

    echo "\n📈 Summary:\n";
    echo "   Accounts updated: $updated_count\n";
    echo "   Total balance adjustment: " . number_format($total_difference, 2) . "\n\n";

    // Verify balance after recalculation
    echo "🔍 Verifying Balance Sheet Balance:\n";
    $stmt = $pdo->prepare("
        SELECT
            SUM(CASE WHEN account_type = 'asset' THEN current_balance ELSE 0 END) as total_assets,
            SUM(CASE WHEN account_type = 'liability' THEN current_balance ELSE 0 END) as total_liabilities,
            SUM(CASE WHEN account_type = 'equity' THEN current_balance ELSE 0 END) as total_equity
        FROM chart_of_accounts
    ");
    $stmt->execute();
    $balances = $stmt->fetch();

    $assets = floatval($balances['total_assets']);
    $liabilities = floatval($balances['total_liabilities']);
    $equity = floatval($balances['total_equity']);
    $balance_check = $assets - ($liabilities + $equity);

    echo "   Assets: " . number_format($assets, 2) . "\n";
    echo "   Liabilities: " . number_format($liabilities, 2) . "\n";
    echo "   Equity: " . number_format($equity, 2) . "\n";
    echo "   Balance Check: " . (abs($balance_check) < 0.01 ? "✅ Balanced" : "❌ Unbalanced (Difference: " . number_format($balance_check, 2) . ")") . "\n\n";

    if (abs($balance_check) < 0.01) {
        echo "🎉 Balance sheet is now balanced!\n";
    } else {
        echo "⚠️ Balance sheet is still unbalanced. Further investigation needed.\n";
    }

} catch (PDOException $e) {
    echo "❌ Error recalculating balances: " . $e->getMessage() . "\n";
}
?>
