<?php
// Purchase Order Accounting Functions

require_once 'accounting_functions.php';
require_once 'payment_methods.php';

class PurchaseOrderAccounting {

    /**
     * Create journal entries when PO is approved (budget commitment)
     * Dr: Encumbrance/Commitment Account
     * Cr: Budget Reserve Account
     */
    public static function createApprovalJournalEntry($pdo, $po_id, $user_id) {
        try {
            // Get PO details
            $stmt = $pdo->prepare("
                SELECT po.*, s.name as supplier_name
                FROM purchase_orders po
                JOIN suppliers s ON po.supplier_id = s.id
                WHERE po.id = ?
            ");
            $stmt->execute([$po_id]);
            $po = $stmt->fetch();

            if (!$po) {
                throw new Exception("Purchase order not found");
            }

            // Generate entry number
            $entry_number = generateEntryNumber($pdo, date('Y-m-d'));

            // Create journal entry
            $stmt = $pdo->prepare("
                INSERT INTO journal_entries
                (entry_number, transaction_date, description, source_type, source_id, status, created_by)
                VALUES (?, ?, ?, 'po_approval', ?, 'posted', ?)
            ");
            $stmt->execute([
                $entry_number,
                date('Y-m-d'),
                "PO Approval: {$po['po_number']} - {$po['supplier_name']}",
                $po_id,
                $user_id
            ]);
            $je_id = $pdo->lastInsertId();

            // Get accounts
            $commitment_account_id = getChartAccountId($pdo, 'Purchase Commitments');
            $budget_account_id = getChartAccountId($pdo, 'Budget Reserve');

            if (!$commitment_account_id || !$budget_account_id) {
                throw new Exception("Required accounts not found");
            }

            // Create journal entry lines
            // Dr: Purchase Commitments (encumbrance)
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, credit_amount, line_number)
                VALUES (?, ?, ?, 0, 1)
            ");
            $stmt->execute([$je_id, $commitment_account_id, $po['total_amount']]);

            // Cr: Budget Reserve
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, credit_amount, line_number)
                VALUES (?, ?, 0, ?, 2)
            ");
            $stmt->execute([$je_id, $budget_account_id, $po['total_amount']]);

            // Update account balances
            $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
            $stmt->execute([$po['total_amount'], $commitment_account_id]); // Debit increases liability

            $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
            $stmt->execute([$po['total_amount'], $budget_account_id]); // Credit decreases equity

            return $je_id;

        } catch (Exception $e) {
            error_log("Error creating PO approval journal entry: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Create journal entries when goods/services are received (accrual basis)
     * Dr: Expense/Asset Account
     * Cr: Accounts Payable
     */
    public static function createReceivingJournalEntry($pdo, $receipt_id, $user_id) {
        try {
            // Get receipt details with PO info
            $stmt = $pdo->prepare("
                SELECT r.*, po.po_number, po.supplier_id, s.name as supplier_name,
                       SUM(ri.total_price) as total_received
                FROM purchase_order_receipts r
                JOIN purchase_orders po ON r.purchase_order_id = po.id
                JOIN suppliers s ON po.supplier_id = s.id
                JOIN purchase_order_receipt_items ri ON r.id = ri.receipt_id
                WHERE r.id = ?
                GROUP BY r.id
            ");
            $stmt->execute([$receipt_id]);
            $receipt = $stmt->fetch();

            if (!$receipt) {
                throw new Exception("Receipt not found");
            }

            // Generate entry number
            $entry_number = generateEntryNumber($pdo, $receipt['receipt_date']);

            // Create journal entry
            $stmt = $pdo->prepare("
                INSERT INTO journal_entries
                (entry_number, transaction_date, description, source_type, source_id, status, created_by)
                VALUES (?, ?, ?, 'po_receipt', ?, 'posted', ?)
            ");
            $stmt->execute([
                $entry_number,
                $receipt['receipt_date'],
                "PO Receipt: {$receipt['po_number']} - {$receipt['supplier_name']} (Receipt: {$receipt['receipt_number']})",
                $receipt_id,
                $user_id
            ]);
            $je_id = $pdo->lastInsertId();

            // Get accounts
            $expense_account_id = getChartAccountId($pdo, 'Purchases/Expenses');
            $ap_account_id = getChartAccountId($pdo, 'Accounts Payable (Suppliers)');

            if (!$expense_account_id || !$ap_account_id) {
                throw new Exception("Required accounts not found");
            }

            // Create journal entry lines
            // Dr: Purchases/Expenses
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, credit_amount, line_number)
                VALUES (?, ?, ?, 0, 1)
            ");
            $stmt->execute([$je_id, $expense_account_id, $receipt['total_received']]);

            // Cr: Accounts Payable
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, credit_amount, line_number)
                VALUES (?, ?, 0, ?, 2)
            ");
            $stmt->execute([$je_id, $ap_account_id, $receipt['total_received']]);

            // Update account balances
            $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
            $stmt->execute([$receipt['total_received'], $expense_account_id]); // Debit increases expense

            $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
            $stmt->execute([$receipt['total_received'], $ap_account_id]); // Credit increases liability

            return $je_id;

        } catch (Exception $e) {
            error_log("Error creating PO receipt journal entry: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Create journal entries when supplier invoice is recorded
     * This may adjust the original receipt entry if amounts differ
     * Dr: Accounts Payable (additional amount)
     * Cr: Expense/Asset Account (additional amount)
     */
    public static function createInvoiceJournalEntry($pdo, $invoice_id, $user_id) {
        try {
            // Get invoice details
            $stmt = $pdo->prepare("
                SELECT si.*, po.po_number, s.name as supplier_name
                FROM supplier_invoices si
                LEFT JOIN purchase_orders po ON si.purchase_order_id = po.id
                JOIN suppliers s ON si.supplier_id = s.id
                WHERE si.id = ?
            ");
            $stmt->execute([$invoice_id]);
            $invoice = $stmt->fetch();

            if (!$invoice) {
                throw new Exception("Invoice not found");
            }

            // Check if there's a difference between received amount and invoiced amount
            $received_amount = 0;
            if ($invoice['purchase_order_id']) {
                $stmt = $pdo->prepare("
                    SELECT SUM(ri.total_price) as received_total
                    FROM purchase_order_receipts r
                    JOIN purchase_order_receipt_items ri ON r.id = ri.receipt_id
                    WHERE r.purchase_order_id = ?
                ");
                $stmt->execute([$invoice['purchase_order_id']]);
                $received = $stmt->fetch();
                $received_amount = $received['received_total'] ?? 0;
            }

            $difference = $invoice['total_amount'] - $received_amount;

            if (abs($difference) > 0.01) { // Only create entry if there's a significant difference
                // Generate entry number
                $entry_number = generateEntryNumber($pdo, $invoice['invoice_date']);

                // Create journal entry
                $stmt = $pdo->prepare("
                    INSERT INTO journal_entries
                    (entry_number, transaction_date, description, source_type, source_id, status, created_by)
                    VALUES (?, ?, ?, 'po_invoice', ?, 'posted', ?)
                ");
                $stmt->execute([
                    $entry_number,
                    $invoice['invoice_date'],
                    "Supplier Invoice: {$invoice['invoice_number']} - {$invoice['supplier_name']}" .
                    ($invoice['po_number'] ? " (PO: {$invoice['po_number']})" : ""),
                    $invoice_id,
                    $user_id
                ]);
                $je_id = $pdo->lastInsertId();

                // Get accounts
                $expense_account_id = getChartAccountId($pdo, 'Purchases/Expenses');
                $ap_account_id = getChartAccountId($pdo, 'Accounts Payable (Suppliers)');

                if (!$expense_account_id || !$ap_account_id) {
                    throw new Exception("Required accounts not found");
                }

                // Create journal entry lines for the difference
                if ($difference > 0) {
                    // Invoice > Received: Additional expense
                    // Dr: Purchases/Expenses
                    $stmt = $pdo->prepare("
                        INSERT INTO journal_entry_lines
                        (journal_entry_id, account_id, debit_amount, credit_amount, line_number)
                        VALUES (?, ?, ?, 0, 1)
                    ");
                    $stmt->execute([$je_id, $expense_account_id, $difference]);

                    // Cr: Accounts Payable
                    $stmt = $pdo->prepare("
                        INSERT INTO journal_entry_lines
                        (journal_entry_id, account_id, debit_amount, credit_amount, line_number)
                        VALUES (?, ?, 0, ?, 2)
                    ");
                    $stmt->execute([$je_id, $ap_account_id, $difference]);

                    // Update balances
                    $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
                    $stmt->execute([$difference, $expense_account_id]); // Debit increases expense

                    $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
                    $stmt->execute([$difference, $ap_account_id]); // Credit increases liability
                } else {
                    // Invoice < Received: Reduce expense (credit)
                    $abs_difference = abs($difference);

                    // Dr: Accounts Payable
                    $stmt = $pdo->prepare("
                        INSERT INTO journal_entry_lines
                        (journal_entry_id, account_id, debit_amount, credit_amount, line_number)
                        VALUES (?, ?, ?, 0, 1)
                    ");
                    $stmt->execute([$je_id, $ap_account_id, $abs_difference]);

                    // Cr: Purchases/Expenses
                    $stmt = $pdo->prepare("
                        INSERT INTO journal_entry_lines
                        (journal_entry_id, account_id, debit_amount, credit_amount, line_number)
                        VALUES (?, ?, 0, ?, 2)
                    ");
                    $stmt->execute([$je_id, $expense_account_id, $abs_difference]);

                    // Update balances
                    $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
                    $stmt->execute([$abs_difference, $ap_account_id]); // Debit decreases liability

                    $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
                    $stmt->execute([$abs_difference, $expense_account_id]); // Credit decreases expense
                }

                return $je_id;
            }

            return null; // No entry needed if amounts match

        } catch (Exception $e) {
            error_log("Error creating PO invoice journal entry: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Create journal entries when payment is made to supplier
     * Dr: Accounts Payable
     * Cr: Cash/Bank Account (based on payment method)
     */
    public static function createPaymentJournalEntry($pdo, $payment_id, $user_id) {
        try {
            // Get payment details
            $stmt = $pdo->prepare("
                SELECT pop.*, po.po_number, s.name as supplier_name, si.invoice_number
                FROM purchase_order_payments pop
                LEFT JOIN purchase_orders po ON pop.purchase_order_id = po.id
                LEFT JOIN suppliers s ON po.supplier_id = s.id
                LEFT JOIN supplier_invoices si ON pop.supplier_invoice_id = si.id
                WHERE pop.id = ?
            ");
            $stmt->execute([$payment_id]);
            $payment = $stmt->fetch();

            if (!$payment) {
                throw new Exception("Payment not found");
            }

            // Generate entry number
            $entry_number = generateEntryNumber($pdo, $payment['payment_date']);

            // Create journal entry
            $stmt = $pdo->prepare("
                INSERT INTO journal_entries
                (entry_number, transaction_date, description, source_type, source_id, status, created_by)
                VALUES (?, ?, ?, 'po_payment', ?, 'posted', ?)
            ");
            $stmt->execute([
                $entry_number,
                $payment['payment_date'],
                "PO Payment: " . ($payment['po_number'] ? "{$payment['po_number']} - " : "") .
                "{$payment['supplier_name']} - {$payment['payment_method']} ({$payment['payment_reference']})",
                $payment_id,
                $user_id
            ]);
            $je_id = $pdo->lastInsertId();

            // Get accounts
            $ap_account_id = getChartAccountId($pdo, 'Accounts Payable (Suppliers)');
            if (!$ap_account_id) {
                throw new Exception("Accounts Payable (Suppliers) account not found. Please ensure the account exists in the chart of accounts.");
            }

            // Get cash account based on payment method
            $cash_account_name = PaymentMethods::getCashAccountForPaymentMethod($payment['payment_method']);
            $cash_account_id = getChartAccountId($pdo, $cash_account_name);
            if (!$cash_account_id) {
                throw new Exception("Cash account '{$cash_account_name}' not found for payment method '{$payment['payment_method']}'. Please ensure the account exists in the chart of accounts.");
            }

            // Create journal entry lines
            // Dr: Accounts Payable
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, line_number, description)
                VALUES (?, ?, ?, 1, ?)
            ");
            $stmt->execute([$je_id, $ap_account_id, $payment['amount'], 'Accounts payable (PO payment)']);

            // Cr: Cash/Bank Account (based on payment method)
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, credit_amount, line_number, description)
                VALUES (?, ?, ?, 2, ?)
            ");
            $stmt->execute([$je_id, $cash_account_id, $payment['amount'], "Cash payment: PO {$payment['po_number']} - {$payment['supplier_name']} ({$cash_account_name})"]);

            // Update account balances
            $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
            $stmt->execute([$payment['amount'], $ap_account_id]); // Debit decreases liability

            $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
            $stmt->execute([$payment['amount'], $cash_account_id]); // Credit decreases asset

            return $je_id;

        } catch (Exception $e) {
            error_log("Error creating PO payment journal entry: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Update PO status based on workflow progress
     */
    public static function updatePurchaseOrderStatus($pdo, $po_id) {
        try {
            // Get current PO status and related data
            $stmt = $pdo->prepare("
                SELECT po.*,
                       COALESCE(SUM(poi.quantity_ordered), 0) as total_ordered,
                       COALESCE(SUM(poi.quantity_received), 0) as total_received,
                       COALESCE(SUM(poi.quantity_invoiced), 0) as total_invoiced,
                       COALESCE(SUM(poi.quantity_paid), 0) as total_paid
                FROM purchase_orders po
                LEFT JOIN purchase_order_items poi ON po.id = poi.purchase_order_id
                WHERE po.id = ?
                GROUP BY po.id
            ");
            $stmt->execute([$po_id]);
            $po = $stmt->fetch();

            if (!$po) {
                throw new Exception("Purchase order not found");
            }

            $new_status = $po['status'];

            // Determine new status based on workflow
            if (isset($po['approved_at']) && $po['approved_at'] && $po['status'] === 'pending_approval') {
                $new_status = 'approved';
            }

            if (isset($po['ordered_at']) && $po['ordered_at'] && in_array($new_status, ['approved', 'pending_approval'])) {
                $new_status = 'ordered';
            }

            if ($po['total_received'] > 0) {
                if ($po['total_received'] >= $po['total_ordered']) {
                    $new_status = 'received';
                } else {
                    $new_status = 'partially_received';
                }
            }

            if ($po['total_invoiced'] > 0) {
                if ($po['total_invoiced'] >= $po['total_ordered']) {
                    $new_status = 'invoiced';
                } else {
                    $new_status = 'partially_invoiced';
                }
            }

            if ($po['total_paid'] > 0) {
                if ($po['total_paid'] >= $po['total_ordered']) {
                    $new_status = 'paid';
                } else {
                    $new_status = 'partially_paid';
                }
            }

            // Check if PO should be closed
            if ($new_status === 'paid' && $po['total_received'] >= $po['total_ordered']) {
                $new_status = 'closed';
            }

            // Update status if changed
            if ($new_status !== $po['status']) {
                $stmt = $pdo->prepare("UPDATE purchase_orders SET status = ? WHERE id = ?");
                $stmt->execute([$new_status, $po_id]);

                // Log status change
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'] ?? null,
                    'po_status_changed',
                    "PO {$po['po_number']} status changed from {$po['status']} to {$new_status}"
                );
            }

            return $new_status;

        } catch (Exception $e) {
            error_log("Error updating PO status: " . $e->getMessage());
            throw $e;
        }
    }
}
?>
