<?php
require_once '../config.php';

$message = '';
$error = '';

// Get filter parameters
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-t');
$report_type = $_GET['report_type'] ?? 'monthly'; // monthly, quarterly, yearly

// Calculate date range based on report type
if ($report_type === 'yearly') {
    $start_date = date('Y-01-01');
    $end_date = date('Y-12-31');
} elseif ($report_type === 'quarterly') {
    $current_quarter = ceil(date('n') / 3);
    $start_date = date('Y-m-01', mktime(0, 0, 0, ($current_quarter - 1) * 3 + 1, 1));
    $end_date = date('Y-m-t', mktime(0, 0, 0, $current_quarter * 3, 1));
}

// Get profit & loss data using double-entry system
try {
    // Main Revenue from journal entries (including tuition fees)
    $stmt = $pdo->prepare("
        SELECT
            coa.account_name as category,
            COALESCE(SUM(jel.credit_amount), 0) as amount
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE je.transaction_date BETWEEN ? AND ?
        AND coa.account_type = 'revenue'
        AND jel.credit_amount > 0
        GROUP BY coa.id, coa.account_name
        ORDER BY amount DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $revenue_entries = $stmt->fetchAll();

    // Calculate totals
    $total_revenue = array_sum(array_column($revenue_entries, 'amount'));
    $total_income = $total_revenue;

    // Expenses from journal entries (debit entries to expense accounts)
    $stmt = $pdo->prepare("
        SELECT
            coa.account_name as category,
            COALESCE(SUM(jel.debit_amount), 0) as amount
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE je.transaction_date BETWEEN ? AND ?
        AND coa.account_type = 'expense'
        AND jel.debit_amount > 0
        GROUP BY coa.id, coa.account_name
        ORDER BY amount DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $expenses_by_category = $stmt->fetchAll();

    // Calculate total expenses
    $total_expenses = array_sum(array_column($expenses_by_category, 'amount'));

    // Calculate profit
    $gross_profit = $total_revenue - $total_expenses;
    $net_profit = $gross_profit; // For now, no taxes or other deductions

    // Revenue breakdown for chart
    $revenue_breakdown = $revenue_entries;

    // Expense breakdown for chart
    $expense_breakdown = array_filter($expenses_by_category, function($expense) {
        return $expense['amount'] > 0;
    });

    // Profit trends (last 6 months) - using journal entries
    $profit_trends = [];
    for ($i = 5; $i >= 0; $i--) {
        $month_start = date('Y-m-01', strtotime("-{$i} months"));
        $month_end = date('Y-m-t', strtotime("-{$i} months"));
        $month_name = date('M Y', strtotime("-{$i} months"));

        // Get revenue from journal entries (credit entries to revenue accounts)
        $stmt = $pdo->prepare("
            SELECT COALESCE(SUM(jel.credit_amount), 0) as revenue
            FROM journal_entries je
            JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
            JOIN chart_of_accounts coa ON jel.account_id = coa.id
            WHERE je.transaction_date BETWEEN ? AND ?
            AND coa.account_type = 'revenue'
            AND jel.credit_amount > 0
        ");
        $stmt->execute([$month_start, $month_end]);
        $revenue = $stmt->fetch()['revenue'];

        // Get expenses from journal entries (debit entries to expense accounts)
        $stmt = $pdo->prepare("
            SELECT COALESCE(SUM(jel.debit_amount), 0) as expenses
            FROM journal_entries je
            JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
            JOIN chart_of_accounts coa ON jel.account_id = coa.id
            WHERE je.transaction_date BETWEEN ? AND ?
            AND coa.account_type = 'expense'
            AND jel.debit_amount > 0
        ");
        $stmt->execute([$month_start, $month_end]);
        $expenses = $stmt->fetch()['expenses'];

        $profit = $revenue - $expenses;

        $profit_trends[] = [
            'month' => $month_name,
            'revenue' => $revenue,
            'expenses' => $expenses,
            'profit' => $profit
        ];
    }

} catch (PDOException $e) {
    $error = "Error fetching profit & loss data: " . $e->getMessage();
}

include 'includes/accounting_header.php';
?>

<div class="profit-loss-container">
    <div class="report-header">
        <h1>📈 Profit & Loss Statement</h1>
        <div class="report-filters">
            <form method="get" class="filter-form">
                <div class="filter-group">
                    <label for="report_type">Report Type:</label>
                    <select name="report_type" id="report_type" onchange="this.form.submit()">
                        <option value="monthly" <?php echo $report_type === 'monthly' ? 'selected' : ''; ?>>Monthly</option>
                        <option value="quarterly" <?php echo $report_type === 'quarterly' ? 'selected' : ''; ?>>Quarterly</option>
                        <option value="yearly" <?php echo $report_type === 'yearly' ? 'selected' : ''; ?>>Yearly</option>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="start_date">Start Date:</label>
                    <input type="date" name="start_date" id="start_date" value="<?php echo $start_date; ?>" onchange="this.form.submit()">
                </div>

                <div class="filter-group">
                    <label for="end_date">End Date:</label>
                    <input type="date" name="end_date" id="end_date" value="<?php echo $end_date; ?>" onchange="this.form.submit()">
                </div>

                <a href="export_profit_loss.php?<?php echo http_build_query($_GET); ?>" class="btn btn-secondary">Export</a>
            </form>
        </div>
    </div>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Profit & Loss Summary -->
    <div class="summary-cards">
        <div class="card revenue-card">
            <div class="card-icon">💰</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($total_income, 2); ?></h3>
                <p>Total Income</p>
                <small>Revenue</small>
            </div>
        </div>

        <div class="card expenses-card">
            <div class="card-icon">💸</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($total_expenses, 2); ?></h3>
                <p>Total Expenses</p>
                <small>Operating costs</small>
            </div>
        </div>

        <div class="card gross-profit-card <?php echo $gross_profit >= 0 ? 'positive' : 'negative'; ?>">
            <div class="card-icon">📊</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($gross_profit, 2); ?></h3>
                <p>Gross Profit</p>
                <small>Revenue - Expenses</small>
            </div>
        </div>

        <div class="card net-profit-card <?php echo $net_profit >= 0 ? 'positive' : 'negative'; ?>">
            <div class="card-icon"><?php echo $net_profit >= 0 ? '📈' : '📉'; ?></div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($net_profit, 2); ?></h3>
                <p>Net Profit</p>
                <small>Bottom line</small>
            </div>
        </div>

        <div class="card profit-margin-card">
            <div class="card-icon">🎯</div>
            <div class="card-content">
                <h3><?php echo $total_revenue > 0 ? round(($net_profit / $total_revenue) * 100, 1) : 0; ?>%</h3>
                <p>Profit Margin</p>
                <small>Net profit as % of revenue</small>
            </div>
        </div>

        <div class="card expense-ratio-card">
            <div class="card-icon">📉</div>
            <div class="card-content">
                <h3><?php echo $total_revenue > 0 ? round(($total_expenses / $total_revenue) * 100, 1) : 0; ?>%</h3>
                <p>Expense Ratio</p>
                <small>Expenses as % of revenue</small>
            </div>
        </div>
    </div>

    <!-- Profit & Loss Statement -->
    <div class="report-section">
        <h2>Statement of Profit & Loss</h2>
        <p class="report-period">For the period: <?php echo date('M d, Y', strtotime($start_date)); ?> to <?php echo date('M d, Y', strtotime($end_date)); ?></p>

        <!-- Revenue Section -->
        <div class="pl-section">
            <h3>Revenue</h3>
            <div class="table-container">
                <table class="pl-table">
                    <thead>
                        <tr>
                            <th>Revenue Source</th>
                            <th>Amount (GHC)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($revenue_breakdown as $revenue): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($revenue['category']); ?></td>
                                <td class="positive"><?php echo number_format($revenue['amount'], 2); ?></td>
                            </tr>
                        <?php endforeach; ?>
                        <tr class="total-row">
                            <td><strong>Total Revenue</strong></td>
                            <td class="positive"><strong><?php echo number_format($total_revenue, 2); ?></strong></td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Expenses Section -->
        <div class="pl-section">
            <h3>Expenses</h3>
            <div class="table-container">
                <table class="pl-table">
                    <thead>
                        <tr>
                            <th>Expense Category</th>
                            <th>Amount (GHC)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($expense_breakdown as $expense): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($expense['category']); ?></td>
                                <td class="negative"><?php echo number_format($expense['amount'], 2); ?></td>
                            </tr>
                        <?php endforeach; ?>
                        <tr class="total-row">
                            <td><strong>Total Expenses</strong></td>
                            <td class="negative"><strong><?php echo number_format($total_expenses, 2); ?></strong></td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Profit Section -->
        <div class="pl-section">
            <h3>Profit</h3>
            <div class="table-container">
                <table class="pl-table">
                    <thead>
                        <tr>
                            <th>Description</th>
                            <th>Amount (GHC)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td>Gross Profit (Revenue - Expenses)</td>
                            <td class="<?php echo $gross_profit >= 0 ? 'positive' : 'negative'; ?>">
                                <strong><?php echo number_format($gross_profit, 2); ?></strong>
                            </td>
                        </tr>
                        <tr class="final-profit">
                            <td><strong>Net Profit</strong></td>
                            <td class="<?php echo $net_profit >= 0 ? 'positive' : 'negative'; ?>">
                                <strong><?php echo number_format($net_profit, 2); ?></strong>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Charts Section -->
    <div class="charts-section">
        <div class="chart-container">
            <h3>Revenue Breakdown</h3>
            <div id="revenueChartContainer">
                <?php if (empty($revenue_breakdown)): ?>
                    <p class="no-data">No revenue data available for the selected period.</p>
                <?php else: ?>
                    <canvas id="revenueChart" width="400" height="200"></canvas>
                <?php endif; ?>
            </div>
        </div>

        <div class="chart-container">
            <h3>Expenses by Category</h3>
            <div id="expensesChartContainer">
                <?php if (empty($expense_breakdown)): ?>
                    <p class="no-data">No expense data available for the selected period.</p>
                <?php else: ?>
                    <canvas id="expensesChart" width="400" height="200"></canvas>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Profit Trends Chart -->
    <div class="report-section">
        <h2>Profit Trends (Last 6 Months)</h2>
        <div class="chart-container">
            <canvas id="profitTrendChart" width="400" height="200"></canvas>
        </div>
    </div>

    <!-- Profitability Analysis -->
    <div class="report-section">
        <h2>💡 Profitability Analysis</h2>
        <div class="analysis-grid">
            <div class="analysis-card">
                <h4>Profitability Status</h4>
                <p class="<?php echo $net_profit >= 0 ? 'positive' : 'negative'; ?>">
                    <?php echo $net_profit >= 0 ? 'Profitable' : 'Loss Making'; ?>
                </p>
                <small>
                    <?php
                    if ($net_profit >= 0) {
                        echo 'Business is generating profit - GHC ' . number_format($net_profit, 2) . ' net profit';
                    } else {
                        echo 'Business is operating at a loss - GHC ' . number_format(abs($net_profit), 2) . ' net loss';
                    }
                    ?>
                </small>
            </div>

            <div class="analysis-card">
                <h4>Revenue vs Expenses</h4>
                <p class="<?php echo $total_revenue >= $total_expenses ? 'positive' : 'negative'; ?>">
                    <?php echo $total_revenue >= $total_expenses ? 'Revenue > Expenses' : 'Expenses > Revenue'; ?>
                </p>
                <small>
                    <?php
                    if ($total_revenue >= $total_expenses) {
                        echo 'Healthy financial position with GHC ' . number_format($total_revenue - $total_expenses, 2) . ' surplus';
                    } else {
                        echo 'Needs attention - expenses exceed revenue by GHC ' . number_format($total_expenses - $total_revenue, 2);
                    }
                    ?>
                </small>
            </div>

            <div class="analysis-card">
                <h4>Break-even Analysis</h4>
                <p class="info">
                    <?php
                    $break_even_ratio = $total_expenses > 0 ? ($total_revenue / $total_expenses) * 100 : 0;
                    echo round($break_even_ratio, 1) . '% of break-even';
                    ?>
                </p>
                <small>
                    <?php
                    if ($break_even_ratio >= 100) echo 'Above break-even point - profitable';
                    elseif ($break_even_ratio >= 80) echo 'Close to break-even - monitor closely';
                    else echo 'Below break-even - needs immediate attention';
                    ?>
                </small>
            </div>

            <div class="analysis-card">
                <h4>Recommendations</h4>
                <p class="info">
                    <?php
                    if ($net_profit < 0) {
                        echo 'Reduce expenses or increase revenue';
                    } elseif ($net_profit > 0 && $net_profit < $total_revenue * 0.1) {
                        echo 'Improve efficiency and cost control';
                    } else {
                        echo 'Maintain current performance';
                    }
                    ?>
                </p>
                <small>
                    <?php
                    if ($net_profit < 0) {
                        echo 'Focus on cutting unnecessary costs and exploring new revenue streams';
                    } elseif ($net_profit > 0 && $net_profit < $total_revenue * 0.1) {
                        echo 'Look for opportunities to optimize operations and reduce waste';
                    } else {
                        echo 'Continue with successful strategies and consider expansion opportunities';
                    }
                    ?>
                </small>
            </div>
        </div>
    </div>
</div>

<style>
.profit-loss-container {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.report-header {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.report-header h1 {
    color: #2c3e50;
    margin: 0 0 20px 0;
}

.filter-form {
    display: flex;
    gap: 20px;
    align-items: end;
    flex-wrap: wrap;
}

.filter-group {
    display: flex;
    flex-direction: column;
    min-width: 150px;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.filter-group select,
.filter-group input {
    padding: 8px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.card {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
    transition: transform 0.2s;
}

.card:hover {
    transform: translateY(-5px);
}

.card-icon {
    font-size: 2.5em;
    margin-bottom: 15px;
}

.card h3 {
    color: #2c3e50;
    font-size: 1.8em;
    margin-bottom: 5px;
}

.card p {
    color: #7f8c8d;
    font-size: 1.1em;
    margin: 0;
}

.card small {
    color: #95a5a6;
    font-size: 0.9em;
}

.revenue-card .card-icon { color: #27ae60; }
.expenses-card .card-icon { color: #e74c3c; }
.gross-profit-card.positive .card-icon,
.net-profit-card.positive .card-icon { color: #27ae60; }
.gross-profit-card.negative .card-icon,
.net-profit-card.negative .card-icon { color: #e74c3c; }
.profit-margin-card .card-icon,
.expense-ratio-card .card-icon { color: #3498db; }

.report-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.report-section h2 {
    color: #2c3e50;
    margin-bottom: 20px;
    border-bottom: 2px solid #e9ecef;
    padding-bottom: 10px;
}

.report-period {
    color: #7f8c8d;
    font-style: italic;
    margin-bottom: 30px;
}

.pl-section {
    margin-bottom: 40px;
}

.pl-section h3 {
    color: #34495e;
    margin-bottom: 20px;
    padding: 10px 15px;
    background: #f8f9fa;
    border-radius: 5px;
}

.table-container {
    overflow-x: auto;
}

.pl-table {
    width: 100%;
    border-collapse: collapse;
}

.pl-table th,
.pl-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.pl-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.pl-table tr:hover {
    background: #f8f9fa;
}

.positive {
    color: #27ae60;
    font-weight: 600;
}

.negative {
    color: #e74c3c;
    font-weight: 600;
}

.total-row {
    background: #f8f9fa;
    font-weight: 600;
    border-top: 2px solid #dee2e6;
}

.final-profit {
    background: #2c5aa0;
    color: white;
    font-weight: 600;
}

.charts-section {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: 30px;
    margin-bottom: 30px;
}

.chart-container {
    background: white;
    padding: 20px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.chart-container h3 {
    margin-bottom: 20px;
    color: #2c3e50;
}

.no-data {
    color: #7f8c8d;
    font-style: italic;
    padding: 40px;
    margin: 0;
}

.analysis-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
}

.analysis-card {
    background: #f8f9fa;
    padding: 20px;
    border-radius: 8px;
    border-left: 4px solid #3498db;
}

.analysis-card h4 {
    margin: 0 0 10px 0;
    color: #2c3e50;
}

.analysis-card p {
    font-size: 1.5em;
    font-weight: 600;
    margin: 0 0 10px 0;
}

.analysis-card.info p { color: #3498db; }
.analysis-card.positive p { color: #27ae60; }
.analysis-card.negative p { color: #e74c3c; }

.analysis-card small {
    color: #7f8c8d;
    line-height: 1.4;
}

.error-message {
    background: #f8d7da;
    color: #721c24;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #f5c6cb;
}

@media (max-width: 768px) {
    .filter-form {
        flex-direction: column;
        align-items: stretch;
    }

    .filter-group {
        min-width: auto;
    }

    .summary-cards {
        grid-template-columns: 1fr;
    }

    .charts-section {
        grid-template-columns: 1fr;
    }

    .analysis-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Revenue Breakdown Chart
    <?php if (!empty($revenue_breakdown)): ?>
    const revenueCtx = document.getElementById('revenueChart').getContext('2d');
    const revenueLabels = <?php echo json_encode(array_column($revenue_breakdown, 'category')); ?>;
    const revenueData = <?php echo json_encode(array_column($revenue_breakdown, 'amount')); ?>;

    new Chart(revenueCtx, {
        type: 'pie',
        data: {
            labels: revenueLabels,
            datasets: [{
                data: revenueData,
                backgroundColor: [
                    'rgba(46, 204, 113, 0.8)',
                    'rgba(52, 152, 219, 0.8)',
                    'rgba(155, 89, 182, 0.8)'
                ],
                borderColor: [
                    'rgba(46, 204, 113, 1)',
                    'rgba(52, 152, 219, 1)',
                    'rgba(155, 89, 182, 1)'
                ],
                borderWidth: 2
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
    <?php endif; ?>

    // Expenses Chart
    <?php if (!empty($expense_breakdown)): ?>
    const expensesCtx = document.getElementById('expensesChart').getContext('2d');
    const expenseLabels = <?php echo json_encode(array_column($expense_breakdown, 'category')); ?>;
    const expenseData = <?php echo json_encode(array_column($expense_breakdown, 'amount')); ?>;

    new Chart(expensesCtx, {
        type: 'bar',
        data: {
            labels: expenseLabels,
            datasets: [{
                label: 'Expenses (GHC)',
                data: expenseData,
                backgroundColor: 'rgba(231, 76, 60, 0.6)',
                borderColor: 'rgba(231, 76, 60, 1)',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return 'GHC ' + value.toLocaleString();
                        }
                    }
                }
            }
        }
    });
    <?php endif; ?>

    // Profit Trend Chart
    const profitTrendCtx = document.getElementById('profitTrendChart').getContext('2d');
    const trendLabels = <?php echo json_encode(array_column($profit_trends, 'month')); ?>;
    const revenueTrendData = <?php echo json_encode(array_column($profit_trends, 'revenue')); ?>;
    const expenseTrendData = <?php echo json_encode(array_column($profit_trends, 'expenses')); ?>;
    const profitTrendData = <?php echo json_encode(array_column($profit_trends, 'profit')); ?>;

    new Chart(profitTrendCtx, {
        type: 'line',
        data: {
            labels: trendLabels,
            datasets: [
                {
                    label: 'Revenue',
                    data: revenueTrendData,
                    borderColor: 'rgba(46, 204, 113, 1)',
                    backgroundColor: 'rgba(46, 204, 113, 0.1)',
                    tension: 0.4
                },
                {
                    label: 'Expenses',
                    data: expenseTrendData,
                    borderColor: 'rgba(231, 76, 60, 1)',
                    backgroundColor: 'rgba(231, 76, 60, 0.1)',
                    tension: 0.4
                },
                {
                    label: 'Profit',
                    data: profitTrendData,
                    borderColor: 'rgba(52, 152, 219, 1)',
                    backgroundColor: 'rgba(52, 152, 219, 0.1)',
                    tension: 0.4
                }
            ]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    ticks: {
                        callback: function(value) {
                            return 'GHC ' + value.toLocaleString();
                        }
                    }
                }
            },
            plugins: {
                legend: {
                    position: 'top'
                }
            }
        }
    });
});
</script>

<?php
include '../includes/admin_footer.php';
?>
