<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';
require_once 'purchase_order_functions.php';

// Check permissions - only admins and accounts users can manage payments
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    header('Location: ../login.php');
    exit;
}

$message = '';
$error = '';

if (isset($_GET['success']) && $_GET['success'] == '1') {
    $message = "Payment recorded successfully!";
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['create_payment'])) {
        // Create new payment
        $supplier_id = $_POST['supplier_id'];
        $po_id = $_POST['po_id'] ?: null;
        $invoice_id = $_POST['invoice_id'] ?: null;
        $payment_date = $_POST['payment_date'];
        $payment_method = $_POST['payment_method'];
        $payment_reference = trim($_POST['payment_reference']);
        $amount = floatval($_POST['amount']);
        $notes = trim($_POST['notes']);

        if (empty($supplier_id) || empty($payment_date) || empty($payment_method) || $amount <= 0) {
            $error = "Supplier, Payment Date, Method, and Amount are required.";
        } else {
            try {
                $pdo->beginTransaction();

                // Insert payment
                $stmt = $pdo->prepare("
                    INSERT INTO purchase_order_payments
                    (supplier_id, purchase_order_id, supplier_invoice_id, payment_date, payment_method, payment_reference, amount, notes, processed_by)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$supplier_id, $po_id, $invoice_id, $payment_date, $payment_method, $payment_reference, $amount, $notes, $_SESSION['user_id']]);
                $payment_id = $pdo->lastInsertId();

                // Create accounting journal entry
                PurchaseOrderAccounting::createPaymentJournalEntry($pdo, $payment_id, $_SESSION['user_id']);

                // Update PO item payment quantities if linked to PO
                if ($po_id) {
                    // Get payment amount and distribute across PO items proportionally
                    $stmt = $pdo->prepare("
                        SELECT poi.id, poi.quantity_invoiced, poi.quantity_paid, poi.unit_price,
                               (poi.quantity_invoiced - poi.quantity_paid) as remaining_to_pay
                        FROM purchase_order_items poi
                        WHERE poi.purchase_order_id = ? AND (poi.quantity_invoiced - poi.quantity_paid) > 0
                    ");
                    $stmt->execute([$po_id]);
                    $po_items = $stmt->fetchAll();

                    $remaining_payment = $amount;
                    foreach ($po_items as $item) {
                        if ($remaining_payment <= 0) break;

                        $item_payment = min($remaining_payment, $item['remaining_to_pay'] * $item['unit_price']);
                        if ($item_payment > 0) {
                            $quantity_to_pay = $item_payment / $item['unit_price'];
                            $stmt = $pdo->prepare("
                                UPDATE purchase_order_items
                                SET quantity_paid = quantity_paid + ?
                                WHERE id = ?
                            ");
                            $stmt->execute([$quantity_to_pay, $item['id']]);
                            $remaining_payment -= $item_payment;
                        }
                    }

                    // Update PO status
                    PurchaseOrderAccounting::updatePurchaseOrderStatus($pdo, $po_id);
                }

                // Update invoice status if linked
                if ($invoice_id) {
                    // Check if invoice is fully paid
                    $stmt = $pdo->prepare("
                        SELECT si.total_amount,
                               COALESCE(SUM(pop.amount), 0) as total_paid
                        FROM supplier_invoices si
                        LEFT JOIN purchase_order_payments pop ON si.id = pop.supplier_invoice_id
                        WHERE si.id = ?
                        GROUP BY si.id
                    ");
                    $stmt->execute([$invoice_id]);
                    $invoice_payment = $stmt->fetch();

                    if ($invoice_payment && $invoice_payment['total_paid'] >= $invoice_payment['total_amount']) {
                        $stmt = $pdo->prepare("UPDATE supplier_invoices SET status = 'paid' WHERE id = ?");
                        $stmt->execute([$invoice_id]);
                    }
                }

                $pdo->commit();

                // Log activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'po_payment_created',
                    "Created payment of GHS $amount to supplier ID: $supplier_id" .
                    ($po_id ? " for PO ID: $po_id" : "") .
                    ($invoice_id ? " for Invoice ID: $invoice_id" : "")
                );

                $message = "Payment recorded successfully!";
                header("Location: po_payments.php?success=1");
                exit;

            } catch (Exception $e) {
                $pdo->rollBack();
                $error = "Error creating payment: " . $e->getMessage();
            }
        }
    }
}

// Get all suppliers for dropdown
try {
    $stmt = $pdo->query("SELECT id, name FROM suppliers WHERE status = 'active' ORDER BY name");
    $suppliers = $stmt->fetchAll();
} catch (PDOException $e) {
    $suppliers = [];
}

// Get all POs that can have payments (received or invoiced)
try {
    $stmt = $pdo->query("
        SELECT po.id, po.po_number, po.supplier_id, s.name AS supplier_name,
               po.total_amount, po.status,
               COALESCE(SUM(poi.quantity_paid * poi.unit_price), 0) as total_paid
        FROM purchase_orders po
        JOIN suppliers s ON po.supplier_id = s.id
        LEFT JOIN purchase_order_items poi ON po.id = poi.purchase_order_id
        WHERE po.status IN ('received', 'partially_received', 'invoiced', 'partially_invoiced', 'partially_paid')
        GROUP BY po.id, po.po_number, po.supplier_id, s.name, po.total_amount, po.status
        HAVING po.total_amount > total_paid
        ORDER BY po.order_date DESC
    ");
    $payable_pos = $stmt->fetchAll();
} catch (PDOException $e) {
    $payable_pos = [];
}

// Get all approved invoices that are not fully paid
try {
    $stmt = $pdo->query("
        SELECT si.id, si.invoice_number, si.supplier_id, s.name AS supplier_name,
               si.total_amount, si.due_date,
               COALESCE(SUM(pop.amount), 0) as total_paid,
               (si.total_amount - COALESCE(SUM(pop.amount), 0)) as remaining_amount
        FROM supplier_invoices si
        JOIN suppliers s ON si.supplier_id = s.id
        LEFT JOIN purchase_order_payments pop ON si.id = pop.supplier_invoice_id
        WHERE si.status IN ('pending_payment', 'partially_paid')
        GROUP BY si.id, si.invoice_number, si.supplier_id, s.name, si.total_amount, si.due_date
        HAVING remaining_amount > 0
        ORDER BY si.due_date ASC
    ");
    $payable_invoices = $stmt->fetchAll();
} catch (PDOException $e) {
    $payable_invoices = [];
}

// Get all payments
try {
    $stmt = $pdo->query("
        SELECT pop.*, s.name AS supplier_name, po.po_number, si.invoice_number,
               u.name AS processed_by_name
        FROM purchase_order_payments pop
        LEFT JOIN suppliers s ON pop.supplier_id = s.id
        LEFT JOIN purchase_orders po ON pop.purchase_order_id = po.id
        LEFT JOIN supplier_invoices si ON pop.supplier_invoice_id = si.id
        LEFT JOIN users u ON pop.processed_by = u.id
        ORDER BY pop.payment_date DESC
    ");
    $payments = $stmt->fetchAll();
} catch (PDOException $e) {
    $payments = [];
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .payment-container {
        max-width: 1400px;
        margin: 0 auto;
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .payment-header {
        background: white;
        padding: 25px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .payment-content {
        display: grid;
        grid-template-columns: 1fr 2fr;
        gap: 30px;
    }

    .create-payment-section,
    .payments-list-section {
        background: white;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .section-title {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
    }

    .form-group {
        margin-bottom: 20px;
    }

    .form-group label {
        display: block;
        font-weight: 600;
        margin-bottom: 5px;
        color: #495057;
    }

    .form-group input,
    .form-group select,
    .form-group textarea {
        width: 100%;
        padding: 10px 12px;
        border: 1px solid #ced4da;
        border-radius: 5px;
        font-size: 14px;
    }

    .form-group textarea {
        resize: vertical;
        min-height: 80px;
    }

    .form-row {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px;
    }

    .btn {
        padding: 10px 20px;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        display: inline-block;
        margin-right: 10px;
    }

    .btn-primary {
        background: #007bff;
        color: white;
    }

    .btn-success {
        background: #28a745;
        color: white;
    }

    .btn-secondary {
        background: #6c757d;
        color: white;
    }

    .message {
        padding: 15px;
        border-radius: 5px;
        margin-bottom: 20px;
    }

    .success-message {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .error-message {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .payments-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    .payments-table th,
    .payments-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .payments-table th {
        background: #f8f9fa;
        font-weight: 600;
        color: #495057;
    }

    .payment-reference {
        font-weight: bold;
        color: #007bff;
    }

    .amount-positive {
        color: #28a745;
        font-weight: 600;
    }

    .amount-negative {
        color: #dc3545;
        font-weight: 600;
    }

    .overdue {
        color: #dc3545;
        font-weight: 600;
    }

    .payable-section {
        margin-top: 20px;
        padding: 20px;
        background: #f8f9fa;
        border-radius: 5px;
    }

    .payable-item {
        display: grid;
        grid-template-columns: 2fr 1fr 1fr 1fr;
        gap: 10px;
        padding: 10px;
        margin-bottom: 10px;
        background: white;
        border-radius: 5px;
        align-items: center;
    }

    .payable-item:hover {
        background: #e9ecef;
        cursor: pointer;
    }

    @media (max-width: 768px) {
        .payment-content {
            grid-template-columns: 1fr;
        }

        .form-row {
            grid-template-columns: 1fr;
        }

        .payable-item {
            grid-template-columns: 1fr;
        }
    }
</style>

<div class="payment-container">
    <div class="payment-header">
        <h1>💰 Purchase Order Payments</h1>
        <p>Record and manage payments to suppliers</p>
    </div>

    <?php if ($message): ?>
        <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="payment-content">
        <div class="create-payment-section">
            <h2 class="section-title">➕ Record New Payment</h2>
            <form method="post">
                <div class="form-group">
                    <label for="supplier_id">Supplier *</label>
                    <select name="supplier_id" id="supplier_id" required onchange="updatePayableOptions()">
                        <option value="">Select Supplier</option>
                        <?php foreach ($suppliers as $supplier): ?>
                            <option value="<?php echo $supplier['id']; ?>"><?php echo htmlspecialchars($supplier['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="po_id">Purchase Order (Optional)</label>
                        <select name="po_id" id="po_id" onchange="updateInvoiceOptions(); updatePaymentAmount()">
                            <option value="">Select Purchase Order</option>
                            <?php foreach ($payable_pos as $po): ?>
                                <option value="<?php echo $po['id']; ?>" data-supplier="<?php echo $po['supplier_id']; ?>"
                                        data-remaining="<?php echo ($po['total_amount'] - $po['total_paid']); ?>">
                                    <?php echo htmlspecialchars($po['po_number']); ?> - <?php echo htmlspecialchars($po['supplier_name']); ?>
                                    (Remaining: GHS <?php echo number_format($po['total_amount'] - $po['total_paid'], 2); ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="invoice_id">Supplier Invoice (Optional)</label>
                        <select name="invoice_id" id="invoice_id" onchange="updatePaymentAmount()">
                            <option value="">Select Invoice</option>
                            <?php foreach ($payable_invoices as $invoice): ?>
                                <option value="<?php echo $invoice['id']; ?>" data-supplier="<?php echo $invoice['supplier_id']; ?>"
                                        data-po="<?php echo $invoice['purchase_order_id'] ?? ''; ?>"
                                        data-remaining="<?php echo $invoice['remaining_amount']; ?>">
                                    <?php echo htmlspecialchars($invoice['invoice_number']); ?> - <?php echo htmlspecialchars($invoice['supplier_name']); ?>
                                    (Remaining: GHS <?php echo number_format($invoice['remaining_amount'], 2); ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="payment_date">Payment Date *</label>
                        <input type="date" name="payment_date" id="payment_date" required>
                    </div>

                    <div class="form-group">
                        <label for="payment_method">Payment Method *</label>
                        <select name="payment_method" id="payment_method" required>
                            <option value="">Select Method</option>
                            <option value="cash">Cash</option>
                            <option value="bank_transfer">Bank Transfer</option>
                            <option value="cheque">Cheque</option>
                            <option value="credit_card">Credit Card</option>
                            <option value="debit_card">Debit Card</option>
                            <option value="mobile_money">Mobile Money</option>
                        </select>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="payment_reference">Payment Reference *</label>
                        <input type="text" name="payment_reference" id="payment_reference" required
                               placeholder="Cheque number, transaction ID, etc.">
                    </div>

                    <div class="form-group">
                        <label for="amount">Amount (GHS) *</label>
                        <input type="number" name="amount" id="amount" step="0.01" min="0.01" required>
                    </div>
                </div>

                <div class="form-group">
                    <label for="notes">Notes</label>
                    <textarea name="notes" id="notes" placeholder="Payment details..."></textarea>
                </div>

                <button type="submit" name="create_payment" class="btn btn-primary">Record Payment</button>
            </form>

            <div class="payable-section">
                <h3>Quick Select Payable Items</h3>
                <div id="payableItems">
                    <?php foreach ($payable_pos as $po): ?>
                        <div class="payable-item" onclick="selectPO(<?php echo $po['id']; ?>)">
                            <div>
                                <strong><?php echo htmlspecialchars($po['po_number']); ?> (PO)</strong><br>
                                <small><?php echo htmlspecialchars($po['supplier_name']); ?></small>
                            </div>
                            <div>GHS <?php echo number_format($po['total_amount'] - $po['total_paid'], 2); ?></div>
                            <div>Status: <?php echo ucfirst(str_replace('_', ' ', $po['status'])); ?></div>
                            <div>
                                <span class="payable-type">Purchase Order</span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    <?php foreach ($payable_invoices as $invoice): ?>
                        <div class="payable-item" onclick="selectInvoice(<?php echo $invoice['id']; ?>)">
                            <div>
                                <strong><?php echo htmlspecialchars($invoice['invoice_number']); ?> (Invoice)</strong><br>
                                <small><?php echo htmlspecialchars($invoice['supplier_name']); ?></small>
                            </div>
                            <div>GHS <?php echo number_format($invoice['remaining_amount'], 2); ?></div>
                            <div>Due: <?php echo date('M j, Y', strtotime($invoice['due_date'])); ?></div>
                            <div>
                                <?php if (strtotime($invoice['due_date']) < time()): ?>
                                    <span class="overdue">Overdue</span>
                                <?php else: ?>
                                    <span>Pending</span>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>

        <div class="payments-list-section">
            <h2 class="section-title">📋 Payment History</h2>
            <table class="payments-table">
                <thead>
                    <tr>
                        <th>Reference</th>
                        <th>Supplier</th>
                        <th>PO/Invoice</th>
                        <th>Date</th>
                        <th>Method</th>
                        <th>Amount</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($payments as $payment): ?>
                        <tr>
                            <td class="payment-reference"><?php echo htmlspecialchars($payment['payment_reference']); ?></td>
                            <td><?php echo htmlspecialchars($payment['supplier_name'] ?? 'N/A'); ?></td>
                            <td>
                                <?php if ($payment['po_number']): ?>
                                    PO: <?php echo htmlspecialchars($payment['po_number']); ?>
                                <?php endif; ?>
                                <?php if ($payment['invoice_number']): ?>
                                    <?php if ($payment['po_number']) echo '<br>'; ?>
                                    Inv: <?php echo htmlspecialchars($payment['invoice_number']); ?>
                                <?php endif; ?>
                                <?php if (!$payment['po_number'] && !$payment['invoice_number']): ?>
                                    N/A
                                <?php endif; ?>
                            </td>
                            <td><?php echo date('M j, Y', strtotime($payment['payment_date'])); ?></td>
                            <td><?php echo ucfirst(str_replace('_', ' ', $payment['payment_method'])); ?></td>
                            <td class="amount-negative">-GHS <?php echo number_format($payment['amount'], 2); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
function updatePayableOptions() {
    const supplierId = document.getElementById('supplier_id').value;

    // Filter PO options
    const poOptions = document.querySelectorAll('#po_id option');
    poOptions.forEach(option => {
        if (option.value === '' || option.getAttribute('data-supplier') === supplierId) {
            option.style.display = 'block';
        } else {
            option.style.display = 'none';
        }
    });

    // Filter invoice options
    const invoiceOptions = document.querySelectorAll('#invoice_id option');
    invoiceOptions.forEach(option => {
        if (option.value === '' || option.getAttribute('data-supplier') === supplierId) {
            option.style.display = 'block';
        } else {
            option.style.display = 'none';
        }
    });

    // Reset selections
    document.getElementById('po_id').value = '';
    document.getElementById('invoice_id').value = '';
    document.getElementById('amount').value = '';
}

function updateInvoiceOptions() {
    const poId = document.getElementById('po_id').value;
    const supplierId = document.getElementById('supplier_id').value;

    // Filter invoice options by PO if selected
    const invoiceOptions = document.querySelectorAll('#invoice_id option');
    invoiceOptions.forEach(option => {
        const optionSupplier = option.getAttribute('data-supplier');
        const optionPO = option.getAttribute('data-po');

        if (option.value === '') {
            option.style.display = 'block';
        } else if (supplierId && optionSupplier !== supplierId) {
            option.style.display = 'none';
        } else if (poId && optionPO && optionPO !== poId) {
            option.style.display = 'none';
        } else {
            option.style.display = 'block';
        }
    });

    // Reset invoice selection if PO changed
    document.getElementById('invoice_id').value = '';
    updatePaymentAmount();
}

function updatePaymentAmount() {
    const poSelect = document.getElementById('po_id');
    const invoiceSelect = document.getElementById('invoice_id');
    const amountInput = document.getElementById('amount');

    const selectedPO = poSelect.options[poSelect.selectedIndex];
    const selectedInvoice = invoiceSelect.options[invoiceSelect.selectedIndex];

    let amount = 0;

    if (selectedInvoice && selectedInvoice.value) {
        amount = parseFloat(selectedInvoice.getAttribute('data-remaining')) || 0;
    } else if (selectedPO && selectedPO.value) {
        amount = parseFloat(selectedPO.getAttribute('data-remaining')) || 0;
    }

    if (amount > 0) {
        amountInput.value = amount.toFixed(2);
    } else {
        amountInput.value = '';
    }
}

function selectInvoice(invoiceId) {
    document.getElementById('invoice_id').value = invoiceId;

    // Find the invoice option and get supplier
    const invoiceOption = document.querySelector(`#invoice_id option[value="${invoiceId}"]`);
    if (invoiceOption) {
        const supplierId = invoiceOption.getAttribute('data-supplier');
        document.getElementById('supplier_id').value = supplierId;
        updatePayableOptions();
        document.getElementById('invoice_id').value = invoiceId;
        updatePaymentAmount();
    }
}

function selectPO(poId) {
    document.getElementById('po_id').value = poId;

    // Find the PO option and get supplier
    const poOption = document.querySelector(`#po_id option[value="${poId}"]`);
    if (poOption) {
        const supplierId = poOption.getAttribute('data-supplier');
        document.getElementById('supplier_id').value = supplierId;
        updatePayableOptions();
        document.getElementById('po_id').value = poId;
        updatePaymentAmount();
    }
}

// Set today's date as default for payment date
document.addEventListener('DOMContentLoaded', function() {
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('payment_date').value = today;
});
</script>

<?php include '../includes/admin_footer.php'; ?>
