<?php

class PaymentMethods {
    // Account codes for different payment methods
    const CASH_ACCOUNT = '1001'; // Cash on Hand
    const BANK_ACCOUNT = '1120'; // Bank Account(s)
    const MOBILE_MONEY_ACCOUNT = '1124'; // Mobile Money Account

    // Payment method mappings
    private static $methodMappings = [
        'cash' => self::CASH_ACCOUNT,
        'cash_on_hand' => self::CASH_ACCOUNT,
        'petty_cash' => self::CASH_ACCOUNT,
        'bank_transfer' => self::BANK_ACCOUNT,
        'bank' => self::BANK_ACCOUNT,
        'cheque' => self::BANK_ACCOUNT,
        'check' => self::BANK_ACCOUNT,
        'bank_deposit' => self::BANK_ACCOUNT,
        'bank_withdrawal' => self::BANK_ACCOUNT,
        'mobile_money' => self::MOBILE_MONEY_ACCOUNT,
        'momo' => self::MOBILE_MONEY_ACCOUNT,
        'airtel_money' => self::MOBILE_MONEY_ACCOUNT,
        'mpesa' => self::MOBILE_MONEY_ACCOUNT,
        'orange_money' => self::MOBILE_MONEY_ACCOUNT,
        'zamtel_money' => self::MOBILE_MONEY_ACCOUNT,
    ];

    /**
     * Get the account code for a payment method
     */
    public static function getAccountCode($paymentMethod) {
        $method = str_replace(' ', '_', strtolower(trim($paymentMethod)));
        return isset(self::$methodMappings[$method]) ? self::$methodMappings[$method] : self::CASH_ACCOUNT;
    }

    /**
     * Get the account name for a payment method
     */
    public static function getAccountName($paymentMethod) {
        $accountCode = self::getAccountCode($paymentMethod);

        $accountNames = [
            self::CASH_ACCOUNT => 'Cash on Hand',
            self::BANK_ACCOUNT => 'Bank Account(s)',
            self::MOBILE_MONEY_ACCOUNT => 'Mobile Money Account',
        ];

        return isset($accountNames[$accountCode]) ? $accountNames[$accountCode] : 'Cash on Hand';
    }

    /**
     * Get all payment method mappings
     */
    public static function getAllMappings() {
        return self::$methodMappings;
    }

    /**
     * Get form options for payment methods
     */
    public static function getFormOptions($exclude = []) {
        $options = [
            ['value' => 'cash', 'label' => 'Cash'],
            ['value' => 'bank_transfer', 'label' => 'Bank Transfer'],
            ['value' => 'cheque', 'label' => 'Cheque'],
            ['value' => 'mobile_money', 'label' => 'Mobile Money'],
        ];

        // Filter out excluded payment methods
        return array_filter($options, function($option) use ($exclude) {
            return !in_array($option['value'], $exclude);
        });
    }

    /**
     * Validate payment method
     */
    public static function isValidMethod($paymentMethod) {
        $method = strtolower(trim($paymentMethod));
        return isset(self::$methodMappings[$method]);
    }

    /**
     * Get payment methods by account type
     */
    public static function getMethodsByAccount($accountCode) {
        $methods = [];
        foreach (self::$methodMappings as $method => $code) {
            if ($code === $accountCode) {
                $methods[] = $method;
            }
        }
        return $methods;
    }

    /**
     * Get cash account name for payment method (used in accounting functions)
     */
    public static function getCashAccountForPaymentMethod($paymentMethod) {
        return self::getAccountName($paymentMethod);
    }

    /**
     * Get the account ID for a payment method from the database
     */
    public static function getAccountId($paymentMethod) {
        global $pdo;

        $accountCode = self::getAccountCode($paymentMethod);

        try {
            $stmt = $pdo->prepare("SELECT id FROM chart_of_accounts WHERE account_code = ? AND account_type = 'Asset' AND is_active = TRUE LIMIT 1");
            $stmt->execute([$accountCode]);
            $result = $stmt->fetch();

            return $result ? $result['id'] : null;
        } catch (PDOException $e) {
            return null;
        }
    }
}

/**
 * Get fee payment cash account name (standalone function for backward compatibility)
 */
function getFeePaymentCashAccount($paymentMethod) {
    return PaymentMethods::getAccountName($paymentMethod);
}
?>
