<?php
require_once '../config.php';
require_once '../includes/access_control.php';
require_once 'accounting_functions.php';

ob_start();

if (false) { // Temporarily disable feature access check to remove subscription upgrade message
    requireFeatureAccess('accounts_management');
} else {
    // Do nothing, allow access temporarily
}

$message = '';
$error = '';

// Handle success message from redirect
if (isset($_GET['success'])) {
    $message = "Salary payment processed successfully!";
}

// Handle payment actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'pay_individual') {
        $payslip_id = $_POST['payslip_id'] ?? 0;
        $payment_method = $_POST['payment_method'] ?? 'cash';
        $comments = $_POST['comments'] ?? '';

        if ($payslip_id) {
            try {
                $pdo->beginTransaction();

                // Get payslip details
                $stmt = $pdo->prepare("
                    SELECT p.*, pr.payroll_month as payroll_run_date, pr.id as payroll_run_id,
                           CONCAT(p.first_name, ' ', p.last_name) as employee_name,
                           p.net_salary as amount, pr.payroll_year,
                           s.staff_id as staff_id_display
                    FROM payslips p
                    JOIN payroll_runs pr ON p.payroll_run_id = pr.id
                    LEFT JOIN staff s ON p.staff_id = s.id
                    WHERE p.id = ? AND p.payment_status = 'pending'
                ");
                $stmt->execute([$payslip_id]);
                $payslip = $stmt->fetch();

                if ($payslip) {
                    // Update payslip payment status
                    $stmt = $pdo->prepare("
                        UPDATE payslips
                        SET payment_status = 'paid', paid_at = NOW(), paid_by = ?, payment_method = ?, payment_comments = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([$_SESSION['user_id'], $payment_method, $comments, $payslip_id]);

                    // Create payroll payment journal entry
                    createPayrollPaymentEntry($pdo, $payslip['payroll_run_id'], 0, 0, $payslip['amount'], $payment_method, date('Y-m-d'), $_SESSION['user_id'], false);
                }

                $pdo->commit();

                // Store receipt data in session for printing
                $_SESSION['salary_receipt_data'] = [
                    'receipt_number' => 'SAL-' . str_pad($payslip_id, 6, '0', STR_PAD_LEFT),
                    'employee_name' => $payslip['employee_name'],
                    'payroll_run' => 'Run #' . $payslip['payroll_run_id'],
                    'payroll_date' => $payslip['payroll_run_date'],
                    'amount' => $payslip['amount'],
                    'payment_date' => date('Y-m-d H:i:s'),
                    'payment_method' => $payment_method,
                    'comments' => $comments,
                    'processed_by' => $_SESSION['username'] ?? 'Unknown',
                    'status' => 'Paid'
                ];

                $message = "Salary payment processed successfully! <a href='print_salary_receipt.php' target='_blank' class='btn btn-primary btn-sm' style='margin-left: 10px;'>🖨️ Print Receipt</a>";
                header("Location: pay_salaries.php?success=1");
                exit;
            } catch (PDOException $e) {
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $error = "Error processing payment: " . $e->getMessage();
            }
        }
    } elseif ($action === 'pay_bulk') {
        $payslip_ids_str = $_POST['payslip_ids'] ?? '';
        $payslip_ids = !empty($payslip_ids_str) ? explode(',', $payslip_ids_str) : [];
        $payment_method = $_POST['payment_method'] ?? 'cash';
        $comments = $_POST['comments'] ?? '';

        if (!empty($payslip_ids)) {
            try {
                $pdo->beginTransaction();

                $total_amount = 0;
                $payroll_run_id = null;

                // Get payslip details and calculate total
                foreach ($payslip_ids as $payslip_id) {
                    $stmt = $pdo->prepare("
                        SELECT p.net_salary, p.payroll_run_id
                        FROM payslips p
                        WHERE p.id = ? AND p.payment_status = 'pending'
                    ");
                    $stmt->execute([$payslip_id]);
                    $payslip = $stmt->fetch();

                    if ($payslip) {
                        $total_amount += $payslip['net_salary'];
                        if (!$payroll_run_id) {
                            $payroll_run_id = $payslip['payroll_run_id'];
                        }
                    }
                }

                if ($total_amount > 0) {
                    // Update payslip payment statuses
                    $placeholders = str_repeat('?,', count($payslip_ids) - 1) . '?';
                    $stmt = $pdo->prepare("
                        UPDATE payslips
                        SET payment_status = 'paid', paid_at = NOW(), paid_by = ?, payment_method = ?, payment_comments = ?
                        WHERE id IN ($placeholders) AND payment_status = 'pending'
                    ");
                    $params = array_merge([$_SESSION['user_id'], $payment_method, $comments], $payslip_ids);
                    $stmt->execute($params);

                    // Create individual payroll payment journal entries for each payslip
                    foreach ($payslip_ids as $payslip_id) {
                        $stmt = $pdo->prepare("
                            SELECT pr.id as payroll_run_id, p.net_salary
                            FROM payslips p
                            JOIN payroll_runs pr ON p.payroll_run_id = pr.id
                            WHERE p.id = ?
                        ");
                        $stmt->execute([$payslip_id]);
                        $payslip_data = $stmt->fetch();

                        if ($payslip_data) {
                            createPayrollPaymentEntry($pdo, $payslip_data['payroll_run_id'], 0, 0, $payslip_data['net_salary'], $payment_method, date('Y-m-d'), $_SESSION['user_id'], false);
                        }
                    }
                }

                $pdo->commit();
                $message = "Bulk salary payments processed successfully!";
                header("Location: pay_salaries.php?success=1");
                exit;
            } catch (PDOException $e) {
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $error = "Error processing bulk payments: " . $e->getMessage();
            }
        }
    }
}

include 'includes/accounting_header.php';

// Get filter parameters
$status_filter = $_GET['status'] ?? 'pending';
$search = $_GET['search'] ?? '';
$payroll_run_filter = $_GET['payroll_run'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$sort_by = $_GET['sort_by'] ?? 'pr.payroll_month';
$sort_order = $_GET['sort_order'] ?? 'DESC';

// Validate sort_by to prevent SQL errors
$allowed_sort_columns = ['p.id', 'pr.payroll_month', 'p.net_salary', 'p.payment_status', 'p.paid_at'];
if (!in_array($sort_by, $allowed_sort_columns)) {
    $sort_by = 'pr.payroll_month';
}

// Build query
$where_conditions = [];
$params = [];

if (!empty($status_filter)) {
    $where_conditions[] = "p.payment_status = ?";
    $params[] = $status_filter;
}

if (!empty($search)) {
    $where_conditions[] = "(s.first_name LIKE ? OR s.last_name LIKE ? OR CONCAT(s.first_name, ' ', s.last_name) LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if (!empty($payroll_run_filter)) {
    $where_conditions[] = "p.payroll_run_id = ?";
    $params[] = $payroll_run_filter;
}

if (!empty($date_from)) {
    $where_conditions[] = "pr.payroll_month >= ?";
    $params[] = $date_from;
}

if (!empty($date_to)) {
    $where_conditions[] = "pr.payroll_month <= ?";
    $params[] = $date_to;
}

$where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get total count for pagination
try {
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total
        FROM payslips p
        JOIN payroll_runs pr ON p.payroll_run_id = pr.id
        LEFT JOIN staff s ON p.staff_id = s.id
        $where_clause
    ");
    $stmt->execute($params);
    $total_payslips = $stmt->fetch()['total'];
    $total_pages = ceil($total_payslips / 20);
    $current_page = max(1, min($total_pages, $_GET['page'] ?? 1));
    $offset = ($current_page - 1) * 20;

    // Get payslips with pagination
    $stmt = $pdo->prepare("
        SELECT p.*, pr.payroll_month as payroll_run_date, pr.id as payroll_run_id,
               CONCAT(p.first_name, ' ', p.last_name) as employee_name,
               p.net_salary as amount, p.payment_status, p.paid_at, p.payment_comments,
               u.username as paid_by_name,
               s.staff_id as staff_id_display
        FROM payslips p
        JOIN payroll_runs pr ON p.payroll_run_id = pr.id
        LEFT JOIN staff s ON p.staff_id = s.id
        LEFT JOIN users u ON p.paid_by = u.id
        $where_clause
        ORDER BY $sort_by $sort_order
        LIMIT ? OFFSET ?
    ");
    $i = 1;
    foreach ($params as $param) {
        $stmt->bindValue($i++, $param);
    }
    $stmt->bindValue($i++, 20, PDO::PARAM_INT);
    $stmt->bindValue($i++, $offset, PDO::PARAM_INT);
    $stmt->execute();
    $payslips = $stmt->fetchAll();

    // Get status summary
    $stmt = $pdo->prepare("
        SELECT payment_status, COUNT(*) as count, COALESCE(SUM(net_salary), 0) as total_amount
        FROM payslips
        GROUP BY payment_status
    ");
    $stmt->execute();
    $status_summary_raw = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Convert to the expected format
    $status_summary = [];
    foreach ($status_summary_raw as $row) {
        $status_summary[$row['payment_status']] = [
            'count' => $row['count'],
            'total_amount' => $row['total_amount']
        ];
    }

    // Get payroll runs for filter
    $stmt = $pdo->query("SELECT id, payroll_month as payroll_run_date FROM payroll_runs ORDER BY payroll_month DESC LIMIT 50");
    $payroll_runs = $stmt->fetchAll();

} catch (PDOException $e) {
    $error = "Error loading payslips: " . $e->getMessage();
    $payslips = [];
    $status_summary = [];
    $payroll_runs = [];
}
?>

<style>
.pay-salaries-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 0;
    background: transparent;
    min-height: auto;
    box-shadow: none;
}

.salary-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 30px;
    border-radius: 12px;
    margin-bottom: 30px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.salary-header h1 {
    margin: 0;
    font-size: 2.2em;
    font-weight: 300;
}

.salary-header p {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.salary-content {
    display: grid;
    grid-template-columns: 1fr 320px;
    gap: 30px;
}

.salary-main {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    border: 1px solid #e1e5e9;
}

.filters-section {
    background: #f8f9fa;
    padding: 25px;
    border-bottom: 1px solid #dee2e6;
}

.filters-form {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 20px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 8px;
    color: #495057;
    font-size: 0.95em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.filter-group input,
.filter-group select {
    padding: 12px 16px;
    border: 2px solid #2E7D32;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.3s ease;
}

.filter-group input:focus,
.filter-group select:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.filter-actions {
    display: flex;
    gap: 12px;
}

.btn {
    padding: 12px 24px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(46, 125, 50, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(46, 125, 50, 0.4);
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
}

.btn-success {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
}

.btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(40, 167, 69, 0.4);
}

.btn-danger {
    background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(220, 53, 69, 0.3);
}

.btn-danger:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(220, 53, 69, 0.4);
}

.btn-warning {
    background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%);
    color: #212529;
    box-shadow: 0 4px 15px rgba(255, 193, 7, 0.3);
}

.btn-warning:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(255, 193, 7, 0.4);
}

.payslips-table {
    width: 100%;
    border-collapse: collapse;
}

.payslips-table th,
.payslips-table td {
    padding: 16px 20px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.payslips-table th {
    background: #fff;
    font-weight: 700;
    color: #000;
    position: sticky;
    top: 0;
    text-transform: uppercase;
    font-size: 0.85em;
    letter-spacing: 0.5px;
}

.payslips-table tr:hover {
    background: #f8f9fa;
}

.employee-name {
    font-weight: 600;
    color: #2c3e50;
    max-width: 200px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.salary-amount {
    font-weight: 700;
    color: #e74c3c;
    font-size: 1.1em;
}

.payroll-run {
    color: #6c757d;
    font-size: 0.9em;
    font-weight: 500;
}

.payroll-date {
    color: #6c757d;
    font-size: 0.9em;
    font-weight: 500;
}

.status-badge {
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 0.8em;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    display: inline-block;
}

.status-pending {
    background: #fff3cd;
    color: #856404;
    border: 1px solid #ffeaa7;
}

.status-paid {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.actions-cell {
    white-space: nowrap;
}

.action-btn {
    padding: 6px 12px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: 0.85em;
    margin-right: 8px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    transition: all 0.3s ease;
}

.pay-btn {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
}

.pay-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 10px rgba(40, 167, 69, 0.3);
}

.bulk-actions {
    background: #f8f9fa;
    padding: 20px;
    border-top: 1px solid #dee2e6;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.bulk-actions select {
    padding: 8px 12px;
    border: 2px solid #dee2e6;
    border-radius: 6px;
    font-size: 14px;
}

.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-bottom: 25px;
}

.summary-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    padding: 20px;
    border-radius: 12px;
    text-align: center;
    border: 1px solid #e1e5e9;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    transition: transform 0.3s ease;
}

.summary-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
}

.summary-card h3 {
    margin: 0 0 8px 0;
    font-size: 2em;
    color: #2c3e50;
    font-weight: 700;
}

.summary-card p {
    margin: 0;
    color: #6c757d;
    font-size: 0.95em;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.pagination {
    display: flex;
    justify-content: center;
    gap: 8px;
    margin-top: 25px;
    padding-top: 25px;
    border-top: 1px solid #dee2e6;
}

.pagination a,
.pagination span {
    padding: 10px 16px;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    text-decoration: none;
    color: #2E7D32;
    font-weight: 600;
    transition: all 0.3s ease;
}

.pagination .active {
    background: #2E7D32;
    color: white;
    border-color: #2E7D32;
}

.pagination .disabled {
    color: #6c757d;
    cursor: not-allowed;
}

.message {
    padding: 15px;
    margin-bottom: 20px;
    border-radius: 8px;
    font-weight: 600;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.no-payslips {
    text-align: center;
    padding: 50px 20px;
    color: #6c757d;
}

.no-payslips h3 {
    margin: 0 0 10px 0;
    color: #495057;
}

.salary-sidebar {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    padding: 25px;
    border: 1px solid #e1e5e9;
}

.salary-sidebar h3 {
    margin-top: 0;
    color: #2c3e50;
    margin-bottom: 20px;
}

.quick-actions {
    display: flex;
    flex-direction: column;
    gap: 10px;
}

.quick-actions .btn {
    text-align: center;
    padding: 12px;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
}

.modal-content {
    background-color: #fefefe;
    margin: 10% auto;
    padding: 0;
    border: 1px solid #888;
    width: 90%;
    max-width: 500px;
    border-radius: 12px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.15);
}

.modal-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 20px;
    border-radius: 12px 12px 0 0;
}

.modal-header h2 {
    margin: 0;
    font-size: 1.5em;
    font-weight: 300;
}

.modal-body {
    padding: 20px;
}

.modal-footer {
    padding: 20px;
    background-color: #f8f9fa;
    border-radius: 0 0 12px 12px;
    display: flex;
    justify-content: flex-end;
    gap: 10px;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #495057;
}

.form-group textarea {
    width: 100%;
    padding: 12px;
    border: 2px solid #dee2e6;
    border-radius: 8px;
    font-size: 14px;
    resize: vertical;
    min-height: 80px;
}

.form-group textarea:focus {
    outline: none;
    border-color: #2E7D32;
    box-shadow: 0 0 0 3px rgba(46, 125, 50, 0.1);
}

.form-group select {
    width: 100%;
    padding: 12px;
    border: 2px solid #dee2e6;
    border-radius: 8px;
    font-size: 14px;
}

.form-group select:focus {
    outline: none;
    border-color: #2E7D32;
    box-shadow: 0 0 0 3px rgba(46, 125, 50, 0.1);
}

.payslip-details {
    background: #f8f9fa;
    padding: 15px;
    border-radius: 8px;
    margin-bottom: 20px;
}

.payslip-details h4 {
    margin: 0 0 10px 0;
    color: #2c3e50;
    font-size: 1.1em;
}

.payslip-details p {
    margin: 5px 0;
    font-size: 0.9em;
    color: #495057;
}

.payslip-details .amount {
    font-weight: 700;
    color: #e74c3c;
    font-size: 1.1em;
}

.checkbox-container {
    display: flex;
    align-items: center;
    gap: 10px;
    margin-bottom: 10px;
}

.checkbox-container input[type="checkbox"] {
    width: 18px;
    height: 18px;
}

.checkbox-container label {
    margin: 0;
    font-weight: 500;
    cursor: pointer;
}
</style>

<div class="pay-salaries-container">
    <div class="salary-header">
        <div>
            <h1>💰 Pay Salaries</h1>
            <p>Process salary payments for employees</p>
        </div>
        <div>
            <a href="process_payroll.php" class="btn btn-success">📊 Process Payroll</a>
            <a href="view_payslips.php" class="btn btn-primary">📄 View Payslips</a>
            <a href="accounting_dashboard.php" class="btn btn-secondary">🏠 Dashboard</a>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="salary-content">
        <div class="salary-main">
            <!-- Summary Cards -->
            <div class="summary-cards">
                <div class="summary-card">
                    <h3><?php echo $status_summary['pending']['count'] ?? 0; ?></h3>
                    <p>Pending Payment</p>
                </div>
                <div class="summary-card">
                    <h3><?php echo $status_summary['paid']['count'] ?? 0; ?></h3>
                    <p>Paid</p>
                </div>
            </div>

            <!-- Filters -->
            <div class="filters-section">
                <form method="get" class="filters-form">
                    <div class="filter-group">
                        <label for="status">Status</label>
                        <select name="status" id="status">
                            <option value="">All Statuses</option>
                            <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="paid" <?php echo $status_filter === 'paid' ? 'selected' : ''; ?>>Paid</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="search">Search Employee</label>
                        <input type="text" name="search" id="search" value="<?php echo htmlspecialchars($search); ?>"
                               placeholder="Employee name...">
                    </div>

                    <div class="filter-group">
                        <label for="payroll_run">Payroll Run</label>
                        <select name="payroll_run" id="payroll_run">
                            <option value="">All Payroll Runs</option>
                            <?php foreach ($payroll_runs as $run): ?>
                                <option value="<?php echo $run['id']; ?>"
                                        <?php echo $payroll_run_filter == $run['id'] ? 'selected' : ''; ?>>
                                    <?php echo date('M d, Y', strtotime($run['payroll_run_date'])); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="date_from">From Date</label>
                        <input type="date" name="date_from" id="date_from" value="<?php echo htmlspecialchars($date_from); ?>">
                    </div>

                    <div class="filter-group">
                        <label for="date_to">To Date</label>
                        <input type="date" name="date_to" id="date_to" value="<?php echo htmlspecialchars($date_to); ?>">
                    </div>

                    <div class="filter-actions">
                        <button type="submit" class="btn btn-primary">🔍 Filter</button>
                        <a href="pay_salaries.php" class="btn btn-secondary">🔄 Reset</a>
                    </div>
                </form>
            </div>

            <!-- Bulk Actions -->
            <div class="bulk-actions">
                <div>
                    <select id="bulk-action">
                        <option value="">Bulk Actions</option>
                        <option value="pay-selected">Pay Selected</option>
                    </select>
                    <button class="btn btn-warning" onclick="executeBulkAction()">Execute</button>
                </div>
                <div>
                    <span id="selected-count">0 selected</span>
                </div>
            </div>

            <!-- Payslips Table -->
            <?php if (empty($payslips)): ?>
                <div class="no-payslips">
                    <h3>No payslips found</h3>
                    <p>Try adjusting your filters or <a href="process_payroll.php">process a new payroll run</a>.</p>
                </div>
            <?php else: ?>
                <div class="table-container">
                    <table class="payslips-table">
                        <thead>
                            <tr>
                                <th><input type="checkbox" id="select-all"></th>
                                <th>Employee</th>
                                <th>Amount</th>
                                <th>Payroll Run</th>
                                <th>Run Date</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($payslips as $payslip): ?>
                                <tr>
                                    <td>
                                        <?php if ($payslip['payment_status'] === 'pending'): ?>
                                            <input type="checkbox" class="payslip-checkbox" value="<?php echo $payslip['id']; ?>">
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="employee-name" title="<?php echo htmlspecialchars($payslip['employee_name']); ?>">
                                            <?php echo htmlspecialchars($payslip['employee_name']); ?>
                                        </div>
                                    </td>
                                    <td class="salary-amount">
                                        GHC <?php echo number_format($payslip['amount'], 2); ?>
                                    </td>
                                    <td class="payroll-run">
                                        Run #<?php echo $payslip['payroll_run_id']; ?>
                                    </td>
                                    <td class="payroll-date">
                                        <?php echo date('M d, Y', strtotime($payslip['payroll_run_date'])); ?>
                                    </td>
                                    <td>
                                        <span class="status-badge status-<?php echo $payslip['payment_status']; ?>">
                                            <?php echo ucfirst($payslip['payment_status']); ?>
                                        </span>
                                        <?php if ($payslip['paid_by_name']): ?>
                                            <div style="font-size: 0.7em; color: #6c757d; margin-top: 2px;">
                                                by <?php echo htmlspecialchars($payslip['paid_by_name']); ?>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td class="actions-cell">
                                        <?php if ($payslip['payment_status'] === 'pending'): ?>
                                            <button class="action-btn pay-btn" onclick="openPaymentModal(<?php echo $payslip['id']; ?>, '<?php echo htmlspecialchars($payslip['employee_name']); ?>', <?php echo $payslip['amount']; ?>)">
                                                💰 Pay
                                            </button>
                                        <?php else: ?>
                                            <button class="action-btn" style="background: #6c757d; cursor: not-allowed;" disabled>
                                                <?php echo ucfirst($payslip['payment_status']); ?>
                                            </button>
                                            <a href="print_salary_receipt.php?payslip=<?php echo $payslip['id']; ?>" target="_blank" class="action-btn" style="background: linear-gradient(135deg, #17a2b8 0%, #138496 100%); color: white; text-decoration: none;">
                                                🖨️ Receipt
                                            </a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <div class="pagination">
                        <?php if ($current_page > 1): ?>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $current_page - 1])); ?>">Previous</a>
                        <?php else: ?>
                            <span class="disabled">Previous</span>
                        <?php endif; ?>

                        <?php for ($i = max(1, $current_page - 2); $i <= min($total_pages, $current_page + 2); $i++): ?>
                            <?php if ($i == $current_page): ?>
                                <span class="active"><?php echo $i; ?></span>
                            <?php else: ?>
                                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                            <?php endif; ?>
                        <?php endfor; ?>

                        <?php if ($current_page < $total_pages): ?>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $current_page + 1])); ?>">Next</a>
                        <?php else: ?>
                            <span class="disabled">Next</span>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>

        <div class="salary-sidebar">
            <h3>📊 Payment Summary</h3>
            <?php if (empty($status_summary)): ?>
                <p style="color: #7f8c8d; font-style: italic;">No payslips to summarize</p>
            <?php else: ?>
                <div style="max-height: 300px; overflow-y: auto;">
                    <?php
                    $status_labels = [
                        'pending' => 'Pending Payment',
                        'paid' => 'Paid'
                    ];
                    $status_colors = [
                        'pending' => '#ffc107',
                        'paid' => '#28a745'
                    ];
                    foreach ($status_summary as $status => $data) {
                        if ($data['count'] > 0) {
                            $color = $status_colors[$status] ?? '#6c757d';
                            echo "<div style='display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #f0f0f0;'>";
                            echo "<div>";
                            echo "<strong style='color: {$color}'>{$status_labels[$status]}</strong>";
                            echo "<div style='font-size: 0.8em; color: #7f8c8d;'>{$data['count']} payslips</div>";
                            echo "</div>";
                            echo "<div style='text-align: right;'>";
                            echo "<div style='font-weight: 600; color: #e74c3c;'>GHC " . number_format($data['total_amount'], 2) . "</div>";
                            echo "</div>";
                            echo "</div>";
                        }
                    }
                    ?>
                </div>
            <?php endif; ?>

            <h3 style="margin-top: 30px;">🔗 Quick Actions</h3>
            <div class="quick-actions">
                <a href="process_payroll.php" class="btn btn-success">Process New Payroll</a>
                <a href="view_payslips.php" class="btn btn-primary">View All Payslips</a>
                <a href="payroll_management.php" class="btn btn-primary">Payroll Dashboard</a>
                <a href="balance_sheet_report.php" class="btn btn-primary">View Balance Sheet</a>
            </div>
        </div>
    </div>
</div>

<!-- Payment Modal -->
<div id="paymentModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2>💰 Pay Salary</h2>
        </div>
        <div class="modal-body">
            <div class="payslip-details">
                <h4 id="employeeName"></h4>
                <p><strong>Amount:</strong> <span id="salaryAmount" class="amount"></span></p>
                <p><strong>Action:</strong> Salary Payment</p>
            </div>

            <form id="paymentForm" method="post">
                <input type="hidden" name="action" value="pay_individual">
                <input type="hidden" name="payslip_id" id="modalPayslipId">

                <div class="form-group">
                    <label for="payment_method">Payment Method</label>
                    <select name="payment_method" id="payment_method" required>
                        <?php
                        require_once 'payment_methods.php';
                        $paymentOptions = PaymentMethods::getFormOptions();
                        foreach ($paymentOptions as $option) {
                            echo "<option value=\"{$option['value']}\">{$option['label']}</option>";
                        }
                        ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="comments">Comments (Optional)</label>
                    <textarea name="comments" id="comments" placeholder="Add any payment notes or reference numbers..."></textarea>
                </div>
            </form>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
            <button type="button" class="btn btn-success" onclick="submitPayment()">Pay Salary</button>
        </div>
    </div>
</div>

<!-- Bulk Payment Modal -->
<div id="bulkPaymentModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2>💰 Bulk Salary Payment</h2>
        </div>
        <div class="modal-body">
            <div class="payslip-details">
                <h4>Bulk Payment</h4>
                <p><strong>Selected Payslips:</strong> <span id="bulkSelectedCount">0</span></p>
                <p><strong>Total Amount:</strong> <span id="bulkTotalAmount" class="amount">GHC 0.00</span></p>
            </div>

            <form id="bulkPaymentForm" method="post">
                <input type="hidden" name="action" value="pay_bulk">
                <input type="hidden" name="payslip_ids" id="bulkPayslipIds">

                <div class="form-group">
                    <label for="bulk_payment_method">Payment Method</label>
                    <select name="payment_method" id="bulk_payment_method" required>
                        <?php
                        require_once 'payment_methods.php';
                        $paymentOptions = PaymentMethods::getFormOptions();
                        foreach ($paymentOptions as $option) {
                            echo "<option value=\"{$option['value']}\">{$option['label']}</option>";
                        }
                        ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="bulk_comments">Comments (Optional)</label>
                    <textarea name="comments" id="bulk_comments" placeholder="Add any payment notes or reference numbers..."></textarea>
                </div>
            </form>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
            <button type="button" class="btn btn-success" onclick="submitBulkPayment()">Pay Salaries</button>
        </div>
    </div>
</div>

<script>
// Select All functionality
document.getElementById('select-all').addEventListener('change', function() {
    const checkboxes = document.querySelectorAll('.payslip-checkbox');
    checkboxes.forEach(checkbox => {
        checkbox.checked = this.checked;
    });
    updateSelectedCount();
});

// Update selected count when individual checkboxes change
document.addEventListener('change', function(e) {
    if (e.target.classList.contains('payslip-checkbox')) {
        updateSelectedCount();
    }
});

function updateSelectedCount() {
    const selected = document.querySelectorAll('.payslip-checkbox:checked').length;
    document.getElementById('selected-count').textContent = selected + ' selected';
}

// Open payment modal for individual payment
function openPaymentModal(payslipId, employeeName, amount) {
    document.getElementById('employeeName').textContent = employeeName;
    document.getElementById('salaryAmount').textContent = 'GHC ' + parseFloat(amount).toFixed(2);
    document.getElementById('modalPayslipId').value = payslipId;
    document.getElementById('paymentModal').style.display = 'block';
}

// Close modal
function closeModal() {
    document.getElementById('paymentModal').style.display = 'none';
    document.getElementById('bulkPaymentModal').style.display = 'none';
}

// Submit individual payment
function submitPayment() {
    document.getElementById('paymentForm').submit();
}

// Execute bulk action
function executeBulkAction() {
    const action = document.getElementById('bulk-action').value;
    if (action === 'pay-selected') {
        const selectedCheckboxes = document.querySelectorAll('.payslip-checkbox:checked');
        if (selectedCheckboxes.length === 0) {
            alert('Please select at least one payslip to pay.');
            return;
        }

        // Collect selected payslip IDs and calculate total
        const payslipIds = [];
        let totalAmount = 0;
        selectedCheckboxes.forEach(checkbox => {
            payslipIds.push(checkbox.value);
            // Get amount from the row (assuming it's in the 3rd column)
            const row = checkbox.closest('tr');
            const amountText = row.cells[2].textContent.replace('GHC ', '').replace(',', '');
            totalAmount += parseFloat(amountText);
        });

        // Populate bulk modal
        document.getElementById('bulkSelectedCount').textContent = selectedCheckboxes.length;
        document.getElementById('bulkTotalAmount').textContent = 'GHC ' + totalAmount.toFixed(2);
        document.getElementById('bulkPayslipIds').value = payslipIds.join(',');

        // Open bulk payment modal
        document.getElementById('bulkPaymentModal').style.display = 'block';
    }
}

// Submit bulk payment
function submitBulkPayment() {
    document.getElementById('bulkPaymentForm').submit();
}

// Close modal when clicking outside
window.onclick = function(event) {
    const paymentModal = document.getElementById('paymentModal');
    const bulkModal = document.getElementById('bulkPaymentModal');
    if (event.target === paymentModal) {
        paymentModal.style.display = 'none';
    }
    if (event.target === bulkModal) {
        bulkModal.style.display = 'none';
    }
}

// Initialize selected count on page load
document.addEventListener('DOMContentLoaded', function() {
    updateSelectedCount();
});
</script>
