<?php
require_once '../config.php';

$message = '';
$message_type = '';

// Get selected tax year, default to current year
$selected_year = isset($_GET['year']) ? (int)$_GET['year'] : date('Y');

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['add_tax_rate'])) {
        $tax_year = $_POST['tax_year'];
        $min_income = $_POST['min_income'];
        $max_income = $_POST['max_income'] ?: null;
        $tax_rate = $_POST['tax_rate'];
        $cumulative_tax = isset($_POST['cumulative_tax']) && $_POST['cumulative_tax'] !== '' ? floatval($_POST['cumulative_tax']) : 0;

        $stmt = $pdo->prepare("
            INSERT INTO tax_rates (tax_year, min_income, max_income, tax_rate, cumulative_tax, is_active)
            VALUES (?, ?, ?, ?, ?, TRUE)
        ");

        if ($stmt->execute([$tax_year, $min_income, $max_income, $tax_rate, $cumulative_tax])) {
            $message = 'Tax rate bracket added successfully.';
            $message_type = 'success';
            $selected_year = $tax_year;
        } else {
            $message = 'Failed to add tax rate bracket.';
            $message_type = 'error';
        }
    } elseif (isset($_POST['edit_tax_rate'])) {
        $rate_id = $_POST['rate_id'];
        $tax_year = $_POST['tax_year'];
        $min_income = $_POST['min_income'];
        $max_income = $_POST['max_income'] ?: null;
        $tax_rate = $_POST['tax_rate'];
        $cumulative_tax = isset($_POST['cumulative_tax']) && $_POST['cumulative_tax'] !== '' ? floatval($_POST['cumulative_tax']) : 0;

        $stmt = $pdo->prepare("
            UPDATE tax_rates
            SET tax_year = ?, min_income = ?, max_income = ?, tax_rate = ?, cumulative_tax = ?
            WHERE id = ?
        ");

        if ($stmt->execute([$tax_year, $min_income, $max_income, $tax_rate, $cumulative_tax, $rate_id])) {
            $message = 'Tax rate bracket updated successfully.';
            $message_type = 'success';
            $selected_year = $tax_year;
        } else {
            $message = 'Failed to update tax rate bracket.';
            $message_type = 'error';
        }
    }
}

// Fetch tax rates for selected year
$stmt = $pdo->prepare("SELECT * FROM tax_rates WHERE tax_year = ? AND is_active = TRUE ORDER BY min_income");
$stmt->execute([$selected_year]);
$tax_rates = $stmt->fetchAll();

// Get available tax years
$stmt = $pdo->query("SELECT DISTINCT tax_year FROM tax_rates ORDER BY tax_year DESC");
$available_years = $stmt->fetchAll(PDO::FETCH_COLUMN);

// Get tax rate for edit
$edit_rate = null;
if (isset($_GET['edit'])) {
    $rate_id = $_GET['edit'];
    $stmt = $pdo->prepare("SELECT * FROM tax_rates WHERE id = ?");
    $stmt->execute([$rate_id]);
    $edit_rate = $stmt->fetch();
    if ($edit_rate) {
        $selected_year = $edit_rate['tax_year'];
    }
}

include 'includes/accounting_header.php';
?>

<style>
.tax-rates-table {
    background: white;
    border-radius: 8px;
    overflow: hidden;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    margin-bottom: 20px;
}

.tax-rates-table table {
    width: 100%;
    border-collapse: collapse;
}

.tax-rates-table th,
.tax-rates-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.tax-rates-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.tax-rates-table tr:hover {
    background: #f8f9fa;
}

.form-row {
    display: flex;
    gap: 20px;
    margin-bottom: 15px;
}

.form-row .form-group {
    flex: 1;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #333;
}

.form-group input,
.form-group select,
.form-group textarea {
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
}

.year-selector {
    background: #f8f9fa;
    padding: 15px;
    border-radius: 8px;
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 15px;
}

.year-selector label {
    font-weight: 600;
    margin: 0;
}

.year-selector select {
    padding: 8px 12px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
}

.no-rates {
    text-align: center;
    padding: 40px;
    color: #7f8c8d;
    background: #f8f9fa;
    border-radius: 8px;
    margin: 20px 0;
}

/* Button styling to match accounting sidebar theme */
.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    border-color: #2E7D32;
    color: white;
    padding: 10px 20px;
    border-radius: 4px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #4CAF50 0%, #66BB6A 100%);
    border-color: #4CAF50;
    color: white;
    transform: translateY(-1px);
    box-shadow: 0 2px 8px rgba(46, 125, 50, 0.3);
}

.btn {
    display: inline-block;
    padding: 8px 16px;
    border-radius: 4px;
    text-decoration: none;
    font-weight: 500;
    border: 1px solid transparent;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-small {
    padding: 6px 12px;
    font-size: 0.9em;
}

.btn:hover {
    transform: translateY(-1px);
}

/* Modal styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    overflow: auto;
    background-color: rgba(0,0,0,0.4);
}

.modal-content {
    background-color: #fefefe;
    margin: 5% auto;
    padding: 20px;
    border: 1px solid #888;
    width: 90%;
    max-width: 600px;
    border-radius: 8px;
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.close {
    color: #aaa;
    float: right;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
}

.close:hover,
.close:focus {
    color: black;
    text-decoration: none;
}

.modal-header {
    padding-bottom: 10px;
    border-bottom: 1px solid #eee;
    margin-bottom: 20px;
}

.modal-body {
    padding: 10px 0;
}

.modal-footer {
    padding-top: 10px;
    border-top: 1px solid #eee;
    text-align: right;
}
</style>

<h2>💰 Manage Tax Rates</h2>

<?php if ($message): ?>
    <div class="message <?php echo $message_type; ?>"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>

<!-- Tax Year Selector -->
<div class="year-selector">
    <label for="tax_year_select">Select Tax Year:</label>
    <select id="tax_year_select" onchange="changeYear(this.value)">
        <?php foreach ($available_years as $year): ?>
            <option value="<?php echo $year; ?>" <?php echo $year == $selected_year ? 'selected' : ''; ?>><?php echo $year; ?></option>
        <?php endforeach; ?>
        <option value="<?php echo date('Y') + 1; ?>" <?php echo (date('Y') + 1) == $selected_year ? 'selected' : ''; ?>><?php echo date('Y') + 1; ?> (New)</option>
    </select>
    <small>Showing tax brackets for <?php echo $selected_year; ?></small>
</div>

<!-- Add Tax Rate Bracket Button -->
<button id="addTaxRateBtn" class="btn btn-primary">Add New Tax Bracket</button>

<!-- Add Tax Rate Modal -->
<div id="addTaxRateModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <span class="close" id="closeAddModal">&times;</span>
            <h3>Add New Tax Bracket for <?php echo $selected_year; ?></h3>
        </div>
        <div class="modal-body">
            <form method="post" action="manage_tax_rates.php" id="addTaxRateForm">
                <input type="hidden" name="tax_year" value="<?php echo $selected_year; ?>">

                <div class="form-row">
                    <div class="form-group">
                        <label for="min_income">Minimum Income (GH₵):</label>
                        <input type="number" step="0.01" id="min_income" name="min_income" min="0" required>
                    </div>
                    <div class="form-group">
                        <label for="max_income">Maximum Income (GH₵):</label>
                        <input type="number" step="0.01" id="max_income" name="max_income" min="0">
                        <small class="form-text text-muted">Leave empty for highest bracket</small>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="tax_rate">Tax Rate (%):</label>
                        <input type="number" step="0.01" id="tax_rate" name="tax_rate" min="0" max="100" required>
                    </div>
                    <div class="form-group">
                        <label for="cumulative_tax">Cumulative Tax (GH₵):</label>
                        <input type="number" step="0.01" id="cumulative_tax" name="cumulative_tax" min="0">
                        <small class="form-text text-muted">Tax from previous brackets (optional)</small>
                    </div>
                </div>
            </form>
        </div>
        <div class="modal-footer">
            <button type="submit" form="addTaxRateForm" name="add_tax_rate" class="btn btn-primary">Add Tax Bracket</button>
            <button type="button" id="cancelAddModal" class="btn">Cancel</button>
        </div>
    </div>
</div>

<!-- Edit Tax Rate Modal -->
<div id="editTaxRateModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <span class="close" id="closeEditModal">&times;</span>
            <h3>Edit Tax Bracket</h3>
        </div>
        <div class="modal-body">
            <form method="post" action="manage_tax_rates.php" id="editTaxRateForm">
                <input type="hidden" name="rate_id" id="edit_rate_id" value="">

                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_tax_year">Tax Year:</label>
                        <input type="number" id="edit_tax_year" name="tax_year" min="2000" max="2100" required>
                    </div>
                    <div class="form-group">
                        <label for="edit_min_income">Minimum Income (GH₵):</label>
                        <input type="number" step="0.01" id="edit_min_income" name="min_income" min="0" required>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_max_income">Maximum Income (GH₵):</label>
                        <input type="number" step="0.01" id="edit_max_income" name="max_income" min="0">
                        <small class="form-text text-muted">Leave empty for highest bracket</small>
                    </div>
                    <div class="form-group">
                        <label for="edit_tax_rate">Tax Rate (%):</label>
                        <input type="number" step="0.01" id="edit_tax_rate" name="tax_rate" min="0" max="100" required>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_cumulative_tax">Cumulative Tax (GH₵):</label>
                        <input type="number" step="0.01" id="edit_cumulative_tax" name="cumulative_tax" min="0">
                        <small class="form-text text-muted">Tax from previous brackets (optional)</small>
                    </div>
                </div>
            </form>
        </div>
        <div class="modal-footer">
            <button type="submit" form="editTaxRateForm" name="edit_tax_rate" class="btn btn-primary">Update Tax Bracket</button>
            <button type="button" id="cancelEditModal" class="btn">Cancel</button>
        </div>
    </div>
</div>

<!-- Tax Rates Table -->
<h3>Tax Brackets for <?php echo $selected_year; ?></h3>

<?php if (empty($tax_rates)): ?>
    <div class="no-rates">
        <h4>No Tax Brackets</h4>
        <p>No tax brackets have been set up for <?php echo $selected_year; ?>. Use the form above to add the first bracket.</p>
    </div>
<?php else: ?>
    <table class="tax-rates-table">
        <thead>
            <tr>
                <th>Income Range</th>
                <th>Tax Rate</th>
                <th>Cumulative Tax</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($tax_rates as $rate): ?>
                <tr>
                    <td>
                        GH₵ <?php echo number_format($rate['min_income'], 2); ?>
                        <?php if ($rate['max_income']): ?>
                            - GH₵ <?php echo number_format($rate['max_income'], 2); ?>
                        <?php else: ?>
                            and above
                        <?php endif; ?>
                    </td>
                    <td><?php echo $rate['tax_rate']; ?>%</td>
                    <td>GH₵ <?php echo number_format($rate['cumulative_tax'], 2); ?></td>
                    <td>
                        <button type="button" class="btn btn-primary btn-small" onclick="openEditModal(<?php echo $rate['id']; ?>, <?php echo $rate['tax_year']; ?>, <?php echo $rate['min_income']; ?>, <?php echo $rate['max_income'] ?: 'null'; ?>, <?php echo $rate['tax_rate']; ?>, <?php echo $rate['cumulative_tax']; ?>)">Edit</button>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
<?php endif; ?>

<script>
// Get the modals
var addModal = document.getElementById("addTaxRateModal");
var editModal = document.getElementById("editTaxRateModal");

// Get the buttons that open the modals
var addBtn = document.getElementById("addTaxRateBtn");

// Get the <span> elements that close the modals
var closeAdd = document.getElementById("closeAddModal");
var closeEdit = document.getElementById("closeEditModal");

// Get the cancel buttons
var cancelAdd = document.getElementById("cancelAddModal");
var cancelEdit = document.getElementById("cancelEditModal");

// When the user clicks the button, open the modal
addBtn.onclick = function() {
    addModal.style.display = "block";
}

// When the user clicks on <span> (x), close the modal
closeAdd.onclick = function() {
    addModal.style.display = "none";
}

closeEdit.onclick = function() {
    editModal.style.display = "none";
}

// When the user clicks on cancel button, close the modal
cancelAdd.onclick = function() {
    addModal.style.display = "none";
}

cancelEdit.onclick = function() {
    editModal.style.display = "none";
}

// When the user clicks anywhere outside of the modal, close it
window.onclick = function(event) {
    if (event.target == addModal) {
        addModal.style.display = "none";
    }
    if (event.target == editModal) {
        editModal.style.display = "none";
    }
}

// Function to open edit modal and populate fields
function openEditModal(id, tax_year, min_income, max_income, tax_rate, cumulative_tax) {
    document.getElementById("edit_rate_id").value = id;
    document.getElementById("edit_tax_year").value = tax_year;
    document.getElementById("edit_min_income").value = min_income;
    document.getElementById("edit_max_income").value = max_income === null ? '' : max_income;
    document.getElementById("edit_tax_rate").value = tax_rate;
    document.getElementById("edit_cumulative_tax").value = cumulative_tax;
    editModal.style.display = "block";
}

function changeYear(year) {
    window.location.href = 'manage_tax_rates.php?year=' + year;
}
</script>

<?php
include '../includes/admin_footer.php';
?>
