<?php
require_once '../config.php';
require_once '../includes/user_activity_logger.php';

$message = '';
$message_type = '';

// Handle session messages
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

// Fetch active allowance types
$allowance_types = $pdo->query("SELECT * FROM allowance_types WHERE is_active = TRUE ORDER BY name")->fetchAll();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['deactivate_salary_structure'])) {
        $structure_id = $_POST['structure_id'];

        try {
            $pdo->beginTransaction();

            $stmt = $pdo->prepare("UPDATE salary_structures SET is_active = FALSE WHERE id = ?");
            $stmt->execute([$structure_id]);

            $pdo->commit();

            // Log the deactivation activity
            UserActivityLogger::logActivity($_SESSION['user_id'], 'deactivate_salary_structure', "Deactivated salary structure ID: {$structure_id}");

            $_SESSION['message'] = 'Salary structure deactivated successfully.';
            $_SESSION['message_type'] = 'success';
            header('Location: manage_salary_structures.php');
            exit();
        } catch (Exception $e) {
            $pdo->rollBack();
            $message = 'Failed to deactivate salary structure: ' . $e->getMessage();
            $message_type = 'error';
        }
    } elseif (isset($_POST['add_salary_structure'])) {
        $staff_id = $_POST['staff_id'];
        $basic_salary = $_POST['basic_salary'];
        $effective_date = $_POST['effective_date'];
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        try {
            $pdo->beginTransaction();

            // Check if there is any structure with the same effective_date for this staff
            $stmt = $pdo->prepare("SELECT id FROM salary_structures WHERE staff_id = ? AND effective_date = ?");
            $stmt->execute([$staff_id, $effective_date]);
            $existing_structure = $stmt->fetch();

            if ($existing_structure) {
                // Deactivate all other active structures for this staff
                $stmt = $pdo->prepare("UPDATE salary_structures SET is_active = FALSE WHERE staff_id = ? AND is_active = TRUE AND id != ?");
                $stmt->execute([$staff_id, $existing_structure['id']]);

                // Update the existing structure and set it active
                $stmt = $pdo->prepare("UPDATE salary_structures SET basic_salary = ?, is_active = TRUE WHERE id = ?");
                $stmt->execute([$basic_salary, $existing_structure['id']]);

                // Delete existing allowances
                $stmt = $pdo->prepare("DELETE FROM staff_allowances WHERE salary_structure_id = ?");
                $stmt->execute([$existing_structure['id']]);

                // Insert updated allowances
                foreach ($allowance_types as $allowance_type) {
                    $allowance_key = 'allowance_' . $allowance_type['id'];
                    $amount = isset($_POST[$allowance_key]) ? floatval($_POST[$allowance_key]) : 0;

                    if ($amount > 0) {
                        $stmt = $pdo->prepare("
                            INSERT INTO staff_allowances (salary_structure_id, allowance_type_id, amount)
                            VALUES (?, ?, ?)
                            ON DUPLICATE KEY UPDATE amount = VALUES(amount)
                        ");
                        $stmt->execute([$existing_structure['id'], $allowance_type['id'], $amount]);
                    }
                }

                $pdo->commit();

                // Log the salary structure update activity
                UserActivityLogger::logActivity($_SESSION['user_id'], 'update_salary_structure', "Updated salary structure ID: {$existing_structure['id']} - Basic Salary: {$basic_salary}");

                $_SESSION['message'] = 'Salary structure updated successfully.';
                $_SESSION['message_type'] = 'success';
                header('Location: manage_salary_structures.php');
                exit();
            } else {
                // Deactivate previous active structure for this staff
                $stmt = $pdo->prepare("UPDATE salary_structures SET is_active = FALSE WHERE staff_id = ? AND is_active = TRUE");
                $stmt->execute([$staff_id]);

                // Insert new structure
                $stmt = $pdo->prepare("
                    INSERT INTO salary_structures (staff_id, basic_salary, effective_date, is_active)
                    VALUES (?, ?, ?, ?)
                ");
                $stmt->execute([$staff_id, $basic_salary, $effective_date, $is_active]);
                $structure_id = $pdo->lastInsertId();

                // Insert allowances into staff_allowances table
                foreach ($allowance_types as $allowance_type) {
                    $allowance_key = 'allowance_' . $allowance_type['id'];
                    $amount = isset($_POST[$allowance_key]) ? floatval($_POST[$allowance_key]) : 0;

                    if ($amount > 0) {
                        $stmt = $pdo->prepare("
                            INSERT INTO staff_allowances (salary_structure_id, allowance_type_id, amount)
                            VALUES (?, ?, ?)
                        ");
                        $stmt->execute([$structure_id, $allowance_type['id'], $amount]);
                    }
                }

                $pdo->commit();

                // Log the salary structure creation activity
                UserActivityLogger::logActivity($_SESSION['user_id'], 'create_salary_structure', "Created salary structure for staff ID: {$staff_id} - Basic Salary: {$basic_salary}");

                $_SESSION['message'] = 'Salary structure added successfully.';
                $_SESSION['message_type'] = 'success';
                header('Location: manage_salary_structures.php');
            }
        } catch (Exception $e) {
            $pdo->rollBack();
            $message = 'Failed to add/update salary structure: ' . $e->getMessage();
            $message_type = 'error';
        }
    } elseif (isset($_POST['edit_salary_structure'])) {
        $structure_id = $_POST['structure_id'];
        $basic_salary = $_POST['basic_salary'];
        $effective_date = $_POST['effective_date'];
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        try {
            $pdo->beginTransaction();

            // Update basic salary structure
            $stmt = $pdo->prepare("
                UPDATE salary_structures
                SET basic_salary = ?, effective_date = ?, is_active = ?
                WHERE id = ?
            ");
            $stmt->execute([$basic_salary, $effective_date, $is_active, $structure_id]);

            // Delete existing allowances for this structure
            $stmt = $pdo->prepare("DELETE FROM staff_allowances WHERE salary_structure_id = ?");
            $stmt->execute([$structure_id]);

            // Insert updated allowances
            foreach ($allowance_types as $allowance_type) {
                $allowance_key = 'allowance_' . $allowance_type['id'];
                $amount = isset($_POST[$allowance_key]) ? floatval($_POST[$allowance_key]) : 0;

                if ($amount > 0) {
                    $stmt = $pdo->prepare("
                        INSERT INTO staff_allowances (salary_structure_id, allowance_type_id, amount)
                        VALUES (?, ?, ?)
                    ");
                    $stmt->execute([$structure_id, $allowance_type['id'], $amount]);
                }
            }

            $pdo->commit();

            // Log the salary structure update activity
            UserActivityLogger::logActivity($_SESSION['user_id'], 'update_salary_structure', "Updated salary structure ID: {$structure_id} - Basic Salary: {$basic_salary}");

            $message = 'Salary structure updated successfully.';
            $message_type = 'success';
        } catch (Exception $e) {
            $pdo->rollBack();
            $message = 'Failed to update salary structure: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// Fetch all staff with their current salary from active salary structures
$stmt = $pdo->query("
    SELECT s.staff_id as id, CONCAT(s.first_name, ' ', s.last_name) as staff_name, s.role as staff_type,
           COALESCE(ss.basic_salary + COALESCE(sa.total_allowances, 0), 0) as current_salary
    FROM staff s
    LEFT JOIN salary_structures ss ON s.staff_id = ss.staff_id AND ss.is_active = TRUE
    LEFT JOIN (
        SELECT salary_structure_id, SUM(amount) as total_allowances
        FROM staff_allowances
        GROUP BY salary_structure_id
    ) sa ON ss.id = sa.salary_structure_id
    ORDER BY staff_name
");
$staff = $stmt->fetchAll();

// Fetch all salary structures
$stmt = $pdo->query("
    SELECT ss.*,
           COALESCE(CONCAT(s.first_name, ' ', s.last_name), ss.staff_id) as staff_name,
           s.role as staff_type
    FROM salary_structures ss
    LEFT JOIN staff s ON ss.staff_id = s.staff_id
    ORDER BY ss.is_active DESC, ss.effective_date DESC
");
$all_structures = $stmt->fetchAll();

// Get staff for edit
$edit_structure = null;
$edit_allowances = [];
if (isset($_GET['edit'])) {
    $structure_id = $_GET['edit'];
    $stmt = $pdo->prepare("
        SELECT ss.*,
               CONCAT(s.first_name, ' ', s.last_name) as staff_name,
               s.role as staff_type
        FROM salary_structures ss
        LEFT JOIN staff s ON ss.staff_id = s.staff_id
        WHERE ss.id = ?
    ");
    $stmt->execute([$structure_id]);
    $edit_structure = $stmt->fetch();

    // Fetch existing allowances for this structure
    if ($edit_structure) {
        $stmt = $pdo->prepare("
            SELECT sa.allowance_type_id, sa.amount, at.name, at.is_taxable, at.description
            FROM staff_allowances sa
            JOIN allowance_types at ON sa.allowance_type_id = at.id
            WHERE sa.salary_structure_id = ?
        ");
        $stmt->execute([$structure_id]);
        $edit_allowances = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Convert to associative array for easy lookup
        $edit_allowances_lookup = [];
        foreach ($edit_allowances as $allowance) {
            $edit_allowances_lookup[$allowance['allowance_type_id']] = $allowance;
        }
        $edit_allowances = $edit_allowances_lookup;
    }
}

function calculateGhanaTax($chargeable_income) {
    // GRA 2024 PAYE tax brackets for monthly chargeable income
    $tax = 0;

    if ($chargeable_income <= 490) {
        $tax = 0;
    } elseif ($chargeable_income <= 600) {
        $tax = ($chargeable_income - 490) * 0.05;
    } elseif ($chargeable_income <= 730) {
        $tax = 5.50 + ($chargeable_income - 600) * 0.10;
    } elseif ($chargeable_income <= 3896.67) {
        $tax = 18.50 + ($chargeable_income - 730) * 0.175;
    } elseif ($chargeable_income <= 19896.67) {
        $tax = 572.67 + ($chargeable_income - 3896.67) * 0.25;
    } elseif ($chargeable_income <= 50416.67) {
        $tax = 4572.67 + ($chargeable_income - 19896.67) * 0.30;
    } else {
        $tax = 13728.67 + ($chargeable_income - 50416.67) * 0.35;
    }

    return $tax;
}

include 'includes/accounting_header.php';
?>

<style>
.salary-form,
.salary-structures {
    background: white;
    border-radius: 12px;
    padding: 25px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    margin-bottom: 30px;
}

.salary-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 15px;
    border: 1px solid #e9ecef;
    border-radius: 8px;
    overflow: hidden;
}

.salary-table th,
.salary-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.salary-table th {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    font-weight: 600;
    font-size: 0.9em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    border-bottom: 2px solid #388E3C;
}

.salary-table tbody tr:hover {
    background-color: #f8f9fa;
}

.form-row {
    display: flex;
    gap: 20px;
    margin-bottom: 15px;
    flex-wrap: wrap;
}

.form-row .form-group {
    flex: 1;
    min-width: 200px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #333;
}

.form-group input,
.form-group select {
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
    transition: border-color 0.3s ease;
}

.form-group input:focus,
.form-group select:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 2px rgba(76, 175, 80, 0.2);
}

.total-salary {
    font-weight: bold;
    color: #27ae60;
    font-size: 1.1em;
}

.taxable-badge {
    background: #ff9800;
    color: white;
    padding: 2px 6px;
    border-radius: 3px;
    font-size: 0.8em;
    margin-left: 5px;
}

.no-structure {
    text-align: center;
    padding: 40px;
    color: #7f8c8d;
    background: #f8f9fa;
    border-radius: 8px;
    margin: 20px 0;
}

.btn {
    padding: 10px 20px;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    font-size: 14px;
    text-decoration: none;
    display: inline-block;
    transition: background-color 0.3s ease;
}

.btn-primary {
    background-color: #4CAF50;
    color: white;
}

.btn-primary:hover {
    background-color: #45a049;
}

.btn-small {
    padding: 5px 10px;
    font-size: 12px;
}

.form-text {
    font-size: 0.9em;
    color: #666;
    margin-top: 2px;
}

@media (max-width: 768px) {
    .salary-form,
    .salary-structures {
        padding: 15px;
    }

    .form-row {
        flex-direction: column;
        gap: 10px;
    }

    .salary-table th,
    .salary-table td {
        padding: 8px 10px;
        font-size: 0.9em;
    }
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
    animation: fadeIn 0.3s ease-out;
}

.modal-content {
    background-color: white;
    margin: 5% auto;
    padding: 0;
    border-radius: 12px;
    width: 90%;
    max-width: 800px;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 10px 30px rgba(0,0,0,0.3);
    animation: slideIn 0.3s ease-out;
}

.modal-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 20px 25px;
    border-radius: 12px 12px 0 0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-header h3 {
    margin: 0;
    font-size: 1.2em;
}

.close {
    color: white;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
    transition: color 0.3s ease;
}

.close:hover {
    color: #ddd;
}

.modal-body {
    padding: 25px;
}

@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

@keyframes slideIn {
    from { transform: translateY(-50px); opacity: 0; }
    to { transform: translateY(0); opacity: 1; }
}

@media (max-width: 768px) {
    .modal-content {
        width: 95%;
        margin: 10% auto;
    }
}
</style>

<h2>💰 Manage Salary Structures</h2>

<?php if ($message): ?>
    <div class="message <?php echo $message_type; ?>"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>

<!-- Add Salary Structure Button -->
<button id="openAddModal" class="btn btn-primary" style="margin-bottom: 20px;">➕ Add New Salary Structure</button>

<!-- Add Salary Structure Modal -->
<div id="addModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>➕ Add New Salary Structure</h3>
            <span class="close" id="closeAddModal">&times;</span>
        </div>
        <div class="modal-body">
            <form method="post" action="manage_salary_structures.php" id="addStructureForm">
    <div class="form-row">
        <div class="form-group">
            <label for="staff_select">Select Staff:</label>
            <select id="staff_select" name="staff_id" required>
                <option value="">-- Select Staff --</option>
                <?php foreach ($staff as $member): ?>
                    <option value="<?php echo $member['id']; ?>">
                        <?php echo htmlspecialchars($member['staff_name']); ?> (<?php echo ucfirst(str_replace('_', ' ', $member['staff_type'])); ?>)
                        (Current: GH₵ <?php echo number_format($member['current_salary'] ?? 0, 2); ?>)
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group">
            <label for="effective_date">Effective Date:</label>
            <input type="date" id="effective_date" name="effective_date" value="<?php echo date('Y-m-d'); ?>" required>
        </div>
    </div>

    <div class="form-row">
        <div class="form-group">
            <label for="basic_salary">Basic Salary (GH₵):</label>
            <input type="number" step="0.01" id="basic_salary" name="basic_salary" required min="0">
        </div>
    </div>

    <!-- Dynamic Allowance Fields -->
    <?php if (!empty($allowance_types)): ?>
        <h4>Allowances</h4>
        <?php
        $allowance_inputs = ['basic_salary'];
        for ($i = 0; $i < count($allowance_types); $i += 2):
            $allowance_type1 = $allowance_types[$i];
            $allowance_type2 = ($i + 1 < count($allowance_types)) ? $allowance_types[$i + 1] : null;
            $allowance_inputs[] = 'allowance_' . $allowance_type1['id'];
            if ($allowance_type2) {
                $allowance_inputs[] = 'allowance_' . $allowance_type2['id'];
            }
        ?>
            <div class="form-row">
                <div class="form-group">
                    <label for="allowance_<?php echo $allowance_type1['id']; ?>">
                        <?php echo htmlspecialchars($allowance_type1['name']); ?> (GH₵)<?php echo $allowance_type1['is_taxable'] ? ' <span class="taxable-badge">Taxable</span>' : ''; ?>
                        <?php if ($allowance_type1['description']): ?>
                            <small class="form-text text-muted"><?php echo htmlspecialchars($allowance_type1['description']); ?></small>
                        <?php endif; ?>
                    </label>
                    <input type="number" step="0.01" id="allowance_<?php echo $allowance_type1['id']; ?>" name="allowance_<?php echo $allowance_type1['id']; ?>" min="0" value="0">
                </div>
                <?php if ($allowance_type2): ?>
                    <div class="form-group">
                        <label for="allowance_<?php echo $allowance_type2['id']; ?>">
                            <?php echo htmlspecialchars($allowance_type2['name']); ?> (GH₵)<?php echo $allowance_type2['is_taxable'] ? ' <span class="taxable-badge">Taxable</span>' : ''; ?>
                            <?php if ($allowance_type2['description']): ?>
                                <small class="form-text text-muted"><?php echo htmlspecialchars($allowance_type2['description']); ?></small>
                            <?php endif; ?>
                        </label>
                        <input type="number" step="0.01" id="allowance_<?php echo $allowance_type2['id']; ?>" name="allowance_<?php echo $allowance_type2['id']; ?>" min="0" value="0">
                    </div>
                <?php endif; ?>
            </div>
        <?php endfor; ?>
    <?php else: ?>
        <div class="form-row">
            <div class="form-group" style="width: 100%; text-align: center; padding: 20px; background: #f8f9fa; border-radius: 8px;">
                <p>No active allowance types configured. <a href="manage_allowances.php">Manage Allowance Types</a></p>
            </div>
        </div>
    <?php endif; ?>

    <div class="form-row">
        <div class="form-group" style="flex-direction: row; align-items: center; gap: 10px;">
            <label>Total Gross Salary:</label>
            <span id="total_gross" class="total-salary">GH₵ 0.00</span>
        </div>
    </div>

    <div class="form-row">
        <div class="form-group" style="flex-direction: row; align-items: center; gap: 10px;">
            <input type="checkbox" id="is_active" name="is_active" value="1" checked>
            <label for="is_active">Active Status</label>
        </div>
    </div>

    <button type="submit" name="add_salary_structure" class="btn btn-primary">Add Salary Structure</button>
</form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Add Modal functionality
    const addModal = document.getElementById('addModal');
    const openAddModal = document.getElementById('openAddModal');
    const closeAddModal = document.getElementById('closeAddModal');

    // Open add modal
    openAddModal.addEventListener('click', function() {
        addModal.style.display = 'block';
    });

    // Close add modal
    closeAddModal.addEventListener('click', function() {
        addModal.style.display = 'none';
    });

    // Close add modal when clicking outside
    window.addEventListener('click', function(event) {
        if (event.target === addModal) {
            addModal.style.display = 'none';
        }
    });

    // Dynamic input calculation for add form
    function setupTotalCalculation() {
        const inputs = document.querySelectorAll('#addStructureForm input[type="number"]');
        inputs.forEach(input => {
            input.addEventListener('input', calculateTotal);
        });
    }

    function calculateTotal() {
        const inputs = document.querySelectorAll('#addStructureForm input[type="number"]');
        let total = 0;
        inputs.forEach(input => {
            const value = parseFloat(input.value) || 0;
            total += value;
        });
        document.getElementById('total_gross').textContent = 'GH₵ ' + total.toFixed(2);
    }

    setupTotalCalculation();



    // Edit Modal functionality
    const editModal = document.getElementById('editModal');
    const closeEditModal = document.getElementById('closeEditModal');

    // Close edit modal
    closeEditModal.addEventListener('click', function() {
        editModal.style.display = 'none';
    });

    // Close edit modal when clicking outside
    window.addEventListener('click', function(event) {
        if (event.target === editModal) {
            editModal.style.display = 'none';
        }
    });

    // Setup total calculation for edit form
    function setupEditTotalCalculation() {
        const inputs = document.querySelectorAll('#editStructureForm input[type="number"]');
        inputs.forEach(input => {
            input.addEventListener('input', calculateEditTotal);
        });
        calculateEditTotal(); // Initial calculation
    }

    function calculateEditTotal() {
        const inputs = document.querySelectorAll('#editStructureForm input[type="number"]');
        let total = 0;
        inputs.forEach(input => {
            const value = parseFloat(input.value) || 0;
            total += value;
        });
        document.getElementById('edit_total_gross').textContent = 'GH₵ ' + total.toFixed(2);
    }

    setupEditTotalCalculation();

    // Check if edit modal should be shown (when ?edit=ID is in URL)
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.has('edit')) {
        editModal.style.display = 'block';
    }
});
</script>

<!-- Edit Salary Structure Modal -->
<div id="editModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>✏️ Edit Salary Structure</h3>
            <span class="close" id="closeEditModal">&times;</span>
        </div>
        <div class="modal-body">
            <?php if ($edit_structure): ?>
                <form method="post" action="manage_salary_structures.php" id="editStructureForm">
                    <input type="hidden" name="structure_id" value="<?php echo $edit_structure['id']; ?>">

                    <div class="form-row">
                        <div class="form-group">
                            <label>Staff:</label>
                            <input type="text" value="<?php echo htmlspecialchars($edit_structure['staff_name']); ?>" readonly>
                        </div>
                        <div class="form-group">
                            <label for="effective_date">Effective Date:</label>
                            <input type="date" id="effective_date" name="effective_date" value="<?php echo $edit_structure['effective_date']; ?>" required>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label for="basic_salary">Basic Salary (GH₵):</label>
                            <input type="number" step="0.01" id="basic_salary" name="basic_salary" value="<?php echo $edit_structure['basic_salary']; ?>" required min="0">
                        </div>
                    </div>

                    <!-- Dynamic Allowance Fields for Edit -->
                    <?php if (!empty($allowance_types)): ?>
                        <h4>Allowances</h4>
                        <?php
                        for ($i = 0; $i < count($allowance_types); $i += 2):
                            $allowance_type1 = $allowance_types[$i];
                            $allowance_type2 = ($i + 1 < count($allowance_types)) ? $allowance_types[$i + 1] : null;
                            $existing_allowance1 = isset($edit_allowances[$allowance_type1['id']]) ? $edit_allowances[$allowance_type1['id']] : null;
                            $current_amount1 = $existing_allowance1 ? $existing_allowance1['amount'] : 0;
                            if ($allowance_type2) {
                                $existing_allowance2 = isset($edit_allowances[$allowance_type2['id']]) ? $edit_allowances[$allowance_type2['id']] : null;
                                $current_amount2 = $existing_allowance2 ? $existing_allowance2['amount'] : 0;
                            }
                        ?>
                            <div class="form-row">
                                <div class="form-group">
                                    <label for="allowance_<?php echo $allowance_type1['id']; ?>">
                                        <?php echo htmlspecialchars($allowance_type1['name']); ?> (GH₵)<?php echo $allowance_type1['is_taxable'] ? ' <span class="taxable-badge">Taxable</span>' : ''; ?>
                                        <?php if ($allowance_type1['description']): ?>
                                            <small class="form-text text-muted"><?php echo htmlspecialchars($allowance_type1['description']); ?></small>
                                        <?php endif; ?>
                                    </label>
                                    <input type="number" step="0.01" id="allowance_<?php echo $allowance_type1['id']; ?>" name="allowance_<?php echo $allowance_type1['id']; ?>" min="0" value="<?php echo $current_amount1; ?>">
                                </div>
                                <?php if ($allowance_type2): ?>
                                    <div class="form-group">
                                        <label for="allowance_<?php echo $allowance_type2['id']; ?>">
                                            <?php echo htmlspecialchars($allowance_type2['name']); ?> (GH₵)<?php echo $allowance_type2['is_taxable'] ? ' <span class="taxable-badge">Taxable</span>' : ''; ?>
                                            <?php if ($allowance_type2['description']): ?>
                                                <small class="form-text text-muted"><?php echo htmlspecialchars($allowance_type2['description']); ?></small>
                                            <?php endif; ?>
                                        </label>
                                        <input type="number" step="0.01" id="allowance_<?php echo $allowance_type2['id']; ?>" name="allowance_<?php echo $allowance_type2['id']; ?>" min="0" value="<?php echo $current_amount2; ?>">
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endfor; ?>
                    <?php else: ?>
                        <div class="form-row">
                            <div class="form-group" style="width: 100%; text-align: center; padding: 20px; background: #f8f9fa; border-radius: 8px;">
                                <p>No active allowance types configured. <a href="manage_allowances.php">Manage Allowance Types</a></p>
                            </div>
                        </div>
                    <?php endif; ?>

                    <div class="form-row">
                        <div class="form-group" style="flex-direction: row; align-items: center; gap: 10px;">
                            <label>Total Gross Salary:</label>
                            <span id="edit_total_gross" class="total-salary">GH₵ 0.00</span>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group" style="flex-direction: row; align-items: center; gap: 10px;">
                            <input type="checkbox" id="edit_is_active" name="is_active" value="1" <?php echo $edit_structure['is_active'] ? 'checked' : ''; ?>>
                            <label for="edit_is_active">Active Status</label>
                        </div>
                    </div>

                    <button type="submit" name="edit_salary_structure" class="btn btn-primary">Update Salary Structure</button>
                    <a href="manage_salary_structures.php" class="btn">Cancel</a>
                </form>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- All Salary Structures -->
<div class="salary-structures">
<h3>📋 All Salary Structures</h3>

<?php if (empty($all_structures)): ?>
    <div class="no-structure">
        <h4>No Salary Structures</h4>
        <p>No salary structures have been set up yet. Use the form above to add the first one.</p>
    </div>
<?php else: ?>
    <table class="salary-table">
        <thead>
            <tr>
                <th>Staff</th>
                <th>Status</th>
                <th>Basic Salary</th>
                <th>Allowances</th>
                <th>SSNIT Employee<br><small>(5.5% of Basic)</small></th>
                <th>Tier 2 Pension (Employer)<br><small>(13% of Basic)</small></th>
                <th>PAYE Tax<br><small>(on Chargeable Income)</small></th>
                <th>Total Deductions</th>
                <th>Net Salary</th>
                <th>Effective Date</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($all_structures as $structure): ?>
                <?php
                // Fetch allowances for this structure
                $stmt = $pdo->prepare("
                    SELECT sa.allowance_type_id, sa.amount, at.name
                    FROM staff_allowances sa
                    JOIN allowance_types at ON sa.allowance_type_id = at.id
                    WHERE sa.salary_structure_id = ?
                ");
                $stmt->execute([$structure['id']]);
                $structure_allowances = $stmt->fetchAll(PDO::FETCH_ASSOC);

                // Calculate totals
                $basic_salary = $structure['basic_salary'];
                $total_allowances = 0;
                $allowance_breakdown = [];
                foreach ($structure_allowances as $allowance) {
                    $total_allowances += $allowance['amount'];
                    $allowance_breakdown[] = htmlspecialchars($allowance['name']) . ': GH₵' . number_format($allowance['amount'], 2);
                }
                $gross_salary = $basic_salary + $total_allowances;

                // Calculate deductions
                $ssnit_employee = $basic_salary * 0.055;
                $ssnit_employer = $basic_salary * 0.13;
                $chargeable_income = $gross_salary - $ssnit_employee;
                $paye_tax = calculateGhanaTax($chargeable_income);
                $total_deductions = $ssnit_employee + $paye_tax;
                $net_salary = $gross_salary - $total_deductions;
                ?>
                <tr>
                    <td><?php echo htmlspecialchars($structure['staff_name']); ?></td>
                    <td>
                        <?php if ($structure['is_active']): ?>
                            <span style="color: #28a745; font-weight: bold;">Active</span>
                        <?php else: ?>
                            <span style="color: #dc3545; font-weight: bold;">Inactive</span>
                        <?php endif; ?>
                    </td>
                    <td>GH₵ <?php echo number_format($basic_salary, 2); ?></td>
                    <td>
                        <span title="<?php echo implode(', ', $allowance_breakdown); ?>" style="cursor: help;">
                            GH₵ <?php echo number_format($total_allowances, 2); ?>
                        </span>
                    </td>
                    <td>GH₵ <?php echo number_format($ssnit_employee, 2); ?><br><small>5.5%</small></td>
                    <td>GH₵ <?php echo number_format($ssnit_employer, 2); ?><br><small>13%</small></td>
                    <td>GH₵ <?php echo number_format($paye_tax, 2); ?><br><small>on GH₵<?php echo number_format($chargeable_income, 2); ?></small></td>
                    <td><strong>GH₵ <?php echo number_format($total_deductions, 2); ?></strong></td>
                    <td><strong style="color: #27ae60;">GH₵ <?php echo number_format($net_salary, 2); ?></strong></td>
                    <td><?php echo date('M d, Y', strtotime($structure['effective_date'])); ?></td>
                    <td>
                        <a href="?edit=<?php echo $structure['id']; ?>" class="btn btn-small" style="background-color: #28a745; color: white;">Edit</a>
                        <form method="post" action="manage_salary_structures.php" style="display: inline;">
                            <input type="hidden" name="structure_id" value="<?php echo $structure['id']; ?>">
                            <button type="submit" name="deactivate_salary_structure" class="btn btn-small" style="background-color: #dc3545; color: white;" onclick="return confirm('Are you sure you want to deactivate this salary structure?')">Deactivate</button>
                        </form>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
<?php endif; ?>
</div>

<?php
include '../includes/admin_footer.php';
?>
