<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';

// Store management is now accessible to all logged-in users


$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_item'])) {
        $name = trim($_POST['name']);
        $category_id = intval($_POST['category_id']);
        $description = trim($_POST['description']);
        $unit = trim($_POST['unit']);
        $quantity = intval($_POST['quantity']);
        $location = trim($_POST['location']);
        $supplier = trim($_POST['supplier']);
        $cost_price = floatval($_POST['cost_price']);
        $selling_price = floatval($_POST['selling_price']);
        $status = $_POST['status'];

        try {
            $stmt = $pdo->prepare("INSERT INTO inventory_items (name, category_id, description, unit, quantity, location, supplier, cost_price, selling_price, status, added_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$name, $category_id, $description, $unit, $quantity, $location, $supplier, $cost_price, $selling_price, $status, $_SESSION['user_id']]);

            // Get the inserted item ID
            $item_id = $pdo->lastInsertId();

            // Log the item addition activity
            AccountantActivityLogger::logActivity(
                $_SESSION['user_id'],
                'inventory_item_added',
                "Added inventory item: {$name} (ID: {$item_id}) - Quantity: {$quantity}, Cost: GHC {$cost_price}, Selling: GHC {$selling_price}",
                $cost_price
            );

            $message = "Inventory item added successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error adding item: " . $e->getMessage();
            $message_type = "error";
        }
    }

    if (isset($_POST['add_category'])) {
        $category_name = trim($_POST['category_name']);
        $category_description = trim($_POST['category_description']);

        if (empty($category_name)) {
            $message = "Category name is required.";
            $message_type = "error";
        } else {
            try {
                $stmt = $pdo->prepare("INSERT INTO inventory_categories (name, description) VALUES (?, ?)");
                $stmt->execute([$category_name, $category_description]);

                // Get the inserted category ID
                $category_id = $pdo->lastInsertId();

                // Log the category addition activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'inventory_category_added',
                    "Added inventory category: {$category_name} (ID: {$category_id})",
                    0
                );

                $message = "Category added successfully!";
                $message_type = "success";
            } catch (PDOException $e) {
                $message = "Error adding category: " . $e->getMessage();
                $message_type = "error";
            }
        }
    }
}

// Fetch categories
try {
    $categories_stmt = $pdo->query("SELECT * FROM inventory_categories ORDER BY name");
    $categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $categories = [];
}

// Fetch items
try {
    $items_stmt = $pdo->query("SELECT ii.*, ic.name as category_name FROM inventory_items ii LEFT JOIN inventory_categories ic ON ii.category_id = ic.id ORDER BY ii.name");
    $items = $items_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $items = [];
}

$page_title = "Store Management - Items";
include 'includes/accounting_header.php';
?>

<div class="container-fluid">
    <h2>Store Management - Items</h2>

    <?php if ($message): ?>
        <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
            <?php echo htmlspecialchars($message); ?>
            <button type="button" class="close" data-dismiss="alert">
                <span>&times;</span>
            </button>
        </div>
    <?php endif; ?>

    <button class="btn btn-facilities mb-3" data-toggle="modal" data-target="#addItemModal">
        <i class="fas fa-plus"></i> Add New Item
    </button>

    <button class="btn btn-secondary mb-3 ml-2" data-toggle="modal" data-target="#manageCategoriesModal">
        <i class="fas fa-tags"></i> Manage Categories
    </button>

    <div class="table-responsive">
        <table class="table table-striped table-hover facilities-table">
            <thead class="thead-dark">
                <tr>
                    <th>Name</th>
                    <th>Category</th>
                    <th>Unit</th>
                    <th>Quantity</th>
                    <th>Location</th>
                    <th>Supplier</th>
                    <th>Cost Price</th>
                    <th>Selling Price</th>
                    <th>Status</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($items as $item): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($item['name']); ?></td>
                        <td><?php echo htmlspecialchars($item['category_name']); ?></td>
                        <td><?php echo htmlspecialchars($item['unit']); ?></td>
                        <td><?php echo htmlspecialchars($item['quantity']); ?></td>
                        <td><?php echo htmlspecialchars($item['location']); ?></td>
                        <td><?php echo htmlspecialchars($item['supplier']); ?></td>
                        <td><?php echo htmlspecialchars($item['cost_price']); ?></td>
                        <td><?php echo htmlspecialchars($item['selling_price']); ?></td>
                        <td><?php echo htmlspecialchars($item['status']); ?></td>
                    </tr>
                <?php endforeach; ?>
                <?php if (empty($items)): ?>
                    <tr>
                        <td colspan="9" class="text-center">No inventory items found.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Add Item Modal -->
<div class="modal fade" id="addItemModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title">Add New Inventory Item</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <label for="name">Name *</label>
                            <input type="text" class="form-control" id="name" name="name" required>
                        </div>
                        <div class="form-group col-md-6">
                            <label for="category_id">Category *</label>
                            <select class="form-control" id="category_id" name="category_id" required>
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="description">Description</label>
                        <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                    </div>
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <label for="unit">Unit</label>
                            <input type="text" class="form-control" id="unit" name="unit" value="pieces">
                        </div>
                        <div class="form-group col-md-6">
                            <label for="quantity">Quantity</label>
                            <input type="number" class="form-control" id="quantity" name="quantity" min="0" value="0">
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <label for="location">Location</label>
                            <input type="text" class="form-control" id="location" name="location">
                        </div>
                        <div class="form-group col-md-6">
                            <label for="supplier">Supplier</label>
                            <input type="text" class="form-control" id="supplier" name="supplier">
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <label for="cost_price">Cost Price</label>
                            <input type="number" step="0.01" class="form-control" id="cost_price" name="cost_price" min="0">
                        </div>
                        <div class="form-group col-md-6">
                            <label for="selling_price">Selling Price</label>
                            <input type="number" step="0.01" class="form-control" id="selling_price" name="selling_price" min="0">
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="status">Status</label>
                        <select class="form-control" id="status" name="status">
                            <option value="active" selected>Active</option>
                            <option value="inactive">Inactive</option>
                            <option value="discontinued">Discontinued</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="add_item" class="btn btn-primary">Add Item</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Manage Categories Modal -->
<div class="modal fade" id="manageCategoriesModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-xl" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Manage Categories</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <!-- Existing Categories -->
                <div class="mb-4">
                    <h6>Existing Categories</h6>
                    <div class="table-responsive">
                        <table class="table table-sm table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Description</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($categories as $category): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($category['id']); ?></td>
                                        <td><?php echo htmlspecialchars($category['name']); ?></td>
                                        <td><?php echo htmlspecialchars($category['description'] ?? ''); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                                <?php if (empty($categories)): ?>
                                    <tr>
                                        <td colspan="3" class="text-center">No categories found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <hr>

                <!-- Add New Category -->
                <div>
                    <h6>Add New Category</h6>
                    <form method="POST">
                        <div class="form-row">
                            <div class="form-group col-md-6">
                                <label for="category_name">Category Name *</label>
                                <input type="text" class="form-control" id="category_name" name="category_name" required>
                            </div>
                            <div class="form-group col-md-6">
                                <label for="category_description">Description</label>
                                <input type="text" class="form-control" id="category_description" name="category_description">
                            </div>
                        </div>
                        <button type="submit" name="add_category" class="btn btn-primary">Add Category</button>
                    </form>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<script>
$('#manageCategoriesModal').on('shown.bs.modal', function () {
    $(this).removeAttr('aria-hidden');
});
</script>

<?php include '../includes/admin_footer.php'; ?>
