<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';

// Check permissions - only admins and accounts users can manage chart of accounts
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    header('Location: ../login.php');
    exit;
}

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_account'])) {
        // Add new account
        $account_code = trim($_POST['account_code']);
        $account_name = trim($_POST['account_name']);
        $account_type = $_POST['account_type'];
        $account_category = !empty($_POST['account_category']) ? $_POST['account_category'] : null;
        $normal_balance = $_POST['normal_balance'];
        $is_control_account = isset($_POST['is_control_account']) ? (int)$_POST['is_control_account'] : 0;
        $allow_manual_posting = isset($_POST['allow_manual_posting']) ? (int)$_POST['allow_manual_posting'] : 1;
        $show_in_reports = isset($_POST['show_in_reports']) ? (int)$_POST['show_in_reports'] : 1;
        $status = $_POST['status'] ?? 'active';
        $is_active = ($status === 'active') ? 1 : 0;
        $description = trim($_POST['description']);
        $parent_id = !empty($_POST['parent_id']) ? $_POST['parent_id'] : null;

        if (empty($account_code) || empty($account_name)) {
            $error = "Account code and name are required.";
        } elseif (!preg_match('/^\d{4}$/', $account_code)) {
            $error = "Account code must be exactly 4 digits.";
        } else {
            try {
                // Check if account code already exists
                $stmt = $pdo->prepare("SELECT id FROM chart_of_accounts WHERE account_code = ?");
                $stmt->execute([$account_code]);
                if ($stmt->fetch()) {
                    $error = "An account with this code already exists.";
                } else {
                    $stmt = $pdo->prepare("
                        INSERT INTO chart_of_accounts (
                            account_code, account_name, account_type, account_category, normal_balance,
                            is_control_account, allow_manual_posting, show_in_reports, status, is_active,
                            description, parent_account_id, is_system_account, created_by
                        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, FALSE, ?)
                    ");
                    $stmt->execute([
                        $account_code, $account_name, $account_type, $account_category, $normal_balance,
                        $is_control_account, $allow_manual_posting, $show_in_reports, $status, $is_active,
                        $description, $parent_id, $_SESSION['user_id']
                    ]);

                    // Log activity
                    AccountantActivityLogger::logActivity(
                        $_SESSION['user_id'],
                        'coa_account_created',
                        "Created new chart of accounts entry: $account_code - $account_name"
                    );

                    $message = "Account added successfully!";
                }
            } catch (PDOException $e) {
                $error = "Error adding account: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['edit_account'])) {
        // Edit existing account
        $id = $_POST['account_id'];
        $account_code = trim($_POST['account_code']);
        $account_name = trim($_POST['account_name']);
        $account_type = $_POST['account_type'];
        $account_category = !empty($_POST['account_category']) ? $_POST['account_category'] : null;
        $normal_balance = $_POST['normal_balance'];
        $is_control_account = isset($_POST['is_control_account']) ? (int)$_POST['is_control_account'] : 0;
        $allow_manual_posting = isset($_POST['allow_manual_posting']) ? (int)$_POST['allow_manual_posting'] : 1;
        $show_in_reports = isset($_POST['show_in_reports']) ? (int)$_POST['show_in_reports'] : 1;
        $status = $_POST['status'] ?? 'active';
        $is_active = ($status === 'active') ? 1 : 0;
        $description = trim($_POST['description']);
        $parent_id = !empty($_POST['parent_id']) ? $_POST['parent_id'] : null;

        if (empty($account_code) || empty($account_name)) {
            $error = "Account code and name are required.";
        } elseif (!preg_match('/^\d{4}$/', $account_code)) {
            $error = "Account code must be exactly 4 digits.";
        } else {
            try {
                // Check if another account with this code exists
                $stmt = $pdo->prepare("SELECT id FROM chart_of_accounts WHERE account_code = ? AND id != ?");
                $stmt->execute([$account_code, $id]);
                if ($stmt->fetch()) {
                    $error = "Another account with this code already exists.";
                } else {
                    $stmt = $pdo->prepare("
                        UPDATE chart_of_accounts
                        SET account_code = ?, account_name = ?, account_type = ?, account_category = ?,
                            normal_balance = ?, is_control_account = ?, allow_manual_posting = ?,
                            show_in_reports = ?, status = ?, is_active = ?, description = ?, parent_account_id = ?
                        WHERE id = ? AND is_system_account = FALSE
                    ");
                    $stmt->execute([
                        $account_code, $account_name, $account_type, $account_category, $normal_balance,
                        $is_control_account, $allow_manual_posting, $show_in_reports, $status, $is_active,
                        $description, $parent_id, $id
                    ]);

                    if ($stmt->rowCount() > 0) {
                        // Log activity
                        AccountantActivityLogger::logActivity(
                            $_SESSION['user_id'],
                            'coa_account_updated',
                            "Updated chart of accounts entry: $account_code - $account_name"
                        );

                        $message = "Account updated successfully!";
                    } else {
                        $error = "Account not found or cannot be edited.";
                    }
                }
            } catch (PDOException $e) {
                $error = "Error updating account: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['toggle_active'])) {
        // Toggle active status (only for non-system accounts)
        $id = $_POST['account_id'];

        try {
            $stmt = $pdo->prepare("
                UPDATE chart_of_accounts
                SET is_active = NOT is_active,
                    status = CASE WHEN is_active = 1 THEN 'inactive' ELSE 'active' END
                WHERE id = ? AND is_system_account = FALSE
            ");
            $stmt->execute([$id]);

            if ($stmt->rowCount() > 0) {
                $status = $_POST['current_status'] == '1' ? 'deactivated' : 'activated';
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'coa_account_status_changed',
                    "Chart of accounts entry ID $id $status"
                );

                $message = "Account status updated successfully!";
            } else {
                $error = "Account not found or cannot be modified.";
            }
        } catch (PDOException $e) {
            $error = "Error updating account status: " . $e->getMessage();
        }
    }
}

// Get all accounts for display
try {
    $stmt = $pdo->query("
        SELECT coa.*, parent.account_name as parent_name, u.username as creator_name,
               (
                   SELECT COUNT(*) FROM journal_entry_lines jel WHERE jel.account_id = coa.id
               ) as usage_count
        FROM chart_of_accounts coa
        LEFT JOIN chart_of_accounts parent ON coa.parent_account_id = parent.id
        LEFT JOIN users u ON coa.created_by = u.id
        ORDER BY coa.account_code ASC
    ");
    $accounts = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading accounts: " . $e->getMessage();
    $accounts = [];
}

// Get parent accounts for dropdown (only active accounts can be parents)
try {
    $stmt = $pdo->query("SELECT id, account_code, account_name FROM chart_of_accounts WHERE parent_account_id IS NULL AND is_active = TRUE ORDER BY account_code");
    $parent_accounts = $stmt->fetchAll();
} catch (PDOException $e) {
    $parent_accounts = [];
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .settings-container {
        max-width: 1400px;
        margin: 0 auto;
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .settings-header {
        background: white;
        padding: 25px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .settings-content {
        display: grid;
        grid-template-columns: 1fr;
        gap: 30px;
    }

    .accounts-list-section {
        background: white;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .add-account-btn {
        margin-bottom: 20px;
    }

    .section-title {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
    }

    .form-group {
        margin-bottom: 20px;
    }

    .form-group label {
        display: block;
        font-weight: 600;
        margin-bottom: 5px;
        color: #495057;
    }

    .form-group input,
    .form-group select,
    .form-group textarea {
        width: 100%;
        padding: 10px 12px;
        border: 1px solid #ced4da;
        border-radius: 5px;
        font-size: 14px;
    }

    .form-group textarea {
        resize: vertical;
        min-height: 80px;
    }

    .form-row {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px;
    }

    .btn {
        padding: 10px 20px;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        display: inline-block;
        margin-right: 10px;
    }

    .btn-primary {
        background: #007bff;
        color: white;
    }

    .btn-success {
        background: #28a745;
        color: white;
    }

    .btn-danger {
        background: #dc3545;
        color: white;
    }

    .btn-secondary {
        background: #6c757d;
        color: white;
    }

    .btn-warning {
        background: #ffc107;
        color: #212529;
    }

    .message {
        padding: 15px;
        border-radius: 5px;
        margin-bottom: 20px;
    }

    .success-message {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .error-message {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .accounts-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    .accounts-table th,
    .accounts-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .accounts-table th {
        background: #f8f9fa;
        font-weight: 600;
        color: #495057;
    }

    .account-code {
        font-weight: bold;
        color: #007bff;
    }

    .account-type {
        display: inline-block;
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: 600;
    }

    .type-asset {
        background: #d1ecf1;
        color: #0c5460;
    }

    .type-liability {
        background: #f8d7da;
        color: #721c24;
    }

    .type-equity {
        background: #d4edda;
        color: #155724;
    }

    .type-revenue {
        background: #fff3cd;
        color: #856404;
    }

    .type-expense {
        background: #e2e3e5;
        color: #383d41;
    }

    .system-account {
        background: #e9ecef;
        color: #495057;
    }

    .actions-column {
        width: 200px;
    }

    .modal {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        z-index: 1000;
    }

    .modal-content {
        background: white;
        margin: 5% auto;
        padding: 30px;
        border-radius: 10px;
        width: 90%;
        max-width: 600px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        max-height: 90vh;
        overflow-y: auto;
    }

    .modal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
        padding-bottom: 10px;
        border-bottom: 1px solid #e9ecef;
    }

    .modal-title {
        margin: 0;
        color: #2c3e50;
    }

    .close {
        font-size: 24px;
        cursor: pointer;
        color: #6c757d;
    }

    .status-badge {
        display: inline-block;
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: 600;
    }

    .status-active {
        background: #d4edda;
        color: #155724;
    }

    .status-inactive {
        background: #f8d7da;
        color: #721c24;
    }

    @media (max-width: 768px) {
        .settings-content {
            grid-template-columns: 1fr;
        }

        .form-row {
            grid-template-columns: 1fr;
        }
    }
</style>

<div class="settings-container">
    <div class="settings-header">
        <h1>📊 Chart of Accounts Management</h1>
        <p>Manage your school's chart of accounts</p>
    </div>

    <?php if ($message): ?>
        <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="settings-content">
        <div class="accounts-list-section">
            <h2 class="section-title">📋 All Accounts</h2>
            <button class="btn btn-primary add-account-btn" onclick="openAddModal()">➕ Add New Account</button>
            <table class="accounts-table">
                <thead>
                    <tr>
                        <th>Code</th>
                        <th>Name</th>
                        <th>Type</th>
                        <th>Category</th>
                        <th>Balance</th>
                        <th>Balance Sheet</th>
                        <th>Status</th>
                        <th>Transactions</th>
                        <th class="actions-column">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($accounts as $account): ?>
                        <tr>
                            <td class="account-code"><?php echo htmlspecialchars($account['account_code']); ?></td>
                            <td>
                                <strong><?php echo htmlspecialchars($account['account_name']); ?></strong>
                                <?php if ($account['description']): ?>
                                    <br><small style="color: #6c757d;"><?php echo htmlspecialchars($account['description']); ?></small>
                                <?php endif; ?>
                                <?php if ($account['parent_name']): ?>
                                    <br><small style="color: #007bff;">Parent: <?php echo htmlspecialchars($account['parent_name']); ?></small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="account-type type-<?php echo strtolower($account['account_type']); ?>">
                                    <?php echo $account['account_type']; ?>
                                </span>
                            </td>
                            <td><?php echo htmlspecialchars($account['account_category'] ?? ''); ?></td>
                            <td><?php echo ucfirst($account['normal_balance']); ?></td>
                            <td>
                                <span class="status-badge <?php echo ($account['show_in_balance_sheet'] ?? 1) ? 'status-active' : 'status-inactive'; ?>">
                                    <?php echo ($account['show_in_balance_sheet'] ?? 1) ? 'Yes' : 'No'; ?>
                                </span>
                            </td>
                            <td>
                                <span class="status-badge <?php echo $account['is_active'] ? 'status-active' : 'status-inactive'; ?>">
                                    <?php echo $account['is_active'] ? 'Active' : 'Inactive'; ?>
                                </span>
                                <?php if ($account['is_system_account']): ?>
                                    <br><span class="account-type system-account">System</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo $account['usage_count']; ?> entries</td>
                            <td class="actions-column">
                                <?php if (!$account['is_system_account']): ?>
                                    <button class="btn btn-secondary btn-sm" onclick="editAccount(<?php echo $account['id']; ?>, '<?php echo addslashes($account['account_code']); ?>', '<?php echo addslashes($account['account_name']); ?>', '<?php echo addslashes($account['account_type']); ?>', '<?php echo addslashes($account['account_category'] ?? ''); ?>', '<?php echo addslashes($account['normal_balance']); ?>', <?php echo $account['is_control_account'] ?? 0; ?>, <?php echo $account['allow_manual_posting'] ?? 1; ?>, <?php echo $account['show_in_reports'] ?? 1; ?>, '<?php echo addslashes($account['status'] ?? 'active'); ?>', '<?php echo addslashes($account['description'] ?? ''); ?>', '<?php echo $account['parent_account_id'] ?? ''; ?>')">Edit</button>
                                    <button class="btn <?php echo $account['is_active'] ? 'btn-warning' : 'btn-success'; ?> btn-sm" onclick="toggleStatus(<?php echo $account['id']; ?>, <?php echo $account['is_active'] ? 'true' : 'false'; ?>)">
                                        <?php echo $account['is_active'] ? 'Deactivate' : 'Activate'; ?>
                                    </button>
                                <?php else: ?>
                                    <em style="color: #6c757d;">System Account</em>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Edit Account Modal -->
<div id="editModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Edit Account</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <form method="post">
            <input type="hidden" name="account_id" id="edit_account_id">
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_account_code">Account Code *</label>
                    <input type="text" name="account_code" id="edit_account_code" required pattern="\d{4}" title="Must be exactly 4 digits">
                </div>
                <div class="form-group">
                    <label for="edit_account_name">Account Name *</label>
                    <input type="text" name="account_name" id="edit_account_name" required>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_account_type">Account Type *</label>
                    <select name="account_type" id="edit_account_type" required>
                        <option value="Asset">Asset</option>
                        <option value="Liability">Liability</option>
                        <option value="Equity">Equity</option>
                        <option value="Revenue">Revenue</option>
                        <option value="Expense">Expense</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="edit_account_category">Account Category</label>
                    <select name="account_category" id="edit_account_category">
                        <option value="">Select Category</option>
                        <option value="Current Asset">Current Asset</option>
                        <option value="Non-Current Asset">Non-Current Asset</option>
                        <option value="Current Liability">Current Liability</option>
                        <option value="Non-Current Liability">Non-Current Liability</option>
                        <option value="Equity">Equity</option>
                        <option value="Revenue">Revenue</option>
                        <option value="Expense">Expense</option>
                    </select>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_normal_balance">Normal Balance *</label>
                    <select name="normal_balance" id="edit_normal_balance" required>
                        <option value="debit">Debit</option>
                        <option value="credit">Credit</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="edit_is_control_account">Is Control Account</label>
                    <select name="is_control_account" id="edit_is_control_account">
                        <option value="0">No</option>
                        <option value="1">Yes</option>
                    </select>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_allow_manual_posting">Allow Manual Posting</label>
                    <select name="allow_manual_posting" id="edit_allow_manual_posting">
                        <option value="1">Yes</option>
                        <option value="0">No</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="edit_show_in_reports">Show in Reports</label>
                    <select name="show_in_reports" id="edit_show_in_reports">
                        <option value="1">Yes</option>
                        <option value="0">No</option>
                    </select>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_status">Status</label>
                    <select name="status" id="edit_status">
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="edit_parent_id">Parent Account</label>
                    <select name="parent_id" id="edit_parent_id">
                        <option value="">None (Top Level)</option>
                        <?php foreach ($parent_accounts as $parent): ?>
                            <option value="<?php echo $parent['id']; ?>">
                                <?php echo htmlspecialchars($parent['account_code'] . ' - ' . $parent['account_name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            <div class="form-group">
                <label for="edit_description">Description</label>
                <textarea name="description" id="edit_description"></textarea>
            </div>
            <button type="submit" name="edit_account" class="btn btn-success">Update Account</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<!-- Add Account Modal -->
<div id="addModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Add New Account</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <form method="post">
            <div class="form-row">
                <div class="form-group">
                    <label for="add_account_code">Account Code *</label>
                    <input type="text" name="account_code" id="add_account_code" required pattern="\d{4}" title="Must be exactly 4 digits">
                </div>
                <div class="form-group">
                    <label for="add_account_name">Account Name *</label>
                    <input type="text" name="account_name" id="add_account_name" required>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="add_account_type">Account Type *</label>
                    <select name="account_type" id="add_account_type" required>
                        <option value="Asset">Asset</option>
                        <option value="Liability">Liability</option>
                        <option value="Equity">Equity</option>
                        <option value="Revenue">Revenue</option>
                        <option value="Expense">Expense</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="add_account_category">Account Category</label>
                    <select name="account_category" id="add_account_category">
                        <option value="">Select Category</option>
                        <option value="Current Asset">Current Asset</option>
                        <option value="Non-Current Asset">Non-Current Asset</option>
                        <option value="Current Liability">Current Liability</option>
                        <option value="Non-Current Liability">Non-Current Liability</option>
                        <option value="Equity">Equity</option>
                        <option value="Revenue">Revenue</option>
                        <option value="Expense">Expense</option>
                    </select>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="add_normal_balance">Normal Balance *</label>
                    <select name="normal_balance" id="add_normal_balance" required>
                        <option value="debit">Debit</option>
                        <option value="credit">Credit</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="add_is_control_account">Is Control Account</label>
                    <select name="is_control_account" id="add_is_control_account">
                        <option value="0">No</option>
                        <option value="1">Yes</option>
                    </select>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="add_allow_manual_posting">Allow Manual Posting</label>
                    <select name="allow_manual_posting" id="add_allow_manual_posting">
                        <option value="1">Yes</option>
                        <option value="0">No</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="add_show_in_reports">Show in Reports</label>
                    <select name="show_in_reports" id="add_show_in_reports">
                        <option value="1">Yes</option>
                        <option value="0">No</option>
                    </select>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="add_status">Status</label>
                    <select name="status" id="add_status">
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="add_parent_id">Parent Account</label>
                    <select name="parent_id" id="add_parent_id">
                        <option value="">None (Top Level)</option>
                        <?php foreach ($parent_accounts as $parent): ?>
                            <option value="<?php echo $parent['id']; ?>">
                                <?php echo htmlspecialchars($parent['account_code'] . ' - ' . $parent['account_name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            <div class="form-group">
                <label for="add_description">Description</label>
                <textarea name="description" id="add_description"></textarea>
            </div>
            <button type="submit" name="add_account" class="btn btn-success">Add Account</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<!-- Status Toggle Modal -->
<div id="statusModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Change Account Status</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <p>Are you sure you want to <span id="status_action"></span> this account?</p>
        <p style="color: #856404; font-weight: 600;">This will affect whether the account appears in dropdowns and reports.</p>
        <form method="post">
            <input type="hidden" name="account_id" id="status_account_id">
            <input type="hidden" name="current_status" id="current_status">
            <button type="submit" name="toggle_active" class="btn btn-warning">Yes, Change Status</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<script>
function editAccount(id, code, name, type, category, balance, isControl, allowManual, showInReports, status, description, parentId) {
    document.getElementById('edit_account_id').value = id;
    document.getElementById('edit_account_code').value = code;
    document.getElementById('edit_account_name').value = name;
    document.getElementById('edit_account_type').value = type;
    document.getElementById('edit_account_category').value = category;
    document.getElementById('edit_normal_balance').value = balance;
    document.getElementById('edit_is_control_account').value = isControl;
    document.getElementById('edit_allow_manual_posting').value = allowManual;
    document.getElementById('edit_show_in_reports').value = showInReports;
    document.getElementById('edit_status').value = status;
    document.getElementById('edit_description').value = description;
    document.getElementById('edit_parent_id').value = parentId;
    document.getElementById('editModal').style.display = 'block';
}

function toggleStatus(id, isActive) {
    document.getElementById('status_account_id').value = id;
    document.getElementById('current_status').value = isActive ? '1' : '0';
    document.getElementById('status_action').textContent = isActive ? 'deactivate' : 'activate';
    document.getElementById('statusModal').style.display = 'block';
}

function openAddModal() {
    document.getElementById('addModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('editModal').style.display = 'none';
    document.getElementById('addModal').style.display = 'none';
    document.getElementById('statusModal').style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    if (event.target.className === 'modal') {
        event.target.style.display = 'none';
    }
}
</script>

<?php include '../includes/admin_footer.php'; ?>
