<?php
require_once '../config.php';
require_once '../includes/access_control.php';
require_once 'asset_audit_functions.php';

// Access control temporarily disabled for assets management page
// requireFeatureAccess('accounts_management');

include 'includes/accounting_header.php';

$message = '';
$error = '';

// Get filter parameters
$search = $_GET['search'] ?? '';
$category_filter = $_GET['category'] ?? '';
$status_filter = $_GET['status'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$sort_by = $_GET['sort_by'] ?? 'purchase_date';
$sort_order = $_GET['sort_order'] ?? 'DESC';
$per_page = (int)($_GET['per_page'] ?? 20);
$per_page = in_array($per_page, [10, 20, 50, 100]) ? $per_page : 20;

// Build query
$where_conditions = [];
$params = [];

if (!empty($search)) {
    $where_conditions[] = "(a.name LIKE ? OR a.description LIKE ? OR a.location LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if (!empty($category_filter)) {
    $where_conditions[] = "a.category_id = ?";
    $params[] = $category_filter;
}

if (!empty($status_filter)) {
    $where_conditions[] = "a.status = ?";
    $params[] = $status_filter;
}

if (!empty($date_from)) {
    $where_conditions[] = "a.purchase_date >= ?";
    $params[] = $date_from;
}

if (!empty($date_to)) {
    $where_conditions[] = "a.purchase_date <= ?";
    $params[] = $date_to;
}

$where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Initialize variables
$total_assets = 0;
$total_pages = 0;
$current_page = 1;
$assets = [];
$category_totals = [];
$total_cost = 0;

// Get total count for pagination
try {
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total
        FROM assets a
        LEFT JOIN asset_categories ac ON a.category_id = ac.id
        $where_clause
    ");
    $stmt->execute($params);
    $result = $stmt->fetch();
    $total_assets = $result['total'];
    $total_pages = ceil($total_assets / $per_page);
    $current_page = max(1, min($total_pages, $_GET['page'] ?? 1));
    $offset = ($current_page - 1) * $per_page;

    // Get assets with pagination
    $stmt = $pdo->prepare("
        SELECT a.*, ac.name as category_name
        FROM assets a
        LEFT JOIN asset_categories ac ON a.category_id = ac.id
        $where_clause
        ORDER BY a.$sort_by $sort_order
        LIMIT ? OFFSET ?
    ");
    // Bind WHERE clause parameters
    foreach ($params as $i => $param) {
        $stmt->bindValue($i + 1, $param);
    }
    // Bind LIMIT and OFFSET as integers
    $stmt->bindValue(count($params) + 1, $per_page, PDO::PARAM_INT);
    $stmt->bindValue(count($params) + 2, $offset, PDO::PARAM_INT);
    $stmt->execute();
    $assets = $stmt->fetchAll();

    // Get category totals
    $stmt = $pdo->prepare("
        SELECT ac.name as category_name, SUM(a.purchase_cost) as total_cost, COUNT(*) as asset_count
        FROM assets a
        LEFT JOIN asset_categories ac ON a.category_id = ac.id
        $where_clause
        GROUP BY a.category_id, ac.name
        ORDER BY total_cost DESC
    ");
    $stmt->execute(array_slice($params, 0, -2)); // Remove LIMIT and OFFSET params
    $category_totals = $stmt->fetchAll();

    // Get total assets cost
    $stmt = $pdo->prepare("
        SELECT SUM(a.purchase_cost) as total_cost
        FROM assets a
        LEFT JOIN asset_categories ac ON a.category_id = ac.id
        $where_clause
    ");
    $stmt->execute(array_slice($params, 0, -2)); // Remove LIMIT and OFFSET params
    $result = $stmt->fetch();
    $total_cost = $result['total_cost'] ?? 0;

} catch (PDOException $e) {
    $error = "Error loading assets: " . $e->getMessage();
    $assets = [];
    $category_totals = [];
    $total_cost = 0;
    $total_assets = 0;
    $total_pages = 0;
}

// Get categories for filter
try {
    $stmt = $pdo->query("SELECT id, name FROM asset_categories ORDER BY name");
    $categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $categories = [];
}
?>

<style>
.asset-list-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 0;
    background: transparent;
    min-height: auto;
    box-shadow: none;
}

.asset-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 30px;
    border-radius: 12px;
    margin-bottom: 30px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.asset-header h1 {
    margin: 0;
    font-size: 2.2em;
    font-weight: 300;
}

.asset-header p {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.asset-content {
    display: grid;
    grid-template-columns: 1fr 320px;
    gap: 30px;
}

.asset-main {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    border: 1px solid #e1e5e9;
}

.table-container {
    overflow-x: auto;
    -webkit-overflow-scrolling: touch;
}

.asset-sidebar {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    padding: 25px;
    border: 1px solid #e1e5e9;
}

.filters-section {
    background: #f8f9fa;
    padding: 25px;
    border-bottom: 1px solid #dee2e6;
}

.filters-form {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 20px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 8px;
    color: #495057;
    font-size: 0.95em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.filter-group input,
.filter-group select {
    padding: 12px 16px;
    border: 2px solid #2E7D32;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.3s ease;
}

.filter-group input:focus,
.filter-group select:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.filter-actions {
    display: flex;
    gap: 12px;
}

.btn {
    padding: 12px 24px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(46, 125, 50, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(46, 125, 50, 0.4);
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
}

.btn-success {
    background: linear-gradient(135deg, #F9A825 0%, #FFC107 100%);
    color: #2E7D32;
    box-shadow: 0 4px 15px rgba(249, 168, 37, 0.3);
}

.btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(249, 168, 37, 0.4);
}

.btn-danger {
    background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
    color: white;
}

.btn-danger:hover {
    background: linear-gradient(135deg, #c82333 0%, #e8680d 100%);
    transform: translateY(-1px);
}

.assets-table {
    width: 100%;
    border-collapse: collapse;
}

.assets-table th,
.assets-table td {
    padding: 16px 20px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.assets-table th {
    background: #fff;
    font-weight: 700;
    color: #000;
    position: sticky;
    top: 0;
    text-transform: uppercase;
    font-size: 0.85em;
    letter-spacing: 0.5px;
}

.assets-table tr:hover {
    background: #f8f9fa;
}

.asset-name {
    font-weight: 600;
    color: #2c3e50;
    max-width: 200px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.asset-cost {
    font-weight: 700;
    color: #e74c3c;
    font-size: 1.1em;
}

.asset-category {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 0.8em;
    display: inline-block;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.asset-status {
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 0.8em;
    display: inline-block;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-active {
    background: #d4edda;
    color: #155724;
}

.status-inactive {
    background: #e2e3e5;
    color: #383d41;
}

.status-maintenance {
    background: #fff3cd;
    color: #856404;
}

.status-leased {
    background: #d1ecf1;
    color: #0c5460;
}

.status-sold {
    background: #d4edda;
    color: #155724;
}

.status-disposed {
    background: #f8d7da;
    color: #721c24;
}

.status-lost {
    background: #fff3cd;
    color: #856404;
}

.status-stolen {
    background: #f8d7da;
    color: #721c24;
}

.status-donated {
    background: #d1ecf1;
    color: #0c5460;
}

.asset-date {
    color: #6c757d;
    font-size: 0.9em;
    font-weight: 500;
}

.asset-location {
    color: #495057;
    font-size: 0.9em;
    max-width: 150px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
    font-weight: 500;
}

.actions-cell {
    white-space: nowrap;
}

.action-btn {
    padding: 6px 12px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: 0.85em;
    margin-right: 8px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    transition: all 0.3s ease;
}

.edit-btn {
    background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%);
    color: #212529;
}

.edit-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 10px rgba(255, 193, 7, 0.3);
}

.delete-btn {
    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
    color: white;
}

.delete-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 10px rgba(220, 53, 69, 0.3);
}

.retire-btn {
    background: linear-gradient(135deg, #6f42c1 0%, #5a32a3 100%);
    color: white;
}

.retire-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 10px rgba(111, 66, 193, 0.3);
}

.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 20px;
    margin-bottom: 25px;
}

.summary-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    padding: 20px;
    border-radius: 12px;
    text-align: center;
    border: 1px solid #e1e5e9;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    transition: transform 0.3s ease;
}

.summary-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
}

.summary-card h3 {
    margin: 0 0 8px 0;
    font-size: 2em;
    color: #2c3e50;
    font-weight: 700;
}

.summary-card p {
    margin: 0;
    color: #6c757d;
    font-size: 0.95em;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.pagination {
    display: flex;
    justify-content: center;
    gap: 8px;
    margin-top: 25px;
    padding-top: 25px;
    border-top: 1px solid #dee2e6;
}

.pagination a,
.pagination span {
    padding: 10px 16px;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    text-decoration: none;
    color: #2E7D32;
    font-weight: 600;
    transition: all 0.3s ease;
}

.pagination .active {
    background: #2E7D32;
    color: white;
    border-color: #2E7D32;
}

.pagination .disabled {
    color: #6c757d;
    cursor: not-allowed;
}

.message {
    padding: 15px;
    margin-bottom: 20px;
    border-radius: 8px;
    font-weight: 600;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.no-assets {
    text-align: center;
    padding: 50px 20px;
    color: #6c757d;
}

.no-assets h3 {
    margin: 0 0 10px 0;
    color: #495057;
}

.category-summary {
    margin-top: 20px;
}

.category-item {
    display: flex;
    justify-content: space-between;
    padding: 10px 0;
    border-bottom: 1px solid #e9ecef;
}

.category-name {
    font-weight: 600;
    color: #2c3e50;
}

.category-stats {
    color: #6c757d;
    font-size: 0.9em;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
    backdrop-filter: blur(5px);
}

.modal-content {
    background-color: #fefefe;
    margin: 5% auto;
    padding: 0;
    border: 1px solid #888;
    border-radius: 12px;
    width: 90%;
    max-width: 600px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.3);
    animation: modalFadeIn 0.3s ease-out;
}

@keyframes modalFadeIn {
    from { opacity: 0; transform: translateY(-50px); }
    to { opacity: 1; transform: translateY(0); }
}

.modal-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 20px 30px;
    border-radius: 12px 12px 0 0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-header h2 {
    margin: 0;
    font-size: 1.5em;
    font-weight: 600;
}

.close {
    color: #aaa;
    float: right;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
    transition: color 0.3s ease;
}

.close:hover,
.close:focus {
    color: white;
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
    margin-bottom: 20px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 8px;
    color: #495057;
    font-size: 0.95em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.form-group input,
.form-group select,
.form-group textarea {
    padding: 12px 16px;
    border: 2px solid #2E7D32;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.3s ease;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.form-group textarea {
    resize: vertical;
    min-height: 80px;
}

.form-actions {
    display: flex;
    justify-content: flex-end;
    gap: 12px;
    padding: 20px 30px;
    border-top: 1px solid #dee2e6;
    background: #f8f9fa;
    border-radius: 0 0 12px 12px;
}
</style>

<div class="asset-list-container">
    <div class="asset-header">
        <div>
            <h1>🏢 Asset Management</h1>
            <p>View, filter, and manage all school assets</p>
        </div>
        <div>
            <a href="add_asset.php" class="btn btn-success">➕ Add New Asset</a>
            <a href="index.php" class="btn btn-secondary">🏠 Dashboard</a>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="asset-content">
        <div class="asset-main">
            <!-- Summary Cards -->
            <div class="summary-cards">
                <div class="summary-card">
                    <h3>GHC <?php echo number_format($total_cost, 2); ?></h3>
                    <p>Total Asset Value</p>
                </div>
                <div class="summary-card">
                    <h3><?php echo $current_page; ?></h3>
                    <p>Current Page</p>
                </div>
                <div class="summary-card">
                    <h3><?php echo $total_assets; ?></h3>
                    <p>Total Assets</p>
                </div>
                <div class="summary-card">
                    <h3><?php echo $total_pages; ?></h3>
                    <p>Total Pages</p>
                </div>
            </div>

            <!-- Filters -->
            <div class="filters-section">
                <form method="get" class="filters-form">
                    <div class="filter-group">
                        <label for="search">Search</label>
                        <input type="text" name="search" id="search" value="<?php echo htmlspecialchars($search); ?>"
                               placeholder="Name, description, location...">
                    </div>

                    <div class="filter-group">
                        <label for="category">Category</label>
                        <select name="category" id="category">
                            <option value="">All Categories</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo $category['id']; ?>"
                                        <?php echo $category_filter == $category['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($category['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="status">Status</label>
                        <select name="status" id="status">
                            <option value="">All Statuses</option>
                            <option value="active" <?php echo $status_filter == 'active' ? 'selected' : ''; ?>>Active</option>
                            <option value="inactive" <?php echo $status_filter == 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                            <option value="maintenance" <?php echo $status_filter == 'maintenance' ? 'selected' : ''; ?>>Maintenance</option>
                            <option value="leased" <?php echo $status_filter == 'leased' ? 'selected' : ''; ?>>Leased</option>
                            <option value="sold" <?php echo $status_filter == 'sold' ? 'selected' : ''; ?>>Sold</option>
                            <option value="disposed" <?php echo $status_filter == 'disposed' ? 'selected' : ''; ?>>Disposed</option>
                            <option value="lost" <?php echo $status_filter == 'lost' ? 'selected' : ''; ?>>Lost</option>
                            <option value="stolen" <?php echo $status_filter == 'stolen' ? 'selected' : ''; ?>>Stolen</option>
                            <option value="donated" <?php echo $status_filter == 'donated' ? 'selected' : ''; ?>>Donated</option>
                            <option value="retired" <?php echo $status_filter == 'retired' ? 'selected' : ''; ?>>Retired</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="date_from">Purchase From</label>
                        <input type="date" name="date_from" id="date_from" value="<?php echo htmlspecialchars($date_from); ?>">
                    </div>

                    <div class="filter-group">
                        <label for="date_to">Purchase To</label>
                        <input type="date" name="date_to" id="date_to" value="<?php echo htmlspecialchars($date_to); ?>">
                    </div>

                    <div class="filter-group">
                        <label for="per_page">Items per page</label>
                        <select name="per_page" id="per_page">
                            <option value="10" <?php echo $per_page == 10 ? 'selected' : ''; ?>>10</option>
                            <option value="20" <?php echo $per_page == 20 ? 'selected' : ''; ?>>20</option>
                            <option value="50" <?php echo $per_page == 50 ? 'selected' : ''; ?>>50</option>
                            <option value="100" <?php echo $per_page == 100 ? 'selected' : ''; ?>>100</option>
                        </select>
                    </div>

                    <div class="filter-actions">
                        <button type="submit" class="btn btn-primary">🔍 Filter</button>
                        <a href="manage_assets.php" class="btn btn-secondary">🔄 Reset</a>
                    </div>
                </form>
            </div>

            <!-- Assets Table -->
            <?php if (empty($assets)): ?>
                <div class="no-assets">
                    <h3>No assets found</h3>
                    <p>Try adjusting your filters or <a href="add_asset.php">record a new asset</a>.</p>
                </div>
            <?php else: ?>
                <div class="table-container">
                    <table class="assets-table">
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th>Purchase Cost</th>
                                <th>Current Value</th>
                                <th>Category</th>
                                <th>Status</th>
                                <th>Purchase Date</th>
                                <th>Location</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($assets as $asset): ?>
                                <tr>
                                    <td>
                                        <div class="asset-name" title="<?php echo htmlspecialchars($asset['name']); ?>">
                                            <?php echo htmlspecialchars($asset['name']); ?>
                                        </div>
                                        <?php if (!empty($asset['description'])): ?>
                                            <div style="font-size: 0.8em; color: #6c757d; margin-top: 2px;">
                                                <?php echo htmlspecialchars(substr($asset['description'], 0, 50)) . (strlen($asset['description']) > 50 ? '...' : ''); ?>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td class="asset-cost">
                                        GHC <?php echo number_format($asset['purchase_cost'], 2); ?>
                                    </td>
                                    <td class="asset-cost">
                                        GHC <?php echo number_format($asset['current_value'] ?? $asset['purchase_cost'], 2); ?>
                                    </td>
                                    <td>
                                        <span class="asset-category">
                                            <?php echo htmlspecialchars($asset['category_name'] ?? 'Uncategorized'); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="asset-status status-<?php echo $asset['status']; ?>">
                                            <?php echo ucfirst($asset['status']); ?>
                                        </span>
                                    </td>
                                    <td class="asset-date">
                                        <?php echo date('M d, Y', strtotime($asset['purchase_date'])); ?>
                                    </td>
                                    <td>
                                        <div class="asset-location" title="<?php echo htmlspecialchars($asset['location'] ?? ''); ?>">
                                            <?php echo htmlspecialchars($asset['location'] ?? 'N/A'); ?>
                                        </div>
                                    </td>
                                    <td class="actions-cell">
                                        <button type="button" class="action-btn edit-btn" onclick="openEditModal(<?php echo $asset['id']; ?>)">
                                            ✏️ Edit
                                        </button>
                                        <button type="button" class="action-btn retire-btn" onclick="openRetireModal(<?php echo $asset['id']; ?>, '<?php echo htmlspecialchars($asset['name']); ?>')">
                                            🏛️ Retire
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <div class="pagination">
                        <?php if ($current_page > 1): ?>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $current_page - 1])); ?>">Previous</a>
                        <?php else: ?>
                            <span class="disabled">Previous</span>
                        <?php endif; ?>

                        <?php for ($i = max(1, $current_page - 2); $i <= min($total_pages, $current_page + 2); $i++): ?>
                            <?php if ($i == $current_page): ?>
                                <span class="active"><?php echo $i; ?></span>
                            <?php else: ?>
                                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                            <?php endif; ?>
                        <?php endfor; ?>

                        <?php if ($current_page < $total_pages): ?>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $current_page + 1])); ?>">Next</a>
                        <?php else: ?>
                            <span class="disabled">Next</span>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>

        <!-- Sidebar with Category Summary -->
        <div class="asset-sidebar">
            <h3 style="margin-top: 0; color: #2c3e50; border-bottom: 2px solid #2E7D32; padding-bottom: 10px;">📊 Category Summary</h3>
            <div class="category-summary">
                <?php if (empty($category_totals)): ?>
                    <p>No category data available.</p>
                <?php else: ?>
                    <?php foreach ($category_totals as $cat): ?>
                        <div class="category-item">
                            <span class="category-name"><?php echo htmlspecialchars($cat['category_name']); ?></span>
                            <span class="category-stats">
                                <?php echo $cat['asset_count']; ?> assets • GHC <?php echo number_format($cat['total_cost'], 2); ?>
                            </span>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Edit Asset Modal -->
<div id="editAssetModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h2>✏️ Edit Asset</h2>
            <span class="close" onclick="closeEditModal()">&times;</span>
        </div>
        <form id="editAssetForm" method="post">
            <input type="hidden" name="asset_id" id="edit_asset_id">
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_name">Asset Name *</label>
                    <input type="text" id="edit_name" name="name" required>
                </div>
                <div class="form-group">
                    <label for="edit_asset_type">Asset Type *</label>
                    <select id="edit_asset_type" name="asset_type" required>
                        <option value="equipment">Equipment</option>
                        <option value="furniture">Furniture</option>
                        <option value="vehicle">Vehicle</option>
                        <option value="building">Building</option>
                        <option value="land">Land</option>
                        <option value="other">Other</option>
                    </select>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_purchase_cost">Purchase Cost (GHC) *</label>
                    <input type="number" id="edit_purchase_cost" name="purchase_cost" step="0.01" required>
                </div>
                <div class="form-group">
                    <label for="edit_current_value">Current Value (GHC)</label>
                    <input type="number" id="edit_current_value" name="current_value" step="0.01">
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_purchase_date">Purchase Date *</label>
                    <input type="date" id="edit_purchase_date" name="purchase_date" required>
                </div>
                <div class="form-group">
                    <label for="edit_category_id">Category *</label>
                    <select id="edit_category_id" name="category_id" required>
                        <?php foreach ($categories as $category): ?>
                            <option value="<?php echo $category['id']; ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_status">Status *</label>
                    <select id="edit_status" name="status" required>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                        <option value="maintenance">Maintenance</option>
                        <option value="leased">Leased</option>
                        <option value="sold">Sold</option>
                        <option value="disposed">Disposed</option>
                        <option value="lost">Lost</option>
                        <option value="stolen">Stolen</option>
                        <option value="donated">Donated</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="edit_serial_number">Serial Number</label>
                    <input type="text" id="edit_serial_number" name="serial_number">
                </div>
            </div>
            <div class="form-group">
                <label for="edit_location">Location</label>
                <input type="text" id="edit_location" name="location">
            </div>
            <div class="form-group">
                <label for="edit_description">Description</label>
                <textarea id="edit_description" name="description" rows="3"></textarea>
            </div>
            <div class="form-actions">
                <button type="submit" class="btn btn-primary">💾 Update Asset</button>
                <button type="button" class="btn btn-secondary" onclick="closeEditModal()">❌ Cancel</button>
            </div>
        </form>
    </div>
</div>

<!-- Delete Asset Modal -->
<div id="deleteAssetModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h2>🗑️ Delete Asset</h2>
            <span class="close" onclick="closeDeleteModal()">&times;</span>
        </div>
        <div style="padding: 30px;">
            <p style="margin: 0 0 20px 0; font-size: 16px; color: #495057;">
                Are you sure you want to delete the asset "<span id="delete_asset_name" style="font-weight: 600;"></span>"?
            </p>
            <p style="margin: 0 0 20px 0; font-size: 14px; color: #6c757d;">
                This action cannot be undone. The asset will be permanently removed from the system.
            </p>
            <div class="form-actions">
                <button type="button" class="btn btn-danger" onclick="confirmDeleteAsset()">🗑️ Delete Asset</button>
                <button type="button" class="btn btn-secondary" onclick="closeDeleteModal()">❌ Cancel</button>
            </div>
        </div>
    </div>
</div>

<script>
function openEditModal(assetId) {
    // Fetch asset data
    fetch(`api/get_asset.php?id=${assetId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const asset = data.asset;
                document.getElementById('edit_asset_id').value = asset.id;
                document.getElementById('edit_name').value = asset.name;
                document.getElementById('edit_asset_type').value = asset.asset_type;
                document.getElementById('edit_purchase_cost').value = asset.purchase_cost;
                document.getElementById('edit_current_value').value = asset.current_value || '';
                document.getElementById('edit_purchase_date').value = asset.purchase_date;
                document.getElementById('edit_category_id').value = asset.category_id;
                document.getElementById('edit_status').value = asset.status;
                document.getElementById('edit_serial_number').value = asset.serial_number || '';
                document.getElementById('edit_location').value = asset.location || '';
                document.getElementById('edit_description').value = asset.description || '';
                document.getElementById('editAssetModal').style.display = 'block';
            } else {
                alert('Error loading asset data: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading asset data');
        });
}

function closeEditModal() {
    document.getElementById('editAssetModal').style.display = 'none';
}

// Handle form submission
document.getElementById('editAssetForm').addEventListener('submit', function(e) {
    e.preventDefault();

    const formData = new FormData(this);

    fetch('api/update_asset.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Asset updated successfully!');
            closeEditModal();
            location.reload(); // Refresh the page to show updated data
        } else {
            alert('Error updating asset: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error updating asset');
    });
});

// Delete Asset Modal Functions
function openDeleteModal(assetId, assetName) {
    document.getElementById('delete_asset_name').textContent = assetName;
    document.getElementById('deleteAssetModal').style.display = 'block';
    // Store asset ID for deletion
    document.getElementById('deleteAssetModal').setAttribute('data-asset-id', assetId);
}

function closeDeleteModal() {
    document.getElementById('deleteAssetModal').style.display = 'none';
}

function confirmDeleteAsset() {
    const assetId = document.getElementById('deleteAssetModal').getAttribute('data-asset-id');
    if (confirm('Are you sure you want to delete this asset? This action cannot be undone.')) {
        // Send delete request
        fetch('api/delete_asset.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ asset_id: assetId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Asset deleted successfully!');
                closeDeleteModal();
                location.reload(); // Refresh the page to show updated data
            } else {
                alert('Error deleting asset: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error deleting asset');
        });
    }
}

// Close modal when clicking outside
window.onclick = function(event) {
    const editModal = document.getElementById('editAssetModal');
    const deleteModal = document.getElementById('deleteAssetModal');
    if (event.target == editModal) {
        closeEditModal();
    }
    if (event.target == deleteModal) {
        closeDeleteModal();
    }
}
</script>

<?php include 'includes/accounting_footer.php'; ?>
