<?php
require_once '../config.php';
require_once 'includes/enhanced_access_control_fixed.php';

// Require permission to access this page - only accounts users can manage accounting users
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'accountant') {
    header('Location: ../login.php');
    exit;
}

$message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_accounting_user'])) {
        $username = trim($_POST['username']);
        $password = $_POST['password'];
        $confirm_password = $_POST['confirm_password'];
        $first_name = trim($_POST['first_name']);
        $last_name = trim($_POST['last_name']);
        $email = trim($_POST['email']);
        $phone = trim($_POST['phone']);

        $errors = [];

        if (empty($username) || empty($password) || empty($confirm_password) || empty($first_name) || empty($last_name)) {
            $errors[] = "Username, password, first name, and last name are required.";
        }

        if ($password !== $confirm_password) {
            $errors[] = "Passwords do not match.";
        }

        // Check if username already exists
        $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
        $stmt->execute([$username]);
        if ($stmt->fetch()) {
            $errors[] = "Username already exists.";
        }

        if (empty($errors)) {
            // Insert into users table
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            $pdo->beginTransaction();
            try {
                $stmt = $pdo->prepare("INSERT INTO users (username, password, role, created_at) VALUES (?, ?, 'accountant', NOW())");
                $stmt->execute([$username, $hashed_password]);
                $user_id = $pdo->lastInsertId();

                // Insert into accounting_users table
                $stmt = $pdo->prepare("INSERT INTO accounting_users (user_id, first_name, last_name, email, phone, is_active, role) VALUES (?, ?, ?, ?, ?, 1, ?)");
                $stmt->execute([$user_id, $first_name, $last_name, $email, $phone, $_POST['role'] ?? 'Accountant']);

                $pdo->commit();
                $message = 'Accounting user added successfully.';
            } catch (Exception $e) {
                $pdo->rollBack();
                $message = 'Error adding accounting user: ' . $e->getMessage();
            }
        } else {
            $message = implode('<br>', $errors);
        }
    } elseif (isset($_POST['edit_accounting_user'])) {
        $accounting_user_id = $_POST['accounting_user_id'];
        $first_name = trim($_POST['first_name']);
        $last_name = trim($_POST['last_name']);
        $email = trim($_POST['email']);
        $phone = trim($_POST['phone']);
        $role = trim($_POST['role']);
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        $stmt = $pdo->prepare("UPDATE accounting_users SET first_name = ?, last_name = ?, email = ?, phone = ?, role = ?, is_active = ?, updated_at = NOW() WHERE id = ?");
        $stmt->execute([$first_name, $last_name, $email, $phone, $role, $is_active, $accounting_user_id]);

        $message = 'Accounting user updated successfully.';
    } elseif (isset($_POST['delete_accounting_user'])) {
        $accounting_user_id = $_POST['accounting_user_id'];

        // Get user_id first
        $stmt = $pdo->prepare("SELECT user_id FROM accounting_users WHERE id = ?");
        $stmt->execute([$accounting_user_id]);
        $accounting_user = $stmt->fetch();

        if ($accounting_user) {
            // Delete accounting_user record (cascade will delete user)
            $stmt = $pdo->prepare("DELETE FROM accounting_users WHERE id = ?");
            $stmt->execute([$accounting_user_id]);

            $message = 'Accounting user deleted successfully.';
        }
    }
}

// Get all accounting users
$stmt = $pdo->query("
    SELECT au.*, u.username
    FROM accounting_users au
    JOIN users u ON au.user_id = u.id
    ORDER BY au.created_at DESC
");
$accounting_users = $stmt->fetchAll();

// Get accounting user for edit
$edit_accounting_user = null;
if (isset($_GET['edit'])) {
    $accounting_user_id = $_GET['edit'];
    $stmt = $pdo->prepare("
        SELECT au.*, u.username
        FROM accounting_users au
        JOIN users u ON au.user_id = u.id
        WHERE au.id = ?
    ");
    $stmt->execute([$accounting_user_id]);
    $edit_accounting_user = $stmt->fetch();
}

include 'includes/accounting_header.php';
?>

<style>
.form-row {
    display: flex;
    gap: 20px;
    margin-bottom: 15px;
}
.form-row .form-group {
    flex: 1;
}
.form-row .form-group:last-child {
    margin-bottom: 0;
}
fieldset {
    margin-bottom: 20px;
    border: 1px solid #ddd;
    border-radius: 5px;
    padding: 15px;
}
legend {
    font-weight: bold;
    color: #333;
}

.accounting-users-table {
    background: white;
    border-radius: 8px;
    overflow: hidden;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.accounting-users-table table {
    width: 100%;
    border-collapse: collapse;
}

.accounting-users-table th,
.accounting-users-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.accounting-users-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.accounting-users-table tr:hover {
    background: #f8f9fa;
}

.status-active {
    color: #28a745;
    font-weight: bold;
}

.status-inactive {
    color: #dc3545;
    font-weight: bold;
}
</style>

<h2>Manage Accounting Users</h2>

<?php if ($message): ?>
    <div class="message"><?php echo $message; ?></div>
<?php endif; ?>

<!-- Button to trigger add modal -->
<button type="button" class="btn btn-primary" data-toggle="modal" data-target="#addAccountingUserModal">Add New Accounting User</button>

<!-- Add Accounting User Modal -->
<div class="modal fade" id="addAccountingUserModal" tabindex="-1" role="dialog" aria-labelledby="addAccountingUserModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addAccountingUserModalLabel">Add New Accounting User</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form method="post" action="manage_accounting_users.php" id="addAccountingUserForm">
                    <fieldset>
                        <legend>Login Credentials</legend>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="username">👤 Username:</label>
                                <input type="text" id="username" name="username" required>
                            </div>
                            <div class="form-group">
                                <label for="password">🔒 Password:</label>
                                <input type="password" id="password" name="password" required>
                            </div>
                            <div class="form-group">
                                <label for="confirm_password">🔒 Confirm Password:</label>
                                <input type="password" id="confirm_password" name="confirm_password" required>
                            </div>
                        </div>
                    </fieldset>

                    <fieldset>
                        <legend>Personal Information</legend>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="first_name">👤 First Name:</label>
                                <input type="text" id="first_name" name="first_name" required>
                            </div>
                            <div class="form-group">
                                <label for="last_name">👤 Last Name:</label>
                                <input type="text" id="last_name" name="last_name" required>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="email">📧 Email:</label>
                                <input type="email" id="email" name="email">
                            </div>
                            <div class="form-group">
                                <label for="phone">📞 Phone:</label>
                                <input type="text" id="phone" name="phone">
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="role">🎭 Role:</label>
                                <select id="role" name="role" required>
                                    <option value="System Administrator">System Administrator</option>
                                    <option value="Accountant" selected>Accountant</option>
                                    <option value="Accounts Clerk">Accounts Clerk</option>
                                    <option value="Payroll Officer">Payroll Officer</option>
                                    <option value="Procurement Officer">Procurement Officer</option>
                                    <option value="Approver">Approver</option>
                                    <option value="Auditor">Auditor</option>
                                    <option value="Headmaster">Headmaster</option>
                                    <option value="Cashier">Cashier</option>
                                    <option value="Inventory Officer">Inventory Officer</option>
                                </select>
                            </div>
                        </div>
                    </fieldset>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="submit" form="addAccountingUserForm" name="add_accounting_user" class="btn btn-primary">Add Accounting User</button>
            </div>
        </div>
    </div>
</div>

<?php if ($edit_accounting_user): ?>
<h3>Edit Accounting User</h3>

<form method="post" action="manage_accounting_users.php">
    <input type="hidden" name="accounting_user_id" value="<?php echo $edit_accounting_user['id']; ?>">

    <fieldset>
        <legend>Login Credentials</legend>
        <div class="form-row">
            <div class="form-group">
                <label for="username">👤 Username:</label>
                <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($edit_accounting_user['username'] ?? ''); ?>" readonly>
                <small class="text-muted">Username cannot be changed</small>
            </div>
        </div>
    </fieldset>

    <fieldset>
        <legend>Personal Information</legend>
        <div class="form-row">
            <div class="form-group">
                <label for="first_name">👤 First Name:</label>
                <input type="text" id="first_name" name="first_name" value="<?php echo htmlspecialchars($edit_accounting_user['first_name'] ?? ''); ?>" required>
            </div>
            <div class="form-group">
                <label for="last_name">👤 Last Name:</label>
                <input type="text" id="last_name" name="last_name" value="<?php echo htmlspecialchars($edit_accounting_user['last_name'] ?? ''); ?>" required>
            </div>
        </div>
        <div class="form-row">
            <div class="form-group">
                <label for="email">📧 Email:</label>
                <input type="email" id="email" name="email" value="<?php echo htmlspecialchars($edit_accounting_user['email'] ?? ''); ?>">
            </div>
            <div class="form-group">
                <label for="phone">📞 Phone:</label>
                <input type="text" id="phone" name="phone" value="<?php echo htmlspecialchars($edit_accounting_user['phone'] ?? ''); ?>">
            </div>
        </div>
        <div class="form-row">
            <div class="form-group">
                <label for="role">🎭 Role:</label>
                <select id="role" name="role" required>
                    <option value="System Administrator" <?php echo ($edit_accounting_user['role'] ?? '') === 'System Administrator' ? 'selected' : ''; ?>>System Administrator</option>
                    <option value="Accountant" <?php echo ($edit_accounting_user['role'] ?? '') === 'Accountant' ? 'selected' : ''; ?>>Accountant</option>
                    <option value="Accounts Clerk" <?php echo ($edit_accounting_user['role'] ?? '') === 'Accounts Clerk' ? 'selected' : ''; ?>>Accounts Clerk</option>
                    <option value="Payroll Officer" <?php echo ($edit_accounting_user['role'] ?? '') === 'Payroll Officer' ? 'selected' : ''; ?>>Payroll Officer</option>
                    <option value="Procurement Officer" <?php echo ($edit_accounting_user['role'] ?? '') === 'Procurement Officer' ? 'selected' : ''; ?>>Procurement Officer</option>
                    <option value="Approver" <?php echo ($edit_accounting_user['role'] ?? '') === 'Approver' ? 'selected' : ''; ?>>Approver</option>
                    <option value="Auditor" <?php echo ($edit_accounting_user['role'] ?? '') === 'Auditor' ? 'selected' : ''; ?>>Auditor</option>
                    <option value="Headmaster" <?php echo ($edit_accounting_user['role'] ?? '') === 'Headmaster' ? 'selected' : ''; ?>>Headmaster</option>
                    <option value="Cashier" <?php echo ($edit_accounting_user['role'] ?? '') === 'Cashier' ? 'selected' : ''; ?>>Cashier</option>
                    <option value="Inventory Officer" <?php echo ($edit_accounting_user['role'] ?? '') === 'Inventory Officer' ? 'selected' : ''; ?>>Inventory Officer</option>
                </select>
            </div>
            <div class="form-group">
                <label for="is_active">
                    <input type="checkbox" id="is_active" name="is_active" <?php echo ($edit_accounting_user['is_active'] ?? 0) ? 'checked' : ''; ?>> Active Account
                </label>
            </div>
        </div>
    </fieldset>

    <button type="submit" name="edit_accounting_user" class="btn btn-primary">Update Accounting User</button>
    <a href="manage_accounting_users.php" class="btn">Cancel</a>
</form>
<?php endif; ?>

<h3>All Accounting Users</h3>

<table class="accounting-users-table">
    <thead>
        <tr>
            <th>Username</th>
            <th>Name</th>
            <th>Email</th>
            <th>Phone</th>
            <th>Status</th>
            <th>Created</th>
            <th>Actions</th>
        </tr>
    </thead>
    <tbody>
        <?php foreach ($accounting_users as $accounting_user): ?>
            <tr>
                <td><?php echo htmlspecialchars($accounting_user['username']); ?></td>
                <td><?php echo htmlspecialchars(trim(($accounting_user['first_name'] ?? '') . ' ' . ($accounting_user['last_name'] ?? ''))); ?></td>
                <td><?php echo htmlspecialchars($accounting_user['email'] ?? ''); ?></td>
                <td><?php echo htmlspecialchars($accounting_user['phone'] ?? ''); ?></td>
                <td>
                    <span class="status-<?php echo ($accounting_user['is_active'] ?? 0) ? 'active' : 'inactive'; ?>">
                        <?php echo ($accounting_user['is_active'] ?? 0) ? 'Active' : 'Inactive'; ?>
                    </span>
                </td>
                <td><?php echo htmlspecialchars($accounting_user['created_at'] ?? ''); ?></td>
                <td>
                    <a href="?edit=<?php echo $accounting_user['id']; ?>" class="btn btn-small" style="background-color: #28a745; color: white;">Edit</a>
                    <form method="post" action="manage_accounting_users.php" style="display:inline;">
                        <input type="hidden" name="accounting_user_id" value="<?php echo $accounting_user['id']; ?>">
                        <button type="submit" name="delete_accounting_user" class="btn btn-small btn-danger" onclick="return confirm('Are you sure you want to delete this accounting user?')">Delete</button>
                    </form>
                </td>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<?php
include 'includes/accounting_footer.php';
?>
