<?php
/**
 * Loan accounting functions for proper accrual-based accounting
 */

require_once 'payment_methods.php';
require_once 'accounting_functions.php';

// Include logger with fallback
$loggerPath = '../src/logger.php';
if (file_exists($loggerPath)) {
    require_once $loggerPath;
} else {
    // Fallback logger class if file doesn't exist
    if (!class_exists('Logger')) {
        class Logger {
            public function info($message) {}
            public function error($message) {}
            public function warning($message) {}
        }
    }

    if (!function_exists('setup_logger')) {
        function setup_logger() {
            return new Logger();
        }
    }
}

/**
 * Create journal entry for loan/liability creation (accrual basis)
 * Debit: Cash/Bank Account (increases asset)
 * Credit: Loans Payable (increases liability)
 */
function createLoanEntry($pdo, $loanId, $amount, $date, $createdBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get loan details
        $stmt = $pdo->prepare("SELECT name, start_date, end_date FROM liabilities WHERE id = ?");
        $stmt->execute([$loanId]);
        $loan = $stmt->fetch();

        if (!$loan) {
            throw new Exception('Loan not found');
        }

        // Calculate loan duration in months
        $startDate = new DateTime($loan['start_date']);
        $endDate = new DateTime($loan['end_date']);
        $interval = $startDate->diff($endDate);
        $months = $interval->y * 12 + $interval->m;

        // Determine account based on duration
        $loansPayableAccount = ($months > 12) ? 'Long-Term Loans' : 'Short-Term Loans';
        $loansPayableId = getChartAccountId($pdo, $loansPayableAccount);
        if (!$loansPayableId) {
            throw new Exception("{$loansPayableAccount} account not found");
        }

        // Get Bank Account(s) account (default for loans)
        $cashAccountId = getChartAccountId($pdo, 'Bank Account(s)');
        if (!$cashAccountId) {
            throw new Exception('Bank Account(s) account not found');
        }

        $entryNumber = generateEntryNumber($pdo, $date);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'loan', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $date,
            "Loan/liability created: {$loan['name']} (Accrual basis)",
            $loanId,
            $createdBy
        ]);
        $entryId = $pdo->lastInsertId();

        // Debit: Cash at Bank (increases asset)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, debit_amount, line_number, description)
            VALUES (?, ?, ?, 1, ?)
        ");
        $stmt->execute([
            $entryId,
            $cashAccountId,
            $amount,
            "Cash received for loan: {$loan['name']}"
        ]);

        // Credit: Loans Payable (increases liability)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, 2, ?)
        ");
        $stmt->execute([
            $entryId,
            $loansPayableId,
            $amount,
            "Loans payable: {$loan['name']}"
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$amount, $cashAccountId]); // Debit increases asset

        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$amount, $loansPayableId]); // Credit increases liability

        if ($manageTransaction) {
            $pdo->commit();
        }

        $logger = setup_logger();
        $logger->info("Loan journal entry created: {$entryNumber} for loan ID {$loanId}");

        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}

/**
 * Create journal entry for loan payment (cash basis)
 * Debit: Loans Payable (decreases liability)
 * Credit: Cash Account (decreases asset)
 */
function createLoanPaymentEntry($pdo, $loanId, $paymentAmount, $paymentMethod, $date, $paidBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get loan details
        $stmt = $pdo->prepare("SELECT name, start_date, end_date FROM liabilities WHERE id = ?");
        $stmt->execute([$loanId]);
        $loan = $stmt->fetch();

        if (!$loan) {
            throw new Exception('Loan not found');
        }

        // Calculate loan duration in months
        $startDate = new DateTime($loan['start_date']);
        $endDate = new DateTime($loan['end_date']);
        $interval = $startDate->diff($endDate);
        $months = $interval->y * 12 + $interval->m;

        // Determine account based on duration
        $loansPayableAccount = ($months > 12) ? 'Long-Term Loans' : 'Short-Term Loans';
        $loansPayableId = getChartAccountId($pdo, $loansPayableAccount);
        if (!$loansPayableId) {
            throw new Exception("{$loansPayableAccount} account not found");
        }

        // Get cash account based on payment method
        $cashAccountName = PaymentMethods::getCashAccountForPaymentMethod($paymentMethod);
        $cashAccountId = getChartAccountId($pdo, $cashAccountName);
        if (!$cashAccountId) {
            throw new Exception("Cash account '{$cashAccountName}' not found");
        }

        $entryNumber = generateEntryNumber($pdo, $date);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'loan_payment', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $date,
            "Loan payment: {$loan['name']} via {$paymentMethod} (Cash basis)",
            $loanId,
            $paidBy
        ]);
        $entryId = $pdo->lastInsertId();

        // Debit: Loans Payable (decreases liability)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, debit_amount, line_number, description)
            VALUES (?, ?, ?, 1, ?)
        ");
        $stmt->execute([
            $entryId,
            $loansPayableId,
            $paymentAmount,
            "Loans payable payment: {$loan['name']}"
        ]);

        // Credit: Cash Account (decreases asset)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, 2, ?)
        ");
        $stmt->execute([
            $entryId,
            $cashAccountId,
            $paymentAmount,
            "Cash payment for loan: {$loan['name']} ({$cashAccountName})"
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
        $stmt->execute([$paymentAmount, $loansPayableId]); // Debit decreases liability

        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
        $stmt->execute([$paymentAmount, $cashAccountId]); // Credit decreases asset

        if ($manageTransaction) {
            $pdo->commit();
        }

        $logger = setup_logger();
        $logger->info("Loan payment journal entry created: {$entryNumber} for loan ID {$loanId}");

        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}
?>
