<?php
require_once '../config.php';

$message = '';
$error = '';

// Get filter parameters
$as_of_date = $_GET['as_of_date'] ?? date('Y-m-t');
$account_type = $_GET['account_type'] ?? 'all';

// Calculate date range for balance calculation (all transactions up to as_of_date)
try {
    // Get all account balances as of the specified date
    $query = "
        SELECT
            coa.id,
            coa.account_code,
            coa.account_name,
            coa.account_type,
            COALESCE(SUM(jel.debit_amount - jel.credit_amount), 0) as balance
        FROM chart_of_accounts coa
        LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
        LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.transaction_date <= ? AND je.status = 'posted'
        WHERE 1=1
    ";

    $params = [$as_of_date];

    if ($account_type !== 'all') {
        $query .= " AND coa.account_type = ?";
        $params[] = $account_type;
    }

    $query .= " GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type ORDER BY coa.account_code ASC";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $account_balances = $stmt->fetchAll();

    // Separate debit and credit balances
    $trial_balance_entries = [];
    $total_debits = 0;
    $total_credits = 0;

    foreach ($account_balances as $account) {
        $balance = $account['balance'];

        // Determine if this is a debit or credit balance based on account type
        $is_debit_balance = in_array($account['account_type'], ['asset', 'expense']);

        if ($is_debit_balance) {
            $debit_balance = abs($balance);
            $credit_balance = 0;
        } else {
            $debit_balance = 0;
            $credit_balance = abs($balance);
        }

        // If balance is negative, it means the opposite side
        if ($balance < 0) {
            if ($is_debit_balance) {
                $debit_balance = 0;
                $credit_balance = abs($balance);
            } else {
                $debit_balance = abs($balance);
                $credit_balance = 0;
            }
        }

        $trial_balance_entries[] = [
            'account_code' => $account['account_code'],
            'account_name' => $account['account_name'],
            'account_type' => $account['account_type'],
            'debit_balance' => $debit_balance,
            'credit_balance' => $credit_balance,
            'net_balance' => $balance
        ];

        $total_debits += $debit_balance;
        $total_credits += $credit_balance;
    }

    // Check if trial balance is balanced
    $is_balanced = abs($total_debits - $total_credits) < 0.01; // Allow for small rounding differences

} catch (PDOException $e) {
    $error = "Error fetching trial balance data: " . $e->getMessage();
    $trial_balance_entries = [];
    $total_debits = $total_credits = 0;
    $is_balanced = false;
}

include 'includes/accounting_header.php';
?>

<div class="trial-balance-container">
    <div class="page-header">
        <h1>⚖️ Trial Balance Report</h1>
        <p>Summary of all account balances as of <?php echo date('M d, Y', strtotime($as_of_date)); ?></p>
    </div>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Filters -->
    <div class="filters-section">
        <form method="get" class="filters-form">
            <div class="filter-row">
                <div class="filter-group">
                    <label for="as_of_date">As of Date:</label>
                    <input type="date" name="as_of_date" id="as_of_date" value="<?php echo $as_of_date; ?>" onchange="this.form.submit()">
                </div>

                <div class="filter-group">
                    <label for="account_type">Account Type:</label>
                    <select name="account_type" id="account_type" onchange="this.form.submit()">
                        <option value="all" <?php echo $account_type === 'all' ? 'selected' : ''; ?>>All Types</option>
                        <option value="asset" <?php echo $account_type === 'asset' ? 'selected' : ''; ?>>Assets</option>
                        <option value="liability" <?php echo $account_type === 'liability' ? 'selected' : ''; ?>>Liabilities</option>
                        <option value="equity" <?php echo $account_type === 'equity' ? 'selected' : ''; ?>>Equity</option>
                        <option value="revenue" <?php echo $account_type === 'revenue' ? 'selected' : ''; ?>>Revenue</option>
                        <option value="expense" <?php echo $account_type === 'expense' ? 'selected' : ''; ?>>Expenses</option>
                    </select>
                </div>

                <div class="filter-group">
                    <a href="trial_balance_report.php" class="btn btn-secondary">Reset</a>
                </div>
            </div>
        </form>
    </div>

    <!-- Balance Status -->
    <div class="balance-status">
        <div class="status-card <?php echo $is_balanced ? 'balanced' : 'unbalanced'; ?>">
            <div class="status-icon">
                <?php echo $is_balanced ? '✅' : '❌'; ?>
            </div>
            <div class="status-content">
                <h3><?php echo $is_balanced ? 'Trial Balance is Balanced' : 'Trial Balance is Unbalanced'; ?></h3>
                <p>
                    <?php if ($is_balanced): ?>
                        Total Debits (GHC <?php echo number_format($total_debits, 2); ?>) equal Total Credits (GHC <?php echo number_format($total_credits, 2); ?>)
                    <?php else: ?>
                        Total Debits: GHC <?php echo number_format($total_debits, 2); ?> | Total Credits: GHC <?php echo number_format($total_credits, 2); ?>
                        <br><small>Difference: GHC <?php echo number_format(abs($total_debits - $total_credits), 2); ?></small>
                    <?php endif; ?>
                </p>
            </div>
        </div>
    </div>

    <!-- Trial Balance Table -->
    <?php if (empty($trial_balance_entries)): ?>
        <div class="no-entries">
            <p>No accounts found for the selected criteria.</p>
        </div>
    <?php else: ?>
        <div class="trial-balance-table-container">
            <div class="table-header">
                <h3>Trial Balance</h3>
                <div class="table-actions">
                    <button onclick="window.print()" class="btn btn-secondary">Print</button>
                    <a href="export_trial_balance.php?<?php echo http_build_query($_GET); ?>" class="btn btn-secondary">Export CSV</a>
                    <a href="export_trial_balance.php?<?php echo http_build_query(array_merge($_GET, ['format' => 'excel'])); ?>" class="btn btn-success">Export Excel</a>
                    <a href="export_trial_balance.php?<?php echo http_build_query(array_merge($_GET, ['format' => 'pdf'])); ?>" class="btn btn-warning">Export PDF</a>
                </div>
            </div>

            <div class="table-responsive">
                <table class="trial-balance-table">
                    <thead>
                        <tr>
                            <th>Account Code</th>
                            <th>Account Name</th>
                            <th>Account Type</th>
                            <th>Debit Balance (GHC)</th>
                            <th>Credit Balance (GHC)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($trial_balance_entries as $entry): ?>
                            <tr class="account-row <?php echo $entry['account_type']; ?>-type">
                                <td><?php echo htmlspecialchars($entry['account_code']); ?></td>
                                <td><?php echo htmlspecialchars($entry['account_name']); ?></td>
                                <td>
                                    <span class="account-type-badge <?php echo $entry['account_type']; ?>">
                                        <?php echo ucfirst($entry['account_type']); ?>
                                    </span>
                                </td>
                                <td class="debit-cell">
                                    <?php echo $entry['debit_balance'] > 0 ? number_format($entry['debit_balance'], 2) : '-'; ?>
                                </td>
                                <td class="credit-cell">
                                    <?php echo $entry['credit_balance'] > 0 ? number_format($entry['credit_balance'], 2) : '-'; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>

                        <!-- Total Row -->
                        <tr class="total-row">
                            <td colspan="3"><strong>TOTAL</strong></td>
                            <td class="total-debit"><strong><?php echo number_format($total_debits, 2); ?></strong></td>
                            <td class="total-credit"><strong><?php echo number_format($total_credits, 2); ?></strong></td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Summary by Account Type -->
        <div class="summary-by-type">
            <h3>Summary by Account Type</h3>
            <div class="summary-grid">
                <?php
                $type_totals = [];
                foreach ($trial_balance_entries as $entry) {
                    $type = $entry['account_type'];
                    if (!isset($type_totals[$type])) {
                        $type_totals[$type] = ['debit' => 0, 'credit' => 0, 'count' => 0];
                    }
                    $type_totals[$type]['debit'] += $entry['debit_balance'];
                    $type_totals[$type]['credit'] += $entry['credit_balance'];
                    $type_totals[$type]['count']++;
                }

                foreach ($type_totals as $type => $totals):
                ?>
                    <div class="summary-card">
                        <h4><?php echo ucfirst($type); ?> Accounts</h4>
                        <div class="summary-stats">
                            <div class="stat">
                                <span class="label">Accounts:</span>
                                <span class="value"><?php echo $totals['count']; ?></span>
                            </div>
                            <div class="stat">
                                <span class="label">Debit Total:</span>
                                <span class="value">GHC <?php echo number_format($totals['debit'], 2); ?></span>
                            </div>
                            <div class="stat">
                                <span class="label">Credit Total:</span>
                                <span class="value">GHC <?php echo number_format($totals['credit'], 2); ?></span>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    <?php endif; ?>
</div>

<style>
.trial-balance-container {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.page-header {
    background: white;
    padding: 30px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.page-header h1 {
    color: #2c3e50;
    margin: 0 0 10px 0;
}

.page-header p {
    color: #7f8c8d;
    margin: 0;
}

.filters-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.filters-form {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.filter-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.filter-group select,
.filter-group input {
    padding: 10px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.btn {
    padding: 10px 16px;
    border: none;
    border-radius: 4px;
    cursor: pointer;
    font-size: 14px;
    text-decoration: none;
    display: inline-block;
    transition: background-color 0.2s;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #545b62;
}

.btn-success {
    background: #28a745;
    color: white;
}

.btn-success:hover {
    background: #1e7e34;
}

.btn-warning {
    background: #ffc107;
    color: #212529;
}

.btn-warning:hover {
    background: #e0a800;
}

.balance-status {
    margin-bottom: 30px;
}

.status-card {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    display: flex;
    align-items: center;
    gap: 20px;
}

.status-card.balanced {
    border-left: 4px solid #27ae60;
}

.status-card.unbalanced {
    border-left: 4px solid #e74c3c;
}

.status-icon {
    font-size: 3em;
}

.status-content h3 {
    color: #2c3e50;
    margin: 0 0 10px 0;
}

.status-content p {
    color: #7f8c8d;
    margin: 0;
}

.trial-balance-table-container {
    background: white;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    overflow: hidden;
    margin-bottom: 30px;
}

.table-header {
    padding: 20px 25px;
    border-bottom: 1px solid #e9ecef;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 10px;
}

.table-header h3 {
    color: #2c3e50;
    margin: 0;
}

.table-actions {
    display: flex;
    gap: 10px;
    flex-wrap: wrap;
}

.table-responsive {
    overflow-x: auto;
}

.trial-balance-table {
    width: 100%;
    border-collapse: collapse;
}

.trial-balance-table th,
.trial-balance-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.trial-balance-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
    position: sticky;
    top: 0;
    z-index: 10;
}

.trial-balance-table tr:hover {
    background: #f8f9fa;
}

.account-type-badge {
    padding: 4px 8px;
    border-radius: 4px;
    font-size: 0.85em;
    font-weight: 600;
    text-transform: uppercase;
}

.account-type-badge.asset { background: #d4edda; color: #155724; }
.account-type-badge.liability { background: #f8d7da; color: #721c24; }
.account-type-badge.equity { background: #d1ecf1; color: #0c5460; }
.account-type-badge.revenue { background: #d4edda; color: #155724; }
.account-type-badge.expense { background: #f8d7da; color: #721c24; }

.debit-cell {
    color: #27ae60;
    font-weight: 600;
}

.credit-cell {
    color: #e74c3c;
    font-weight: 600;
}

.total-row {
    background: #2c3e50;
    color: white;
    font-weight: 600;
}

.total-row td {
    border-top: 2px solid #495057;
}

.total-debit,
.total-credit {
    font-size: 1.1em;
}

.no-entries {
    background: white;
    padding: 40px;
    border-radius: 10px;
    text-align: center;
    color: #7f8c8d;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.summary-by-type {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.summary-by-type h3 {
    color: #2c3e50;
    margin: 0 0 20px 0;
    text-align: center;
}

.summary-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
}

.summary-card {
    background: #f8f9fa;
    padding: 20px;
    border-radius: 8px;
    border-left: 4px solid #3498db;
}

.summary-card h4 {
    color: #2c3e50;
    margin: 0 0 15px 0;
    text-align: center;
}

.summary-stats {
    display: flex;
    flex-direction: column;
    gap: 10px;
}

.stat {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.stat .label {
    font-weight: 600;
    color: #495057;
}

.stat .value {
    font-weight: 600;
    color: #2c3e50;
}

.error-message {
    background: #f8d7da;
    color: #721c24;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #f5c6cb;
}

@media (max-width: 768px) {
    .filter-row {
        grid-template-columns: 1fr;
    }

    .status-card {
        flex-direction: column;
        text-align: center;
    }

    .table-header {
        flex-direction: column;
        text-align: center;
    }

    .table-actions {
        justify-content: center;
    }

    .trial-balance-table th,
    .trial-balance-table td {
        padding: 8px 10px;
        font-size: 14px;
    }

    .summary-grid {
        grid-template-columns: 1fr;
    }
}

@media print {
    .filters-section,
    .table-actions,
    .btn,
    .summary-by-type {
        display: none !important;
    }

    .trial-balance-container {
        padding: 0;
        background: white;
    }

    .page-header {
        margin-bottom: 20px;
    }

    .balance-status {
        margin-bottom: 20px;
    }

    .trial-balance-table-container {
        box-shadow: none;
        border: 1px solid #ddd;
    }
}
</style>

<?php include '../includes/admin_footer.php'; ?>
