<?php
/**
 * Password Reset Utility Class
 * Handles password reset token generation, validation, and email sending
 */

require_once 'email_notification.php';

class PasswordReset {
    private $pdo;
    private $emailNotification;

    public function __construct() {
        global $pdo;
        $this->pdo = $pdo;
        $this->emailNotification = new EmailNotification();
    }

    /**
     * Generate a secure random token
     */
    private function generateToken() {
        return bin2hex(random_bytes(32));
    }

    /**
     * Create a password reset token for a user
     */
    public function createResetToken($username) {
        try {
            // Find user by username
            $stmt = $this->pdo->prepare("SELECT id, first_name, last_name FROM users WHERE username = ? AND is_active = 1");
            $stmt->execute([$username]);
            $user = $stmt->fetch();

            if (!$user) {
                return ['success' => false, 'message' => 'User not found or account is inactive.'];
            }

            // Generate token and expiry (24 hours from now)
            $token = $this->generateToken();
            $expiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));

            // Delete any existing unused tokens for this user
            $stmt = $this->pdo->prepare("DELETE FROM password_reset_tokens WHERE user_id = ? AND used = FALSE");
            $stmt->execute([$user['id']]);

            // Insert new token
            $stmt = $this->pdo->prepare("
                INSERT INTO password_reset_tokens (user_id, token, email, expires_at)
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute([$user['id'], $token, $username, $expiresAt]);

            return [
                'success' => true,
                'token' => $token,
                'user' => $user
            ];

        } catch (Exception $e) {
            error_log("Password reset token creation failed: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to create reset token.'];
        }
    }

    /**
     * Send password reset email
     */
    public function sendResetEmail($username, $token, $user) {
        $resetLink = "http://" . $_SERVER['HTTP_HOST'] . "/school/reset_password.php?token=" . $token;

        $subject = "Password Reset Request - Elitek School Management System";

        $message = "
        <html>
        <head>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0; }
                .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 8px 8px; }
                .button { display: inline-block; background: #667eea; color: white; padding: 12px 24px; text-decoration: none; border-radius: 6px; margin: 20px 0; }
                .footer { text-align: center; margin-top: 20px; color: #666; font-size: 12px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>Elitek School Management System</h1>
                    <p>Password Reset Request</p>
                </div>
                <div class='content'>
                    <h2>Hello " . htmlspecialchars(trim($user['first_name'] . ' ' . $user['last_name'])) . ",</h2>
                    <p>You have requested to reset your password. Click the button below to create a new password:</p>

                    <a href='{$resetLink}' class='button'>Reset Password</a>

                    <p><strong>Important:</strong> This link will expire in 24 hours for security reasons.</p>
                    <p>If you didn't request this password reset, please ignore this email. Your password will remain unchanged.</p>

                    <p>For security reasons, please do not share this email with anyone.</p>
                </div>
                <div class='footer'>
                    <p>This is an automated message from Elitek School Management System. Please do not reply to this email.</p>
                </div>
            </div>
        </body>
        </html>
        ";

        return $this->emailNotification->send($username, $subject, $message, true);
    }

    /**
     * Validate a reset token
     */
    public function validateToken($token) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT prt.*, u.username, u.first_name, u.last_name
                FROM password_reset_tokens prt
                JOIN users u ON prt.user_id = u.id
                WHERE prt.token = ? AND prt.used = FALSE AND prt.expires_at > NOW()
            ");
            $stmt->execute([$token]);
            $result = $stmt->fetch();

            return $result ?: false;

        } catch (Exception $e) {
            error_log("Token validation failed: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Reset user password using token
     */
    public function resetPassword($token, $newPassword) {
        try {
            // Validate token
            $tokenData = $this->validateToken($token);
            if (!$tokenData) {
                return ['success' => false, 'message' => 'Invalid or expired reset token.'];
            }

            // Validate new password
            $passwordErrors = PasswordSecurity::validatePasswordStrength($newPassword);
            if (!empty($passwordErrors)) {
                return ['success' => false, 'message' => 'Password does not meet requirements: ' . implode(', ', $passwordErrors)];
            }

            // Hash new password
            $hashedPassword = PasswordSecurity::hashPassword($newPassword);

            // Update user password
            $stmt = $this->pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
            $stmt->execute([$hashedPassword, $tokenData['user_id']]);

            // Mark token as used
            $stmt = $this->pdo->prepare("UPDATE password_reset_tokens SET used = TRUE WHERE token = ?");
            $stmt->execute([$token]);

            // Log the password reset
            require_once 'user_activity_logger.php';
            UserActivityLogger::logActivity($tokenData['user_id'], 'password_reset', "Password reset successful from IP: " . $_SERVER['REMOTE_ADDR']);

            return ['success' => true, 'message' => 'Password has been reset successfully.'];

        } catch (Exception $e) {
            error_log("Password reset failed: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to reset password.'];
        }
    }

    /**
     * Clean up expired tokens (can be called by a cron job)
     */
    public function cleanupExpiredTokens() {
        try {
            $stmt = $this->pdo->prepare("DELETE FROM password_reset_tokens WHERE expires_at < NOW()");
            $stmt->execute();
            return true;
        } catch (Exception $e) {
            error_log("Token cleanup failed: " . $e->getMessage());
            return false;
        }
    }
}
?>
