<?php
/**
 * Input Validation and Sanitization Utilities
 * Provides secure input handling and validation
 */

class InputValidation {
    /**
     * Sanitize string input
     */
    public static function sanitizeString($input, $allowHtml = false) {
        if ($allowHtml) {
            // Allow basic HTML but strip dangerous tags
            return strip_tags($input, '<p><br><strong><em><u><h1><h2><h3><h4><h5><h6><ul><ol><li><a>');
        } else {
            return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
        }
    }

    /**
     * Sanitize email input
     */
    public static function sanitizeEmail($email) {
        return filter_var(trim($email), FILTER_SANITIZE_EMAIL);
    }

    /**
     * Validate email format
     */
    public static function validateEmail($email) {
        return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
    }

    /**
     * Sanitize and validate integer input
     */
    public static function sanitizeInt($input, $min = null, $max = null) {
        $input = filter_var($input, FILTER_SANITIZE_NUMBER_INT);
        $int = (int)$input;

        if ($min !== null && $int < $min) {
            return false;
        }

        if ($max !== null && $int > $max) {
            return false;
        }

        return $int;
    }

    /**
     * Sanitize and validate float input
     */
    public static function sanitizeFloat($input, $min = null, $max = null) {
        $input = filter_var($input, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
        $float = (float)$input;

        if ($min !== null && $float < $min) {
            return false;
        }

        if ($max !== null && $float > $max) {
            return false;
        }

        return $float;
    }

    /**
     * Sanitize URL input
     */
    public static function sanitizeUrl($url) {
        return filter_var($url, FILTER_SANITIZE_URL);
    }

    /**
     * Validate URL format
     */
    public static function validateUrl($url) {
        return filter_var($url, FILTER_VALIDATE_URL) !== false;
    }

    /**
     * Sanitize filename for file uploads
     */
    public static function sanitizeFilename($filename) {
        // Remove any path components
        $filename = basename($filename);
        // Remove dangerous characters
        $filename = preg_replace('/[^a-zA-Z0-9._-]/', '', $filename);
        return $filename;
    }

    /**
     * Validate file upload
     */
    public static function validateFileUpload($file, $allowedTypes = [], $maxSize = 5242880) { // 5MB default
        $errors = [];

        if (!isset($file['error']) || is_array($file['error'])) {
            $errors[] = 'Invalid file upload';
            return $errors;
        }

        switch ($file['error']) {
            case UPLOAD_ERR_OK:
                break;
            case UPLOAD_ERR_NO_FILE:
                $errors[] = 'No file sent';
                break;
            case UPLOAD_ERR_INI_SIZE:
            case UPLOAD_ERR_FORM_SIZE:
                $errors[] = 'File too large';
                break;
            default:
                $errors[] = 'Unknown upload error';
                break;
        }

        if ($file['size'] > $maxSize) {
            $errors[] = 'File size exceeds limit';
        }

        if (!empty($allowedTypes)) {
            $fileInfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($fileInfo, $file['tmp_name']);
            finfo_close($fileInfo);

            if (!in_array($mimeType, $allowedTypes)) {
                $errors[] = 'File type not allowed';
            }
        }

        return $errors;
    }

    /**
     * Get sanitized POST data
     */
    public static function getPostData($key, $type = 'string', $options = []) {
        if (!isset($_POST[$key])) {
            return null;
        }

        $value = $_POST[$key];

        switch ($type) {
            case 'string':
                return self::sanitizeString($value, $options['allowHtml'] ?? false);
            case 'email':
                return self::sanitizeEmail($value);
            case 'int':
                return self::sanitizeInt($value, $options['min'] ?? null, $options['max'] ?? null);
            case 'float':
                return self::sanitizeFloat($value, $options['min'] ?? null, $options['max'] ?? null);
            case 'url':
                return self::sanitizeUrl($value);
            default:
                return self::sanitizeString($value);
        }
    }

    /**
     * Get sanitized GET data
     */
    public static function getGetData($key, $type = 'string', $options = []) {
        if (!isset($_GET[$key])) {
            return null;
        }

        $value = $_GET[$key];

        switch ($type) {
            case 'string':
                return self::sanitizeString($value, $options['allowHtml'] ?? false);
            case 'email':
                return self::sanitizeEmail($value);
            case 'int':
                return self::sanitizeInt($value, $options['min'] ?? null, $options['max'] ?? null);
            case 'float':
                return self::sanitizeFloat($value, $options['min'] ?? null, $options['max'] ?? null);
            case 'url':
                return self::sanitizeUrl($value);
            default:
                return self::sanitizeString($value);
        }
    }

    /**
     * Validate required fields
     */
    public static function validateRequired($data, $requiredFields) {
        $errors = [];

        foreach ($requiredFields as $field) {
            if (!isset($data[$field]) || empty(trim($data[$field]))) {
                $errors[] = ucfirst(str_replace('_', ' ', $field)) . ' is required';
            }
        }

        return $errors;
    }

    /**
     * Escape output for HTML display
     */
    public static function escapeOutput($data) {
        if (is_array($data)) {
            return array_map([self::class, 'escapeOutput'], $data);
        }

        return htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    }

    /**
     * Clean SQL input (additional layer beyond PDO)
     */
    public static function cleanSqlInput($input) {
        // Remove SQL comments
        $input = preg_replace('/\/\*.*?\*\//', '', $input);
        $input = preg_replace('/--.*?$/', '', $input);

        // Remove dangerous SQL keywords (basic protection)
        $dangerous = ['union', 'select', 'insert', 'update', 'delete', 'drop', 'create', 'alter'];
        $input = str_ireplace($dangerous, '', $input);

        return $input;
    }
}
?>
