<?php
require_once '../config.php';
require_once '../includes/access_control.php';

// Get parameters from URL
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-t');
$report_type = $_GET['report_type'] ?? 'monthly';

// Calculate date range based on report type
if ($report_type === 'yearly') {
    $start_date = date('Y-01-01');
    $end_date = date('Y-12-31');
} elseif ($report_type === 'quarterly') {
    $current_quarter = ceil(date('n') / 3);
    $start_date = date('Y-m-01', mktime(0, 0, 0, ($current_quarter - 1) * 3 + 1, 1));
    $end_date = date('Y-m-t', mktime(0, 0, 0, $current_quarter * 3, 1));
}

// Set headers for file download
$filename = "profit_loss_report_" . date('Y-m-d', strtotime($start_date)) . "_to_" . date('Y-m-d', strtotime($end_date)) . ".csv";
header('Content-Type: text/csv');
header('Content-Disposition: attachment; filename="' . $filename . '"');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Open output stream
$output = fopen('php://output', 'w');

// Get profit & loss data
try {
    // Revenue/Income
    $stmt = $pdo->prepare("
        SELECT
            'Tuition Fees' as category,
            COALESCE(SUM(sf.paid_amount), 0) as amount
        FROM student_fees sf
        WHERE sf.paid_at BETWEEN ? AND ?
    ");
    $stmt->execute([$start_date, $end_date]);
    $tuition_revenue = $stmt->fetch();

    $stmt = $pdo->prepare("
        SELECT
            'Other Fees' as category,
            COALESCE(SUM(sf.paid_amount), 0) as amount
        FROM student_fees sf
        JOIN fees f ON sf.fee_id = f.id
        WHERE sf.paid_at BETWEEN ? AND ?
        AND f.name NOT LIKE '%tuition%'
    ");
    $stmt->execute([$start_date, $end_date]);
    $other_fees_revenue = $stmt->fetch();

    $stmt = $pdo->prepare("
        SELECT
            'Other Income' as category,
            COALESCE(SUM(amount), 0) as amount
        FROM expenses
        WHERE expense_date BETWEEN ? AND ?
        AND category_id IN (SELECT id FROM budget_categories WHERE name = 'Other Income')
    ");
    $stmt->execute([$start_date, $end_date]);
    $other_income = $stmt->fetch();

    // Expenses by category
    $stmt = $pdo->prepare("
        SELECT
            bc.name as category,
            COALESCE(SUM(e.amount), 0) as amount
        FROM budget_categories bc
        LEFT JOIN expenses e ON bc.id = e.category_id
        WHERE e.expense_date BETWEEN ? AND ?
        GROUP BY bc.id, bc.name
        ORDER BY amount DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $expenses_by_category = $stmt->fetchAll();

    // Calculate totals
    $total_revenue = ($tuition_revenue['amount'] ?? 0) + ($other_fees_revenue['amount'] ?? 0) + ($other_income['amount'] ?? 0);
    $total_expenses = array_sum(array_column($expenses_by_category, 'amount'));
    $gross_profit = $total_revenue - $total_expenses;
    $net_profit = $gross_profit;

    // Write CSV headers
    fputcsv($output, ['Profit & Loss Statement']);
    fputcsv($output, ['Period:', date('M d, Y', strtotime($start_date)) . ' to ' . date('M d, Y', strtotime($end_date))]);
    fputcsv($output, ['Report Type:', ucfirst($report_type)]);
    fputcsv($output, []);
    fputcsv($output, ['']);

    // Revenue Section
    fputcsv($output, ['REVENUE']);
    fputcsv($output, ['Source', 'Amount (GHC)']);

    fputcsv($output, [
        $tuition_revenue['category'] ?? 'Tuition Fees',
        number_format($tuition_revenue['amount'] ?? 0, 2)
    ]);

    fputcsv($output, [
        $other_fees_revenue['category'] ?? 'Other Fees',
        number_format($other_fees_revenue['amount'] ?? 0, 2)
    ]);

    fputcsv($output, [
        $other_income['category'] ?? 'Other Income',
        number_format($other_income['amount'] ?? 0, 2)
    ]);

    fputcsv($output, [
        'TOTAL REVENUE',
        number_format($total_revenue, 2)
    ]);

    fputcsv($output, ['']);

    // Expenses Section
    fputcsv($output, ['EXPENSES']);
    fputcsv($output, ['Category', 'Amount (GHC)']);

    foreach ($expenses_by_category as $expense) {
        fputcsv($output, [
            $expense['category'],
            number_format($expense['amount'], 2)
        ]);
    }

    fputcsv($output, [
        'TOTAL EXPENSES',
        number_format($total_expenses, 2)
    ]);

    fputcsv($output, ['']);

    // Profit Section
    fputcsv($output, ['PROFIT ANALYSIS']);
    fputcsv($output, ['Description', 'Amount (GHC)']);

    fputcsv($output, [
        'Gross Profit (Revenue - Expenses)',
        number_format($gross_profit, 2)
    ]);

    fputcsv($output, [
        'Net Profit',
        number_format($net_profit, 2)
    ]);

    fputcsv($output, ['']);

    // Summary
    fputcsv($output, ['SUMMARY']);
    fputcsv($output, ['Metric', 'Value']);

    $profit_margin = $total_revenue > 0 ? round(($net_profit / $total_revenue) * 100, 2) : 0;
    $expense_ratio = $total_revenue > 0 ? round(($total_expenses / $total_revenue) * 100, 2) : 0;

    fputcsv($output, ['Total Revenue', 'GHC ' . number_format($total_revenue, 2)]);
    fputcsv($output, ['Total Expenses', 'GHC ' . number_format($total_expenses, 2)]);
    fputcsv($output, ['Net Profit', 'GHC ' . number_format($net_profit, 2)]);
    fputcsv($output, ['Profit Margin', $profit_margin . '%']);
    fputcsv($output, ['Expense Ratio', $expense_ratio . '%']);

    fputcsv($output, ['']);

    // Profitability Status
    fputcsv($output, ['ANALYSIS']);
    fputcsv($output, ['Status', 'Details']);

    $profitability_status = $net_profit >= 0 ? 'Profitable' : 'Loss Making';
    $status_details = $net_profit >= 0
        ? 'Business is generating profit - GHC ' . number_format($net_profit, 2) . ' net profit'
        : 'Business is operating at a loss - GHC ' . number_format(abs($net_profit), 2) . ' net loss';

    fputcsv($output, [$profitability_status, $status_details]);

    $break_even_ratio = $total_expenses > 0 ? ($total_revenue / $total_expenses) * 100 : 0;
    $break_even_status = $break_even_ratio >= 100 ? 'Above break-even' : 'Below break-even';
    $break_even_details = $break_even_ratio >= 100
        ? 'Operating profitably'
        : 'Needs attention - expenses exceed revenue';

    fputcsv($output, [$break_even_status, $break_even_details]);

    fclose($output);

} catch (PDOException $e) {
    // Handle error
    fputcsv($output, ['Error: ' . $e->getMessage()]);
    fclose($output);
    exit();
}
?>
