<?php
require_once '../config.php';

// Financial report export is now accessible to all logged-in users regardless of role or subscription

// Get parameters
$export_type = $_GET['type'] ?? 'pdf';
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-t');

// Get financial data
try {
    // Revenue data
    $stmt = $pdo->prepare("
        SELECT
            DATE_FORMAT(paid_at, '%Y-%m') as month,
            COUNT(*) as payment_count,
            SUM(paid_amount) as total_amount
        FROM student_fees
        WHERE paid_at BETWEEN ? AND ?
        GROUP BY DATE_FORMAT(paid_at, '%Y-%m')
        ORDER BY month
    ");
    $stmt->execute([$start_date, $end_date]);
    $revenue_data = $stmt->fetchAll();

    // Fee breakdown
    $stmt = $pdo->prepare("
        SELECT
            f.name as fee_name,
            COUNT(sf.id) as assigned_count,
            COUNT(CASE WHEN sf.paid = 1 THEN 1 END) as paid_count,
            SUM(f.amount) as total_amount,
            SUM(sf.paid_amount) as collected_amount
        FROM fees f
        LEFT JOIN student_fees sf ON f.id = sf.fee_id
        GROUP BY f.id, f.name
        ORDER BY total_amount DESC
    ");
    $stmt->execute();
    $fee_breakdown = $stmt->fetchAll();

    // Outstanding students
    $stmt = $pdo->prepare("
        SELECT
            s.name as student_name,
            s.class,
            SUM(f.amount - sf.paid_amount) as outstanding_amount
        FROM students s
        JOIN student_fees sf ON s.id = sf.student_id
        JOIN fees f ON sf.fee_id = f.id
        WHERE sf.paid = 0 OR sf.paid_amount < f.amount
        GROUP BY s.id, s.name, s.class
        ORDER BY outstanding_amount DESC
        LIMIT 20
    ");
    $stmt->execute();
    $outstanding_students = $stmt->fetchAll();

} catch (PDOException $e) {
    die("Error fetching data: " . $e->getMessage());
}

// Generate filename
$filename = "financial_report_" . date('Y-m-d') . "." . $export_type;

// Set headers based on export type
switch ($export_type) {
    case 'pdf':
        header('Content-Type: application/pdf');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        generatePDFReport($revenue_data, $fee_breakdown, $outstanding_students, $start_date, $end_date);
        break;

    case 'excel':
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        generateExcelReport($revenue_data, $fee_breakdown, $outstanding_students, $start_date, $end_date);
        break;

    case 'csv':
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        generateCSVReport($revenue_data, $fee_breakdown, $outstanding_students, $start_date, $end_date);
        break;

    default:
        die("Invalid export type");
}

function generatePDFReport($revenue_data, $fee_breakdown, $outstanding_students, $start_date, $end_date) {
    // This would require a PDF library like TCPDF or FPDF
    // For now, we'll create a simple HTML representation
    echo "<html><head><title>Financial Report</title></head><body>";
    echo "<h1>Financial Report</h1>";
    echo "<p>Period: " . date('M d, Y', strtotime($start_date)) . " - " . date('M d, Y', strtotime($end_date)) . "</p>";

    echo "<h2>Revenue Summary</h2>";
    echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
    echo "<tr><th>Month</th><th>Payments</th><th>Total Amount</th></tr>";
    foreach ($revenue_data as $row) {
        echo "<tr>";
        echo "<td>" . htmlspecialchars($row['month']) . "</td>";
        echo "<td>" . htmlspecialchars($row['payment_count']) . "</td>";
        echo "<td>GHC " . number_format($row['total_amount'], 2) . "</td>";
        echo "</tr>";
    }
    echo "</table>";

    echo "<h2>Fee Performance</h2>";
    echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
    echo "<tr><th>Fee Type</th><th>Assigned</th><th>Collected</th><th>Collection Rate</th><th>Total Amount</th><th>Collected Amount</th></tr>";
    foreach ($fee_breakdown as $fee) {
        $rate = $fee['assigned_count'] > 0 ? round(($fee['paid_count'] / $fee['assigned_count']) * 100, 2) : 0;
        echo "<tr>";
        echo "<td>" . htmlspecialchars($fee['fee_name']) . "</td>";
        echo "<td>" . htmlspecialchars($fee['assigned_count']) . "</td>";
        echo "<td>" . htmlspecialchars($fee['paid_count']) . "</td>";
        echo "<td>" . $rate . "%</td>";
        echo "<td>GHC " . number_format($fee['total_amount'], 2) . "</td>";
        echo "<td>GHC " . number_format($fee['collected_amount'], 2) . "</td>";
        echo "</tr>";
    }
    echo "</table>";

    echo "</body></html>";
}

function generateExcelReport($revenue_data, $fee_breakdown, $outstanding_students, $start_date, $end_date) {
    echo "Financial Report\n";
    echo "Period: " . date('M d, Y', strtotime($start_date)) . " - " . date('M d, Y', strtotime($end_date)) . "\n\n";

    echo "Revenue Summary\n";
    echo "Month\tPayments\tTotal Amount\n";
    foreach ($revenue_data as $row) {
        echo $row['month'] . "\t" . $row['payment_count'] . "\tGHC " . number_format($row['total_amount'], 2) . "\n";
    }

    echo "\nFee Performance\n";
    echo "Fee Type\tAssigned\tCollected\tCollection Rate\tTotal Amount\tCollected Amount\n";
    foreach ($fee_breakdown as $fee) {
        $rate = $fee['assigned_count'] > 0 ? round(($fee['paid_count'] / $fee['assigned_count']) * 100, 2) : 0;
        echo $fee['fee_name'] . "\t" . $fee['assigned_count'] . "\t" . $fee['paid_count'] . "\t" . $rate . "%\tGHC " . number_format($fee['total_amount'], 2) . "\tGHC " . number_format($fee['collected_amount'], 2) . "\n";
    }
}

function generateCSVReport($revenue_data, $fee_breakdown, $outstanding_students, $start_date, $end_date) {
    echo "Financial Report\n";
    echo "Period: " . date('M d, Y', strtotime($start_date)) . " - " . date('M d, Y', strtotime($end_date)) . "\n\n";

    echo "Revenue Summary\n";
    echo "\"Month\",\"Payments\",\"Total Amount\"\n";
    foreach ($revenue_data as $row) {
        echo "\"" . $row['month'] . "\"," . $row['payment_count'] . ",\"GHC " . number_format($row['total_amount'], 2) . "\"\n";
    }

    echo "\nFee Performance\n";
    echo "\"Fee Type\",\"Assigned\",\"Collected\",\"Collection Rate\",\"Total Amount\",\"Collected Amount\"\n";
    foreach ($fee_breakdown as $fee) {
        $rate = $fee['assigned_count'] > 0 ? round(($fee['paid_count'] / $fee['assigned_count']) * 100, 2) : 0;
        echo "\"" . $fee['fee_name'] . "\"," . $fee['assigned_count'] . "," . $fee['paid_count'] . ",\"" . $rate . "%\",\"GHC " . number_format($fee['total_amount'], 2) . "\",\"GHC " . number_format($fee['collected_amount'], 2) . "\"\n";
    }
}
?>
