<?php
require_once '../config.php';
require_once 'asset_audit_functions.php';

include 'includes/accounting_header.php';

$message = '';
$error = '';

/**
 * Calculate depreciation or appreciation for an asset
 * @param array $asset Asset data
 * @param string $calculation_date Date to calculate depreciation/appreciation for (default: today)
 * @return array Calculation results (depreciation or appreciation)
 */
function calculateAssetDepreciation($asset, $calculation_date = null) {
    if (!$calculation_date) {
        $calculation_date = date('Y-m-d');
    }

    $purchase_date = strtotime($asset['purchase_date']);
    $calc_date = strtotime($calculation_date);

    // Calculate months since purchase
    $months_elapsed = floor(($calc_date - $purchase_date) / (30 * 24 * 60 * 60)); // Approximate months

    if ($months_elapsed <= 0) {
        return [
            'current_value' => $asset['purchase_cost'],
            'accumulated_depreciation' => 0,
            'monthly_depreciation' => 0,
            'months_elapsed' => 0,
            'depreciation_percentage' => 0,
            'calculation_type' => 'none'
        ];
    }

    // Check if asset appreciates or depreciates
    $appreciation_method = $asset['appreciation_method'] ?? 'none';
    $depreciation_method = $asset['depreciation_method'] ?? 'straight_line';

    if ($appreciation_method !== 'none') {
        // Asset appreciates
        $appreciation_rate = $asset['appreciation_rate'] ?? 0;

        if ($appreciation_method === 'compound') {
            // Compound appreciation: (1 + rate/100)^years - 1
            $years_elapsed = $months_elapsed / 12;
            $compound_factor = pow(1 + ($appreciation_rate / 100), $years_elapsed);
            $accumulated_appreciation = $asset['purchase_cost'] * ($compound_factor - 1);
            $current_value = $asset['purchase_cost'] * $compound_factor;
            $monthly_appreciation = $accumulated_appreciation / $months_elapsed;
        } else {
            // Simple appreciation (fixed percentage)
            $monthly_appreciation = ($asset['purchase_cost'] * $appreciation_rate) / 100;
            $accumulated_appreciation = $monthly_appreciation * $months_elapsed;
            $current_value = $asset['purchase_cost'] + $accumulated_appreciation;
        }

        $appreciation_percentage = ($accumulated_appreciation / $asset['purchase_cost']) * 100;

        return [
            'current_value' => round($current_value, 2),
            'accumulated_depreciation' => round(-$accumulated_appreciation, 2), // Negative for appreciation
            'monthly_depreciation' => round(-$monthly_appreciation, 2), // Negative for appreciation
            'months_elapsed' => $months_elapsed,
            'depreciation_percentage' => round(-$appreciation_percentage, 2), // Negative for appreciation
            'calculation_type' => 'appreciation'
        ];
    } else {
        // Asset depreciates (default behavior)
        $monthly_depreciation = ($asset['purchase_cost'] * ($asset['depreciation_rate'] ?? 0)) / 100;
        $accumulated_depreciation = min($monthly_depreciation * $months_elapsed, $asset['purchase_cost']);
        $current_value = max(0, $asset['purchase_cost'] - $accumulated_depreciation);
        $depreciation_percentage = ($accumulated_depreciation / $asset['purchase_cost']) * 100;

        return [
            'current_value' => round($current_value, 2),
            'accumulated_depreciation' => round($accumulated_depreciation, 2),
            'monthly_depreciation' => round($monthly_depreciation, 2),
            'months_elapsed' => $months_elapsed,
            'depreciation_percentage' => round($depreciation_percentage, 2),
            'calculation_type' => 'depreciation'
        ];
    }
}

/**
 * Update asset depreciation values in database
 * @param PDO $pdo
 * @param int $asset_id
 * @param array $depreciation_data
 * @return bool
 */
function updateAssetDepreciation($pdo, $asset_id, $depreciation_data) {
    try {
        $stmt = $pdo->prepare("
            UPDATE assets
            SET current_value = ?, accumulated_depreciation = ?, updated_at = NOW()
            WHERE id = ?
        ");

        $stmt->execute([
            $depreciation_data['current_value'],
            $depreciation_data['accumulated_depreciation'],
            $asset_id
        ]);

        return true;
    } catch (Exception $e) {
        error_log("Failed to update asset depreciation: " . $e->getMessage());
        return false;
    }
}

// Handle form submission for calculating depreciation
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['calculate_single'])) {
        $asset_id = intval($_POST['asset_id']);
        $calculation_date = $_POST['calculation_date'] ?? date('Y-m-d');

        try {
            // Get asset data
            $stmt = $pdo->prepare("SELECT * FROM assets WHERE id = ?");
            $stmt->execute([$asset_id]);
            $asset = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$asset) {
                $error = "Asset not found.";
            } else {
                // Calculate depreciation
                $depreciation = calculateAssetDepreciation($asset, $calculation_date);

                // Update asset in database
                if (updateAssetDepreciation($pdo, $asset_id, $depreciation)) {
                    // Log the depreciation calculation
                    logAssetAction($pdo, $asset_id, $_SESSION['user_id'] ?? 1, 'depreciation_calculated', null, [
                        'calculation_date' => $calculation_date,
                        'current_value' => $depreciation['current_value'],
                        'accumulated_depreciation' => $depreciation['accumulated_depreciation']
                    ], [
                        'current_value' => ['old' => $asset['current_value'], 'new' => $depreciation['current_value']],
                        'accumulated_depreciation' => ['old' => $asset['accumulated_depreciation'], 'new' => $depreciation['accumulated_depreciation']]
                    ]);

                    $message = "Depreciation calculated successfully for asset: " . htmlspecialchars($asset['name']);
                } else {
                    $error = "Failed to update asset depreciation.";
                }
            }
        } catch (Exception $e) {
            $error = "Error calculating depreciation: " . $e->getMessage();
        }
    } elseif (isset($_POST['calculate_all'])) {
        $calculation_date = $_POST['calculation_date'] ?? date('Y-m-d');

        try {
            // Get all assets
            $stmt = $pdo->query("SELECT * FROM assets WHERE status IN ('active', 'leased')");
            $assets = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $updated_count = 0;
            foreach ($assets as $asset) {
                $depreciation = calculateAssetDepreciation($asset, $calculation_date);
                if (updateAssetDepreciation($pdo, $asset['id'], $depreciation)) {
                    $updated_count++;
                }
            }

            $message = "Depreciation calculated for {$updated_count} assets.";
        } catch (Exception $e) {
            $error = "Error calculating depreciation for all assets: " . $e->getMessage();
        }
    }
}

// Get assets for the form
try {
    $stmt = $pdo->query("SELECT id, name, purchase_cost, current_value, accumulated_depreciation, depreciation_rate, purchase_date FROM assets ORDER BY name");
    $assets = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $assets = [];
    $error = "Error loading assets.";
}
?>

<style>
.depreciation-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
}

.depreciation-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 30px;
    border-radius: 12px;
    margin-bottom: 30px;
    text-align: center;
}

.depreciation-header h1 {
    margin: 0;
    font-size: 2.2em;
    font-weight: 300;
}

.depreciation-header p {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.calculation-section {
    background: #f8f9fa;
    padding: 25px;
    border-radius: 12px;
    margin-bottom: 30px;
    border: 1px solid #dee2e6;
}

.calculation-form {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    align-items: end;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 8px;
    color: #495057;
    font-size: 0.95em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.form-group select,
.form-group input {
    padding: 12px 16px;
    border: 2px solid #2E7D32;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.3s ease;
}

.form-group select:focus,
.form-group input:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.btn {
    padding: 12px 24px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(46, 125, 50, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(46, 125, 50, 0.4);
}

.btn-success {
    background: linear-gradient(135deg, #F9A825 0%, #FFC107 100%);
    color: #2E7D32;
    box-shadow: 0 4px 15px rgba(249, 168, 37, 0.3);
}

.btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(249, 168, 37, 0.4);
}

.assets-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
}

.assets-table th,
.assets-table td {
    padding: 16px 20px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.assets-table th {
    background: #fff;
    font-weight: 700;
    color: #000;
    text-transform: uppercase;
    font-size: 0.85em;
    letter-spacing: 0.5px;
}

.assets-table tr:hover {
    background: #f8f9fa;
}

.asset-name {
    font-weight: 600;
    color: #2c3e50;
}

.asset-cost {
    font-weight: 700;
    color: #e74c3c;
}

.asset-value {
    font-weight: 600;
    color: #27ae60;
}

.depreciation-rate {
    color: #e67e22;
    font-weight: 600;
}

.message {
    padding: 15px;
    margin-bottom: 20px;
    border-radius: 8px;
    font-weight: 600;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.calculation-result {
    background: white;
    border: 1px solid #dee2e6;
    border-radius: 8px;
    padding: 20px;
    margin-top: 20px;
}

.calculation-result h3 {
    margin-top: 0;
    color: #2c3e50;
    border-bottom: 2px solid #2E7D32;
    padding-bottom: 10px;
}

.result-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
    margin-top: 15px;
}

.result-item {
    background: #f8f9fa;
    padding: 15px;
    border-radius: 8px;
    text-align: center;
}

.result-item .value {
    font-size: 1.5em;
    font-weight: 700;
    color: #2c3e50;
    display: block;
}

.result-item .label {
    font-size: 0.9em;
    color: #6c757d;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-top: 5px;
}
</style>

<div class="depreciation-container">
    <div class="depreciation-header">
        <h1>📊 Asset Depreciation Calculator</h1>
        <p>Calculate and update depreciation for school assets</p>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Calculation Form -->
    <div class="calculation-section">
        <h2 style="margin-top: 0; color: #2c3e50;">Calculate Depreciation</h2>

        <form method="post" class="calculation-form">
            <div class="form-group">
                <label for="asset_id">Select Asset</label>
                <select name="asset_id" id="asset_id" required>
                    <option value="">Choose an asset...</option>
                    <?php foreach ($assets as $asset): ?>
                        <option value="<?php echo $asset['id']; ?>">
                            <?php echo htmlspecialchars($asset['name']); ?> (GHC <?php echo number_format($asset['purchase_cost'], 2); ?>)
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="form-group">
                <label for="calculation_date">Calculation Date</label>
                <input type="date" name="calculation_date" id="calculation_date" value="<?php echo date('Y-m-d'); ?>" required>
            </div>

            <div class="form-group">
                <button type="submit" name="calculate_single" class="btn btn-primary">🧮 Calculate Single Asset</button>
            </div>

            <div class="form-group">
                <button type="submit" name="calculate_all" class="btn btn-success">🔄 Calculate All Assets</button>
            </div>
        </form>
    </div>

    <!-- Assets Table with Current Depreciation -->
    <h2 style="color: #2c3e50; border-bottom: 2px solid #2E7D32; padding-bottom: 10px;">📋 Asset Depreciation Overview</h2>

    <?php if (empty($assets)): ?>
        <p>No assets found in the system.</p>
    <?php else: ?>
        <table class="assets-table">
            <thead>
                <tr>
                    <th>Asset Name</th>
                    <th>Purchase Cost</th>
                    <th>Current Value</th>
                    <th>Accumulated Depreciation</th>
                    <th>Depreciation Rate</th>
                    <th>Purchase Date</th>
                    <th>Depreciation %</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($assets as $asset): ?>
                    <?php
                    $depreciation = calculateAssetDepreciation($asset);
                    $depreciation_percentage = ($asset['purchase_cost'] > 0) ? (($asset['accumulated_depreciation'] / $asset['purchase_cost']) * 100) : 0;
                    ?>
                    <tr>
                        <td class="asset-name"><?php echo htmlspecialchars($asset['name']); ?></td>
                        <td class="asset-cost">GHC <?php echo number_format($asset['purchase_cost'], 2); ?></td>
                        <td class="asset-value">GHC <?php echo number_format($asset['current_value'] ?? $asset['purchase_cost'], 2); ?></td>
                        <td class="asset-cost">GHC <?php echo number_format($asset['accumulated_depreciation'] ?? 0, 2); ?></td>
                        <td class="depreciation-rate"><?php echo number_format($asset['depreciation_rate'], 2); ?>%</td>
                        <td><?php echo date('M d, Y', strtotime($asset['purchase_date'])); ?></td>
                        <td><?php echo number_format($depreciation_percentage, 2); ?>%</td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>

    <!-- Calculation Result (if any) -->
    <?php if (isset($_POST['calculate_single']) && !$error && isset($asset) && isset($depreciation)): ?>
        <div class="calculation-result">
            <h3>📈 <?php echo ucfirst($depreciation['calculation_type']); ?> Calculation Result</h3>
            <p><strong>Asset:</strong> <?php echo htmlspecialchars($asset['name']); ?></p>
            <p><strong>Calculation Date:</strong> <?php echo date('M d, Y', strtotime($calculation_date)); ?></p>
            <p><strong>Calculation Type:</strong> <?php echo ucfirst($depreciation['calculation_type']); ?></p>

            <div class="result-grid">
                <div class="result-item">
                    <span class="value">GHC <?php echo number_format($depreciation['current_value'], 2); ?></span>
                    <span class="label">Current Value</span>
                </div>
                <div class="result-item">
                    <span class="value">GHC <?php echo number_format(abs($depreciation['accumulated_depreciation']), 2); ?></span>
                    <span class="label">Accumulated <?php echo ucfirst($depreciation['calculation_type']); ?></span>
                </div>
                <div class="result-item">
                    <span class="value">GHC <?php echo number_format(abs($depreciation['monthly_depreciation']), 2); ?></span>
                    <span class="label">Monthly <?php echo ucfirst($depreciation['calculation_type']); ?></span>
                </div>
                <div class="result-item">
                    <span class="value"><?php echo $depreciation['months_elapsed']; ?> months</span>
                    <span class="label">Time Elapsed</span>
                </div>
                <div class="result-item">
                    <span class="value"><?php echo number_format(abs($depreciation['depreciation_percentage']), 2); ?>%</span>
                    <span class="label"><?php echo ucfirst($depreciation['calculation_type']); ?> %</span>
                </div>
            </div>
        </div>
    <?php endif; ?>
</div>

<script>
// Auto-calculate depreciation when asset is selected
document.getElementById('asset_id').addEventListener('change', function() {
    const assetId = this.value;
    if (assetId) {
        // Could add AJAX call here to show live calculation preview
        console.log('Asset selected:', assetId);
    }
});
</script>
