<?php
require_once '../config.php';

$message = '';
$error = '';

// Get filter parameters
$academic_year_id = $_GET['academic_year_id'] ?? '';
$category_id = $_GET['category_id'] ?? '';
$report_period = $_GET['report_period'] ?? 'yearly'; // yearly, monthly, quarterly

// Get academic years for filter
$stmt = $pdo->query("SELECT id, year FROM academic_years ORDER BY year DESC");
$academic_years = $stmt->fetchAll();

// Get budget categories for filter
$stmt = $pdo->query("SELECT id, name FROM budget_categories ORDER BY name");
$budget_categories = $stmt->fetchAll();

// Get school settings for print functionality
$school_settings = $pdo->query("SELECT school_name, address, phone, email FROM school_settings LIMIT 1")->fetch(PDO::FETCH_ASSOC);

// Build query based on filters
$where_conditions = [];
$params = [];

if (!empty($academic_year_id)) {
    $where_conditions[] = "b.academic_year_id = ?";
    $params[] = $academic_year_id;
}

if (!empty($category_id)) {
    $where_conditions[] = "b.category_id = ?";
    $params[] = $category_id;
}

$where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get budget vs actual data
try {
    if ($report_period === 'monthly') {
        // Monthly breakdown - Note: This needs proper monthly budget allocation logic
        // For now, showing yearly data grouped by month
        $stmt = $pdo->prepare("
            SELECT
                bc.name as category_name,
                ay.year as academic_year,
                b.planned_amount as budget_amount,
                COALESCE(SUM(CASE WHEN b.budget_type = 'expense' THEN e.amount ELSE i.amount END), 0) as actual_amount,
                (b.planned_amount - COALESCE(SUM(CASE WHEN b.budget_type = 'expense' THEN e.amount ELSE i.amount END), 0)) as variance,
                CASE
                    WHEN b.planned_amount > 0
                    THEN ROUND(((b.planned_amount - COALESCE(SUM(CASE WHEN b.budget_type = 'expense' THEN e.amount ELSE i.amount END), 0)) / b.planned_amount) * 100, 2)
                    ELSE 0
                END as variance_percentage,
                CASE WHEN b.budget_type = 'expense' THEN MONTH(e.expense_date) ELSE MONTH(i.income_date) END as period_month,
                CASE WHEN b.budget_type = 'expense' THEN YEAR(e.expense_date) ELSE YEAR(i.income_date) END as period_year,
                b.budget_type
            FROM budgets b
            LEFT JOIN budget_categories bc ON b.category_id = bc.id
            LEFT JOIN academic_years ay ON b.academic_year_id = ay.id
            LEFT JOIN expenses e ON bc.id = e.category_id AND b.budget_type = 'expense'
                AND e.expense_date BETWEEN ay.start_date AND ay.end_date
            LEFT JOIN incomes i ON bc.id = i.category_id AND b.budget_type = 'income'
                AND i.income_date BETWEEN ay.start_date AND ay.end_date
            " . $where_clause . "
            GROUP BY bc.id, b.id, ay.id, period_month, period_year, b.budget_type
            ORDER BY bc.name, ay.year, period_year, period_month
        ");
        $stmt->execute($params);
        $budget_data = $stmt->fetchAll();
    } else {
        // Yearly breakdown
        $stmt = $pdo->prepare("
            SELECT
                bc.name as category_name,
                ay.year as academic_year,
                SUM(b.planned_amount) as budget_amount,
                COALESCE(SUM(CASE WHEN b.budget_type = 'expense' THEN e.amount ELSE i.amount END), 0) as actual_amount,
                (SUM(b.planned_amount) - COALESCE(SUM(CASE WHEN b.budget_type = 'expense' THEN e.amount ELSE i.amount END), 0)) as variance,
                CASE
                    WHEN SUM(b.planned_amount) > 0
                    THEN ROUND(((SUM(b.planned_amount) - COALESCE(SUM(CASE WHEN b.budget_type = 'expense' THEN e.amount ELSE i.amount END), 0)) / SUM(b.planned_amount)) * 100, 2)
                    ELSE 0
                END as variance_percentage,
                b.budget_type
            FROM budgets b
            LEFT JOIN budget_categories bc ON b.category_id = bc.id
            LEFT JOIN academic_years ay ON b.academic_year_id = ay.id
            LEFT JOIN expenses e ON bc.id = e.category_id AND b.budget_type = 'expense'
                AND YEAR(e.expense_date) BETWEEN SUBSTRING_INDEX(REPLACE(ay.year, '/', '-'), '-', 1) AND SUBSTRING_INDEX(REPLACE(ay.year, '/', '-'), '-', -1)
            LEFT JOIN incomes i ON bc.id = i.category_id AND b.budget_type = 'income'
                AND YEAR(i.income_date) BETWEEN SUBSTRING_INDEX(REPLACE(ay.year, '/', '-'), '-', 1) AND SUBSTRING_INDEX(REPLACE(ay.year, '/', '-'), '-', -1)
            " . $where_clause . "
            GROUP BY bc.id, ay.id, b.budget_type
            ORDER BY bc.name, ay.year
        ");
        $stmt->execute($params);
        $budget_data = $stmt->fetchAll();
    }

    // Calculate totals
    $total_budget = array_sum(array_column($budget_data, 'budget_amount'));
    $total_actual = array_sum(array_column($budget_data, 'actual_amount'));
    $total_variance = $total_budget - $total_actual;
    $total_variance_percentage = $total_budget > 0 ? round(($total_variance / $total_budget) * 100, 2) : 0;

    // Get total current revenue (from journal entries - double-entry system)
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(jel.credit_amount), 0) as total_revenue
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE YEAR(je.transaction_date) = YEAR(CURRENT_DATE)
        AND coa.account_type = 'revenue'
        AND jel.credit_amount > 0
        AND je.status = 'posted'
    ");
    $stmt->execute();
    $total_revenue = $stmt->fetch()['total_revenue'] ?? 0;

    // Get budget utilization by category
    $stmt = $pdo->prepare("
        SELECT
            bc.name as category_name,
            COUNT(b.id) as budget_items,
            SUM(b.planned_amount) as total_budget,
            COALESCE(SUM(e.amount), 0) as total_actual,
            CASE
                WHEN SUM(b.planned_amount) > 0
                THEN ROUND((COALESCE(SUM(e.amount), 0) / SUM(b.planned_amount)) * 100, 2)
                ELSE 0
            END as utilization_percentage
        FROM budget_categories bc
        LEFT JOIN budgets b ON bc.id = b.category_id
        LEFT JOIN academic_years ay ON b.academic_year_id = ay.id
        LEFT JOIN expenses e ON bc.id = e.category_id
            AND YEAR(e.expense_date) BETWEEN SUBSTRING_INDEX(REPLACE(ay.year, '/', '-'), '-', 1) AND SUBSTRING_INDEX(REPLACE(ay.year, '/', '-'), '-', -1)
        GROUP BY bc.id, bc.name
        ORDER BY utilization_percentage DESC
    ");
    $stmt->execute();
    $category_utilization = $stmt->fetchAll();

} catch (PDOException $e) {
    $error = "Error fetching budget data: " . $e->getMessage();
}

include 'includes/accounting_header.php';
?>

<div class="budget-report-container">
    <div class="report-header">
        <h1>📊 Budget vs Actual Report</h1>
        <div class="report-filters">
            <form method="get" class="filter-form">
                <div class="filter-group">
                    <label for="academic_year_id">Academic Year:</label>
                    <select name="academic_year_id" id="academic_year_id">
                        <option value="">All Years</option>
                        <?php foreach ($academic_years as $year): ?>
                            <option value="<?php echo $year['id']; ?>" <?php echo $academic_year_id == $year['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($year['year']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="category_id">Category:</label>
                    <select name="category_id" id="category_id">
                        <option value="">All Categories</option>
                        <?php foreach ($budget_categories as $category): ?>
                            <option value="<?php echo $category['id']; ?>" <?php echo $category_id == $category['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($category['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="report_period">Period:</label>
                    <select name="report_period" id="report_period">
                        <option value="yearly" <?php echo $report_period === 'yearly' ? 'selected' : ''; ?>>Yearly</option>
                        <option value="monthly" <?php echo $report_period === 'monthly' ? 'selected' : ''; ?>>Monthly</option>
                    </select>
                </div>

                <button type="submit" class="btn btn-primary">Filter</button>
                <a href="export_budget_report.php?<?php echo http_build_query($_GET); ?>" class="btn btn-secondary">Export</a>
                <button type="button" onclick="printReport()" class="btn btn-primary">Print Report</button>
            </form>
        </div>
    </div>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Current Revenue Display -->
    <div class="revenue-display">
        <div class="revenue-card">
            <div class="revenue-icon">💰</div>
            <div class="revenue-content">
                <h2>GHC <?php echo number_format($total_revenue, 2); ?></h2>
                <p>Total Current Revenue (<?php echo date('Y'); ?>)</p>
            </div>
        </div>
    </div>

    <!-- Summary Cards -->
    <div class="summary-cards">
        <div class="card budget-card">
            <div class="card-icon">💰</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($total_budget, 2); ?></h3>
                <p>Total Budget</p>
            </div>
        </div>

        <div class="card actual-card">
            <div class="card-icon">📊</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($total_actual, 2); ?></h3>
                <p>Actual Spending</p>
            </div>
        </div>

        <div class="card variance-card <?php echo $total_variance >= 0 ? 'positive' : 'negative'; ?>">
            <div class="card-icon"><?php echo $total_variance >= 0 ? '📈' : '📉'; ?></div>
            <div class="card-content">
                <h3>GHC <?php echo number_format(abs($total_variance), 2); ?></h3>
                <p>Variance (<?php echo $total_variance >= 0 ? 'Under Budget' : 'Over Budget'; ?>)</p>
            </div>
        </div>

        <div class="card utilization-card">
            <div class="card-icon">🎯</div>
            <div class="card-content">
                <h3><?php echo $total_budget > 0 ? round(($total_actual / $total_budget) * 100, 1) : 0; ?>%</h3>
                <p>Budget Utilization</p>
            </div>
        </div>
    </div>

    <!-- Budget vs Actual Table -->
    <div class="report-section">
        <h2>Detailed Budget vs Actual Analysis</h2>
        <div class="table-container">
            <table class="budget-table">
                <thead>
                    <tr>
                        <th>Category</th>
                        <th>Academic Year</th>
                        <th>Budget Amount</th>
                        <th>Actual Amount</th>
                        <th>Variance</th>
                        <th>Variance %</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($budget_data)): ?>
                        <tr>
                            <td colspan="7" class="no-data">No budget data available for the selected criteria.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($budget_data as $row): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($row['category_name']); ?></td>
                                <td><?php echo htmlspecialchars($row['academic_year']); ?></td>
                                <td>GHC <?php echo number_format($row['budget_amount'], 2); ?></td>
                                <td>GHC <?php echo number_format($row['actual_amount'], 2); ?></td>
                                <td class="<?php echo $row['variance'] >= 0 ? 'positive' : 'negative'; ?>">
                                    GHC <?php echo number_format(abs($row['variance']), 2); ?>
                                </td>
                                <td class="<?php echo $row['variance'] >= 0 ? 'positive' : 'negative'; ?>">
                                    <?php echo $row['variance_percentage']; ?>%
                                </td>
                                <td>
                                    <span class="status-badge <?php
                                        if ($row['variance_percentage'] > 20) echo 'excellent';
                                        elseif ($row['variance_percentage'] > 5) echo 'good';
                                        elseif ($row['variance_percentage'] > -5) echo 'warning';
                                        else echo 'danger';
                                    ?>">
                                        <?php
                                        if ($row['variance_percentage'] > 20) echo 'Excellent';
                                        elseif ($row['variance_percentage'] > 5) echo 'Good';
                                        elseif ($row['variance_percentage'] > -5) echo 'On Track';
                                        elseif ($row['variance_percentage'] > -20) echo 'Over Budget';
                                        else echo 'Critical';
                                        ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Category Utilization Chart -->
    <div class="report-section">
        <h2>Budget Utilization by Category</h2>
        <div class="chart-container">
            <canvas id="utilizationChart" width="400" height="200"></canvas>
        </div>
    </div>

    <!-- Recommendations -->
    <div class="report-section">
        <h2>💡 Recommendations</h2>
        <div class="recommendations">
            <?php if ($total_variance_percentage > 15): ?>
                <div class="recommendation success">
                    <strong>Excellent Budget Management!</strong> You're significantly under budget. Consider allocating surplus to priority areas or savings.
                </div>
            <?php elseif ($total_variance_percentage > 5): ?>
                <div class="recommendation good">
                    <strong>Good Budget Control!</strong> You're managing your budget well. Continue monitoring expenses closely.
                </div>
            <?php elseif ($total_variance_percentage > -5): ?>
                <div class="recommendation warning">
                    <strong>Budget on Track!</strong> Your spending is aligned with your budget. Maintain current spending patterns.
                </div>
            <?php elseif ($total_variance_percentage > -15): ?>
                <div class="recommendation danger">
                    <strong>Budget Overrun Warning!</strong> You're exceeding your budget. Review expenses and implement cost control measures.
                </div>
            <?php else: ?>
                <div class="recommendation critical">
                    <strong>Critical Budget Situation!</strong> Immediate action required. Review all expenses and implement strict cost controls.
                </div>
            <?php endif; ?>

            <?php
            // Find categories with highest overruns
            $overruns = array_filter($budget_data, function($row) {
                return $row['variance'] < 0;
            });
            usort($overruns, function($a, $b) {
                return abs($a['variance']) <=> abs($b['variance']);
            });

            if (!empty($overruns)): ?>
                <div class="recommendation info">
                    <strong>Priority Areas for Review:</strong>
                    <?php
                    $top_overruns = array_slice($overruns, 0, 3);
                    foreach ($top_overruns as $overrun) {
                        echo htmlspecialchars($overrun['category_name']) . " (Over by GHC " . number_format(abs($overrun['variance']), 2) . "), ";
                    }
                    ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
.budget-report-container {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.revenue-display {
    margin-bottom: 30px;
}

.revenue-card {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 30px;
    border-radius: 15px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.2);
    text-align: center;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 20px;
}

.revenue-icon {
    font-size: 3em;
}

.revenue-content h2 {
    margin: 0;
    font-size: 2.5em;
    font-weight: bold;
}

.revenue-content p {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.report-header {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.report-header h1 {
    color: #2c3e50;
    margin: 0 0 20px 0;
}

.filter-form {
    display: flex;
    gap: 20px;
    align-items: end;
    flex-wrap: wrap;
}

.filter-group {
    display: flex;
    flex-direction: column;
    min-width: 150px;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.filter-group select {
    padding: 8px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.card {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
    transition: transform 0.2s;
}

.card:hover {
    transform: translateY(-5px);
}

.card-icon {
    font-size: 2.5em;
    margin-bottom: 15px;
}

.card h3 {
    color: #2c3e50;
    font-size: 1.8em;
    margin-bottom: 5px;
}

.card p {
    color: #7f8c8d;
    font-size: 1.1em;
    margin: 0;
}

.budget-card .card-icon { color: #3498db; }
.actual-card .card-icon { color: #27ae60; }
.variance-card.positive .card-icon { color: #27ae60; }
.variance-card.negative .card-icon { color: #e74c3c; }
.utilization-card .card-icon { color: #9b59b6; }

.report-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.report-section h2 {
    color: #2c3e50;
    margin-bottom: 20px;
    border-bottom: 2px solid #e9ecef;
    padding-bottom: 10px;
}

.table-container {
    overflow-x: auto;
}

.budget-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
}

.budget-table th,
.budget-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.budget-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
    position: sticky;
    top: 0;
}

.budget-table tr:hover {
    background: #f8f9fa;
}

.positive {
    color: #27ae60;
    font-weight: 600;
}

.negative {
    color: #e74c3c;
    font-weight: 600;
}

.status-badge {
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 0.8em;
    font-weight: bold;
    text-transform: uppercase;
}

.status-badge.excellent { background: #d4edda; color: #155724; }
.status-badge.good { background: #d1ecf1; color: #0c5460; }
.status-badge.warning { background: #fff3cd; color: #856404; }
.status-badge.danger { background: #f8d7da; color: #721c24; }
.status-badge.critical { background: #f5c6cb; color: #721c24; }

.chart-container {
    margin: 20px 0;
    text-align: center;
}

.recommendations {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.recommendation {
    padding: 15px;
    border-radius: 8px;
    border-left: 4px solid;
}

.recommendation.success { background: #d4edda; border-color: #28a745; }
.recommendation.good { background: #d1ecf1; border-color: #17a2b8; }
.recommendation.warning { background: #fff3cd; border-color: #ffc107; }
.recommendation.danger { background: #f8d7da; border-color: #dc3545; }
.recommendation.critical { background: #f5c6cb; border-color: #dc3545; }
.recommendation.info { background: #cce7ff; border-color: #0066cc; }

.error-message {
    background: #f8d7da;
    color: #721c24;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #f5c6cb;
}

.no-data {
    text-align: center;
    color: #6c757d;
    font-style: italic;
    padding: 40px;
}

@media (max-width: 768px) {
    .filter-form {
        flex-direction: column;
        align-items: stretch;
    }

    .filter-group {
        min-width: auto;
    }

    .summary-cards {
        grid-template-columns: 1fr;
    }

    .card {
        padding: 20px;
    }
}
</style>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// Pass PHP data to JavaScript for printing
const budgetData = {
    total_revenue: <?php echo $total_revenue; ?>,
    total_budget: <?php echo $total_budget; ?>,
    total_actual: <?php echo $total_actual; ?>,
    total_variance: <?php echo $total_variance; ?>,
    total_variance_percentage: <?php echo $total_variance_percentage; ?>,
    budget_data: <?php echo json_encode($budget_data); ?>,
    academic_year: '<?php echo htmlspecialchars($academic_years[array_search($academic_year_id, array_column($academic_years, 'id'))]['year'] ?? 'All Years'); ?>',
    category: '<?php echo htmlspecialchars($budget_categories[array_search($category_id, array_column($budget_categories, 'id'))]['name'] ?? 'All Categories'); ?>',
    report_period: '<?php echo ucfirst($report_period); ?>',
    school_settings: {
        school_name: '<?php echo htmlspecialchars($school_settings['school_name'] ?? 'School Name'); ?>',
        address: '<?php echo htmlspecialchars($school_settings['address'] ?? 'School Address'); ?>',
        phone: '<?php echo htmlspecialchars($school_settings['phone'] ?? 'School Phone'); ?>',
        email: '<?php echo htmlspecialchars($school_settings['email'] ?? 'School Email'); ?>'
    }
};

document.addEventListener('DOMContentLoaded', function() {
    // Budget Utilization Chart
    const utilizationCtx = document.getElementById('utilizationChart').getContext('2d');
    const categoryNames = <?php echo json_encode(array_column($category_utilization, 'category_name')); ?>;
    const utilizationData = <?php echo json_encode(array_column($category_utilization, 'utilization_percentage')); ?>;

    new Chart(utilizationCtx, {
        type: 'bar',
        data: {
            labels: categoryNames,
            datasets: [{
                label: 'Budget Utilization (%)',
                data: utilizationData,
                backgroundColor: 'rgba(52, 152, 219, 0.6)',
                borderColor: 'rgba(52, 152, 219, 1)',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    beginAtZero: true,
                    max: 100,
                    title: {
                        display: true,
                        text: 'Utilization (%)'
                    }
                },
                x: {
                    title: {
                        display: true,
                        text: 'Category'
                    }
                }
            },
            plugins: {
                legend: {
                    display: false
                }
            }
        }
    });
});

// Print Report Function
function printReport() {
    // Create a new window for printing
    var printWindow = window.open('', '_blank');

    // Get data from budgetData
    var totalRevenue = budgetData.total_revenue || 0;
    var totalBudget = budgetData.total_budget || 0;
    var totalActual = budgetData.total_actual || 0;
    var totalVariance = budgetData.total_variance || 0;
    var totalVariancePercentage = budgetData.total_variance_percentage || 0;
    var budgetDataRows = budgetData.budget_data || [];
    var schoolSettings = budgetData.school_settings || {};

    // Build the print content
    var printContent = `
        <!DOCTYPE html>
        <html>
        <head>
            <title>Budget vs Actual Report</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    margin: 20px;
                    line-height: 1.4;
                    color: #000;
                }
                .header {
                    text-align: center;
                    border-bottom: 2px solid #000;
                    padding-bottom: 20px;
                    margin-bottom: 30px;
                }
                .school-info {
                    margin-bottom: 20px;
                }
                .report-title {
                    font-size: 24px;
                    font-weight: bold;
                    margin: 20px 0;
                    color: #000;
                }
                .report-info {
                    margin-bottom: 20px;
                    font-size: 14px;
                }
                .summary-section {
                    display: flex;
                    justify-content: space-around;
                    margin-bottom: 30px;
                    padding: 20px;
                    background: #f8f9fa;
                    border: 1px solid #ddd;
                }
                .summary-item {
                    text-align: center;
                    flex: 1;
                    padding: 10px;
                }
                .summary-item h3 {
                    font-size: 24px;
                    margin: 10px 0;
                    color: #000;
                    font-weight: bold;
                }
                .summary-item p {
                    color: #000;
                    margin: 5px 0;
                    font-size: 12px;
                    font-weight: normal;
                }
                .revenue-display {
                    text-align: center;
                    margin-bottom: 30px;
                    padding: 20px;
                    border: 1px solid #ddd;
                }
                .revenue-content h2 {
                    font-size: 28px;
                    margin: 10px 0;
                    color: #000;
                    font-weight: bold;
                }
                .revenue-content p {
                    color: #000;
                    margin: 5px 0;
                    font-size: 14px;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 20px;
                    font-size: 12px;
                    border: 1px solid #000;
                }
                th, td {
                    border: 1px solid #000;
                    padding: 8px 12px;
                    text-align: left;
                }
                th {
                    background-color: #f8f9fa;
                    font-weight: bold;
                    color: #000;
                    border-bottom: 2px solid #000;
                }
                .positive {
                    color: #000;
                    font-weight: bold;
                }
                .negative {
                    color: #000;
                    font-weight: bold;
                }
                .status-badge {
                    display: inline-block;
                    padding: 2px 8px;
                    border-radius: 12px;
                    font-size: 10px;
                    font-weight: bold;
                    text-transform: uppercase;
                    border: 1px solid #000;
                }
                .status-excellent { background: #fff; color: #000; }
                .status-good { background: #fff; color: #000; }
                .status-warning { background: #fff; color: #000; }
                .status-danger { background: #fff; color: #000; }
                .status-critical { background: #fff; color: #000; }
                .footer {
                    margin-top: 40px;
                    text-align: center;
                    font-size: 12px;
                    color: #000;
                    border-top: 1px solid #000;
                    padding-top: 20px;
                }
                @media print {
                    body { margin: 0; }
                    .no-print { display: none; }
                }
            </style>
        </head>
        <body>
            <div class="header">
                <div class="school-info">
                    <h1>${schoolSettings.school_name || 'School Name'}</h1>
                    <p>${schoolSettings.address || 'School Address'}</p>
                    <p>Phone: ${schoolSettings.phone || 'School Phone'} | Email: ${schoolSettings.email || 'School Email'}</p>
                </div>
                <h2 class="report-title">Budget vs Actual Report</h2>
                <div class="report-info">
                    <p><strong>Academic Year:</strong> ${budgetData.academic_year}</p>
                    <p><strong>Category:</strong> ${budgetData.category}</p>
                    <p><strong>Report Period:</strong> ${budgetData.report_period}</p>
                    <p><strong>Generated on:</strong> ${new Date().toLocaleDateString()} ${new Date().toLocaleTimeString()}</p>
                </div>
            </div>

            <div class="revenue-display">
                <div class="revenue-content">
                    <h2>GHC ${totalRevenue.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</h2>
                    <p>Total Current Revenue (${new Date().getFullYear()})</p>
                </div>
            </div>

            <div class="summary-section">
                <div class="summary-item">
                    <h3>GHC ${totalBudget.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</h3>
                    <p>Total Budget</p>
                </div>
                <div class="summary-item">
                    <h3>GHC ${totalActual.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</h3>
                    <p>Actual Spending</p>
                </div>
                <div class="summary-item">
                    <h3>GHC ${Math.abs(totalVariance).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</h3>
                    <p>Variance (${totalVariance >= 0 ? 'Under Budget' : 'Over Budget'})</p>
                </div>
                <div class="summary-item">
                    <h3>${totalBudget > 0 ? ((totalActual / totalBudget) * 100).toFixed(1) : 0}%</h3>
                    <p>Budget Utilization</p>
                </div>
            </div>

            <table>
                <thead>
                    <tr>
                        <th>Category</th>
                        <th>Academic Year</th>
                        <th>Budget Amount</th>
                        <th>Actual Amount</th>
                        <th>Variance</th>
                        <th>Variance %</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>`;

    if (budgetDataRows.length === 0) {
        printContent += `
                    <tr>
                        <td colspan="7" style="text-align: center;">No budget data available for the selected criteria.</td>
                    </tr>`;
    } else {
        budgetDataRows.forEach(function(row) {
            var statusClass = '';
            var statusText = '';
            if (row.variance_percentage > 20) {
                statusClass = 'status-excellent';
                statusText = 'Excellent';
            } else if (row.variance_percentage > 5) {
                statusClass = 'status-good';
                statusText = 'Good';
            } else if (row.variance_percentage > -5) {
                statusClass = 'status-warning';
                statusText = 'On Track';
            } else if (row.variance_percentage > -20) {
                statusClass = 'status-danger';
                statusText = 'Over Budget';
            } else {
                statusClass = 'status-critical';
                statusText = 'Critical';
            }

            printContent += `
                    <tr>
                        <td>${row.category_name || ''}</td>
                        <td>${row.academic_year || ''}</td>
                        <td>GHC ${parseFloat(row.budget_amount || 0).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</td>
                        <td>GHC ${parseFloat(row.actual_amount || 0).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</td>
                        <td class="${parseFloat(row.variance || 0) >= 0 ? 'positive' : 'negative'}">
                            GHC ${Math.abs(parseFloat(row.variance || 0)).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}
                        </td>
                        <td class="${parseFloat(row.variance_percentage || 0) >= 0 ? 'positive' : 'negative'}">
                            ${parseFloat(row.variance_percentage || 0).toFixed(2)}%
                        </td>
                        <td>
                            <span class="status-badge ${statusClass}">${statusText}</span>
                        </td>
                    </tr>`;
        });
    }

    printContent += `
                </tbody>
            </table>

            <div class="footer">
                <p>This report was generated by the School Management System</p>
                <p>Report Date: ${new Date().toLocaleDateString()}</p>
            </div>
        </body>
        </html>`;

    // Write content to the new window
    printWindow.document.write(printContent);
    printWindow.document.close();

    // Wait for content to load then print
    printWindow.onload = function() {
        printWindow.print();
        printWindow.close();
    };
}
</script>

<?php
include '../includes/admin_footer.php';
?>
