<?php
require_once '../../config.php';
require_once '../../includes/access_control.php';

// Access control temporarily disabled for assets management page
// requireFeatureAccess('accounts_management');

header('Content-Type: application/json');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['asset_id'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Asset ID is required']);
    exit;
}

$asset_id = (int)$input['asset_id'];

if ($asset_id <= 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid asset ID']);
    exit;
}

try {
    // Check if asset exists
    $stmt = $pdo->prepare("SELECT id, name FROM assets WHERE id = ?");
    $stmt->execute([$asset_id]);
    $asset = $stmt->fetch();

    if (!$asset) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Asset not found']);
        exit;
    }

    // Start transaction for safe deletion
    $pdo->beginTransaction();

    // Delete the asset
    $stmt = $pdo->prepare("DELETE FROM assets WHERE id = ?");
    $stmt->execute([$asset_id]);

    // Log the deletion (if audit table exists)
    if (tableExists($pdo, 'asset_audit_logs')) {
        $stmt = $pdo->prepare("
            INSERT INTO asset_audit_logs (asset_id, action, old_values, performed_by, performed_at)
            VALUES (?, 'deleted', ?, ?, NOW())
        ");
        $stmt->execute([
            $asset_id,
            json_encode(['name' => $asset['name']]),
            $_SESSION['user_id'] ?? null
        ]);
    }

    // Commit transaction
    $pdo->commit();

    echo json_encode([
        'success' => true,
        'message' => 'Asset deleted successfully'
    ]);

} catch (PDOException $e) {
    // Rollback on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }

    error_log("Error deleting asset: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error deleting asset: ' . $e->getMessage()
    ]);
}

// Helper function to check if table exists
function tableExists($pdo, $table) {
    try {
        $result = $pdo->query("SELECT 1 FROM $table LIMIT 1");
        return true;
    } catch (Exception $e) {
        return false;
    }
}
?>
