<?php
ob_start();
require_once '../config.php';
require_once '../includes/access_control.php';
require_once 'asset_audit_functions.php';
require_once 'payment_methods.php';

// Access control temporarily disabled for assets management page
// requireFeatureAccess('accounts_management');

include 'includes/accounting_header.php';

$message = '';
$error = '';

// Handle success messages from URL parameters
if (isset($_GET['success'])) {
    if ($_GET['success'] === 'added' && isset($_GET['name'])) {
        $message = "Asset '" . htmlspecialchars($_GET['name']) . "' added successfully!";
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $acquisition_type = $_POST['acquisition_type'];
        $name = trim($_POST['name']);
        $description = trim($_POST['description']);
        $category_id = $_POST['category_id'];
        $purchase_date = $_POST['purchase_date'];
        $location = trim($_POST['location']);
        $status = $_POST['status'];
        $serial_number = trim($_POST['serial_number']);

        // Set payment method based on acquisition type
        $payment_method = ($acquisition_type === 'donated') ? 'donated' : $_POST['payment_method'];

        // Handle cost/value based on acquisition type
        if ($acquisition_type === 'purchased') {
            $purchase_cost = $_POST['purchase_cost'];
            $current_value = $purchase_cost; // For purchased assets, current value equals purchase cost

            if (empty($purchase_cost) || !is_numeric($purchase_cost)) {
                throw new Exception('Valid purchase cost is required for purchased assets');
            }
        } else { // donated
            $current_value = $_POST['current_value'];
            $purchase_cost = 0; // Donated assets have zero purchase cost

            if (empty($current_value) || !is_numeric($current_value)) {
                throw new Exception('Fair market value is required for donated assets');
            }
        }

        // Validation
        if (empty($acquisition_type)) {
            throw new Exception('Acquisition type is required');
        }
        if (empty($name)) {
            throw new Exception('Asset name is required');
        }
        if (empty($purchase_date)) {
            throw new Exception('Acquisition date is required');
        }

        // Get category defaults for depreciation
        $category_defaults = [];
        try {
            $stmt = $pdo->prepare("SELECT default_depreciation_method, default_depreciation_rate FROM asset_categories WHERE id = ?");
            $stmt->execute([$category_id]);
            $category_defaults = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
        } catch (PDOException $e) {
            // Continue with defaults if query fails
        }

        // Get additional form fields
        $depreciation_method = $category_defaults['default_depreciation_method'] ?? 'straight_line';
        $depreciation_rate = $category_defaults['default_depreciation_rate'] ?? 0;
        $salvage_value = $_POST['salvage_value'] ?? 0;
        $useful_life_years = $_POST['useful_life_years'] ?? 5;
        $useful_life_months = $_POST['useful_life_months'] ?? 0;
        $depreciation_frequency = $_POST['depreciation_frequency'] ?? 'monthly';
        $appreciation_method = $_POST['appreciation_method'] ?? 'none';
        $appreciation_rate = $_POST['appreciation_rate'] ?? 0;
        $insured_value = $_POST['insured_value'] ?? null;
        $insurance_policy_number = trim($_POST['insurance_policy_number'] ?? '');
        $insurance_expiry_date = $_POST['insurance_expiry_date'] ?? null;

        // Calculate depreciation dates
        $depreciation_start_date = $purchase_date;
        $depreciation_end_date = date('Y-m-d', strtotime("+$useful_life_years years +$useful_life_months months", strtotime($purchase_date)));

        // Insert asset with comprehensive fields
        $stmt = $pdo->prepare("
            INSERT INTO assets (
                name, description, category_id, purchase_date, purchase_cost, current_value,
                depreciation_method, depreciation_rate, salvage_value, useful_life_years, useful_life_months,
                depreciation_frequency, appreciation_method, appreciation_rate,
                depreciation_start_date, depreciation_end_date,
                insured_value, insurance_policy_number, insurance_expiry_date,
                location, status, serial_number
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $name, $description, $category_id, $purchase_date, $purchase_cost, $current_value,
            $depreciation_method, $depreciation_rate, $salvage_value, $useful_life_years, $useful_life_months,
            $depreciation_frequency, $appreciation_method, $appreciation_rate,
            $depreciation_start_date, $depreciation_end_date,
            $insured_value, $insurance_policy_number ?: null, $insurance_expiry_date,
            $location, $status, $serial_number
        ]);

        $asset_id = $pdo->lastInsertId();

        // Create journal entry for asset acquisition
        require_once 'accounting_functions.php';
        require_once 'create_asset_acquisition_function.php';
        try {
            // For donated assets, use current_value as the amount; for purchased, use purchase_cost
            $amount = ($payment_method === 'donated') ? $current_value : $purchase_cost;
            createAssetAcquisitionEntry(
                $pdo,
                $asset_id,
                $amount,
                $payment_method,
                $purchase_date,
                $_SESSION['user_id'] ?? 1
            );
        } catch (Exception $e) {
            // Log error but don't fail the asset creation
            error_log("Failed to create journal entry for asset $asset_id: " . $e->getMessage());
        }

        // Log the create action
        logAssetAction($pdo, $asset_id, $_SESSION['user_id'] ?? 1, 'create', null, [
            'name' => $name,
            'description' => $description,
            'category_id' => $category_id,
            'purchase_date' => $purchase_date,
            'purchase_cost' => $purchase_cost,
            'current_value' => $current_value,
            'location' => $location,
            'status' => $status,
            'serial_number' => $serial_number,
            'created_at' => date('Y-m-d H:i:s')
        ], []);

        // Redirect to prevent form resubmission on refresh
        header('Location: add_asset.php?success=added&name=' . urlencode($name));
        exit;
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Get categories for dropdown with defaults
try {
    $stmt = $pdo->query("
        SELECT id, name, default_depreciation_method, default_depreciation_rate,
               default_depreciation_frequency, default_appreciation_method,
               default_appreciation_rate, revaluation_allowed
        FROM asset_categories ORDER BY name
    ");
    $categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $categories = [];
}
?>

<style>
.add-asset-container {
    max-width: 800px;
    margin: 0 auto;
    padding: 0;
    background: transparent;
    min-height: auto;
    box-shadow: none;
}

.asset-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 30px;
    border-radius: 12px;
    margin-bottom: 30px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.asset-header h1 {
    margin: 0;
    font-size: 2.2em;
    font-weight: 300;
}

.asset-header p {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.form-container {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    padding: 30px;
    border: 1px solid #e1e5e9;
}

.form-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 25px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 8px;
    color: #495057;
    font-size: 0.95em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.form-group input,
.form-group select,
.form-group textarea {
    padding: 12px 16px;
    border: 2px solid #2E7D32;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.3s ease;
    font-family: inherit;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.form-group textarea {
    resize: vertical;
    min-height: 80px;
}

.form-actions {
    display: flex;
    gap: 15px;
    justify-content: center;
    margin-top: 30px;
    padding-top: 30px;
    border-top: 1px solid #dee2e6;
}

.btn {
    padding: 12px 30px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(46, 125, 50, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(46, 125, 50, 0.4);
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
}

.message {
    padding: 15px;
    margin-bottom: 20px;
    border-radius: 8px;
    font-weight: 600;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.required {
    color: #dc3545;
}

.depreciation-fields, .appreciation-fields {
    display: none;
}
</style>

<div class="add-asset-container">
    <div class="asset-header">
        <div>
            <h1>➕ Add New Asset</h1>
            <p>Record a new school asset in the system</p>
        </div>
        <div>
            <a href="manage_assets.php" class="btn btn-secondary">🏠 Back to Assets</a>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="form-container">
        <form method="post">
            <!-- Acquisition Type Section -->
            <div class="form-group" style="grid-column: 1 / -1; margin-bottom: 30px;">
                <label for="acquisition_type">Acquisition Type <span class="required">*</span></label>
                <select id="acquisition_type" name="acquisition_type" required>
                    <option value="">Select Acquisition Type</option>
                    <option value="purchased">Purchased</option>
                    <option value="donated">Donated</option>
                </select>
            </div>

            <div class="form-grid">
                <div class="form-group">
                    <label for="category_id">Category <span class="required">*</span></label>
                    <select id="category_id" name="category_id" required>
                        <option value="">Select Category</option>
                        <?php foreach ($categories as $category): ?>
                            <option value="<?php echo $category['id']; ?>">
                                <?php echo htmlspecialchars($category['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="name">Asset Name <span class="required">*</span></label>
                    <input type="text" id="name" name="name" required placeholder="e.g., Dell Laptop, Classroom Desk">
                </div>

                <div class="form-group">
                    <label for="purchase_date">Acquisition Date <span class="required">*</span></label>
                    <input type="date" id="purchase_date" name="purchase_date" required>
                </div>

                <!-- Purchased Asset Fields -->
                <div class="form-group purchased-fields">
                    <label for="purchase_cost">Purchase Cost (GHC) <span class="required">*</span></label>
                    <input type="number" id="purchase_cost" name="purchase_cost" step="0.01" min="0"
                           placeholder="0.00">
                </div>

                <div class="form-group purchased-fields">
                    <label for="payment_method">Payment Method <span class="required">*</span></label>
                    <select id="payment_method" name="payment_method">
                        <?php
                        $paymentOptions = PaymentMethods::getFormOptions(['donated']); // Exclude donated as it's handled separately
                        foreach ($paymentOptions as $option) {
                            echo '<option value="' . htmlspecialchars($option['value']) . '">' . htmlspecialchars($option['label']) . '</option>';
                        }
                        ?>
                    </select>
                </div>

                <!-- Donated Asset Fields -->
                <div class="form-group donated-fields" style="display: none;">
                    <label for="current_value">Fair Market Value (GHC) <span class="required">*</span></label>
                    <input type="number" id="current_value" name="current_value" step="0.01" min="0"
                           placeholder="Enter the fair market value of the donated asset">
                </div>



                <div class="form-group">
                    <label for="salvage_value">Salvage Value (GHC)</label>
                    <input type="number" id="salvage_value" name="salvage_value" step="0.01" min="0"
                           placeholder="Residual value at end of useful life">
                </div>

                <div class="form-group">
                    <label for="useful_life_years">Useful Life (Years)</label>
                    <input type="number" id="useful_life_years" name="useful_life_years" min="1" max="50" value="5">
                </div>

                <div class="form-group">
                    <label for="useful_life_months">Additional Months</label>
                    <input type="number" id="useful_life_months" name="useful_life_months" min="0" max="11" value="0">
                </div>

                <!-- Depreciation Settings -->
                <div class="form-group depreciation-fields" id="depreciation_method_group">
                    <label for="depreciation_method">Depreciation Method</label>
                    <select id="depreciation_method" name="depreciation_method" disabled>
                        <option value="straight_line">Straight Line</option>
                        <option value="declining_balance">Declining Balance</option>
                        <option value="units_of_production">Units of Production</option>
                    </select>
                </div>

                <div class="form-group depreciation-fields" id="depreciation_rate_group">
                    <label for="depreciation_rate">Depreciation Rate (%)</label>
                    <input type="number" id="depreciation_rate" name="depreciation_rate" step="0.01" min="0" max="100"
                           placeholder="e.g., 10.00 for 10% annual depreciation" readonly>
                </div>

                <div class="form-group depreciation-fields" id="depreciation_frequency_group">
                    <label for="depreciation_frequency">Depreciation Frequency</label>
                    <select id="depreciation_frequency" name="depreciation_frequency" disabled>
                        <option value="monthly">Monthly</option>
                        <option value="quarterly">Quarterly</option>
                        <option value="annually">Annually</option>
                    </select>
                </div>

                <!-- Appreciation Settings -->
                <div class="form-group appreciation-fields" id="appreciation_method_group">
                    <label for="appreciation_method">Appreciation Method</label>
                    <select id="appreciation_method" name="appreciation_method" disabled>
                        <option value="none">None</option>
                        <option value="fixed_percentage">Fixed Percentage</option>
                        <option value="compound">Compound</option>
                        <option value="market_based">Market Based</option>
                    </select>
                </div>

                <div class="form-group appreciation-fields" id="appreciation_rate_group">
                    <label for="appreciation_rate">Appreciation Rate (% per year)</label>
                    <input type="number" id="appreciation_rate" name="appreciation_rate" step="0.01" min="0" max="100"
                           placeholder="e.g., 5.00 for 5% appreciation per year" readonly>
                </div>

                <!-- Insurance Information -->
                <div class="form-group">
                    <label for="insured_value">Insured Value (GHC)</label>
                    <input type="number" id="insured_value" name="insured_value" step="0.01" min="0"
                           placeholder="Insurance coverage amount">
                </div>

                <div class="form-group">
                    <label for="insurance_policy_number">Insurance Policy Number</label>
                    <input type="text" id="insurance_policy_number" name="insurance_policy_number"
                           placeholder="Policy number">
                </div>

                <div class="form-group">
                    <label for="insurance_expiry_date">Insurance Expiry Date</label>
                    <input type="date" id="insurance_expiry_date" name="insurance_expiry_date">
                </div>

                <div class="form-group">
                    <label for="status">Status</label>
                    <select id="status" name="status">
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                        <option value="maintenance">Maintenance</option>
                        <option value="leased">Leased</option>
                        <option value="sold">Sold</option>
                        <option value="disposed">Disposed</option>
                        <option value="lost">Lost</option>
                        <option value="stolen">Stolen</option>
                        <option value="donated">Donated</option>
                    </select>
                </div>

                <div class="form-group">
                    <label for="location">Location</label>
                    <input type="text" id="location" name="location"
                           placeholder="e.g., Room 101, Admin Office">
                </div>

                <div class="form-group">
                    <label for="serial_number">Serial Number</label>
                    <input type="text" id="serial_number" name="serial_number"
                           placeholder="Manufacturer serial number">
                </div>
            </div>

            <div class="form-group" style="margin-top: 25px;">
                <label for="description">Description</label>
                <textarea id="description" name="description" rows="4"
                          placeholder="Additional details about the asset..."></textarea>
            </div>

            <div class="form-actions">
                <button type="submit" class="btn btn-primary">💾 Save Asset</button>
                <a href="manage_assets.php" class="btn btn-secondary">❌ Cancel</a>
            </div>
        </form>
    </div>
</div>

<script>
// Category defaults data
const categoryDefaults = <?php echo json_encode(array_column($categories, null, 'id')); ?>;

// Function to enable/disable category-based fields
function toggleCategoryFields(readonly) {
    const fields = ['depreciation_method', 'depreciation_rate', 'depreciation_frequency', 'appreciation_method', 'appreciation_rate'];
    fields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) {
            if (field.tagName === 'SELECT') {
                field.disabled = readonly;
            } else {
                field.readOnly = readonly;
            }
            field.style.backgroundColor = readonly ? '#f8f9fa' : '';
            field.style.opacity = readonly ? '0.7' : '1';
        }
    });
}

// Function to show/hide field groups based on category
function toggleFieldGroups(categoryId) {
    const depreciationGroups = ['depreciation_method_group', 'depreciation_rate_group', 'depreciation_frequency_group'];
    const appreciationGroups = ['appreciation_method_group', 'appreciation_rate_group'];

    if (categoryId && categoryDefaults[categoryId]) {
        const defaults = categoryDefaults[categoryId];

        // Show depreciation fields if category has depreciation method
        if (defaults.default_depreciation_method) {
            depreciationGroups.forEach(id => {
                document.getElementById(id).style.display = 'block';
            });
        } else {
            depreciationGroups.forEach(id => {
                document.getElementById(id).style.display = 'none';
            });
        }

        // Show appreciation fields if category has appreciation method
        if (defaults.default_appreciation_method && defaults.default_appreciation_method !== 'none') {
            appreciationGroups.forEach(id => {
                document.getElementById(id).style.display = 'block';
            });
        } else {
            appreciationGroups.forEach(id => {
                document.getElementById(id).style.display = 'none';
            });
        }
    } else {
        // Hide all category-based fields if no category selected
        depreciationGroups.forEach(id => {
            document.getElementById(id).style.display = 'none';
        });
        appreciationGroups.forEach(id => {
            document.getElementById(id).style.display = 'none';
        });
    }
}

// Auto-fill fields based on category selection
document.getElementById('category_id').addEventListener('change', function() {
    const categoryId = this.value;
    if (categoryId && categoryDefaults[categoryId]) {
        const defaults = categoryDefaults[categoryId];

        // Set depreciation method and rate
        if (defaults.default_depreciation_method) {
            document.getElementById('depreciation_method').value = defaults.default_depreciation_method;
        }
        if (defaults.default_depreciation_rate) {
            document.getElementById('depreciation_rate').value = defaults.default_depreciation_rate;
        }
        if (defaults.default_depreciation_frequency) {
            document.getElementById('depreciation_frequency').value = defaults.default_depreciation_frequency;
        }

        // Set appreciation method and rate
        if (defaults.default_appreciation_method) {
            document.getElementById('appreciation_method').value = defaults.default_appreciation_method;
        }
        if (defaults.default_appreciation_rate) {
            document.getElementById('appreciation_rate').value = defaults.default_appreciation_rate;
        }

        // Show/hide field groups based on category
        toggleFieldGroups(categoryId);

        // Make fields readonly since they are auto-filled from category
        toggleCategoryFields(true);
    } else {
        // Clear fields
        document.getElementById('depreciation_method').value = 'straight_line';
        document.getElementById('depreciation_rate').value = '';
        document.getElementById('depreciation_frequency').value = 'monthly';
        document.getElementById('appreciation_method').value = 'none';
        document.getElementById('appreciation_rate').value = '';

        // Hide all category-based fields
        toggleFieldGroups(null);

        // Keep fields readonly
        toggleCategoryFields(true);
    }
});

// Auto-fill current value when purchase cost changes
document.getElementById('purchase_cost').addEventListener('input', function() {
    const currentValue = document.getElementById('current_value');
    if (!currentValue.value) {
        currentValue.value = this.value;
    }
});

// Set default date to today
document.getElementById('purchase_date').valueAsDate = new Date();

// Auto-set payment method to "donated" when status is "donated"
document.getElementById('status').addEventListener('change', function() {
    const paymentMethod = document.getElementById('payment_method');
    if (this.value === 'donated') {
        paymentMethod.value = 'donated';
        paymentMethod.disabled = true;
    } else {
        paymentMethod.disabled = false;
    }
});

// Toggle form fields based on acquisition type
document.getElementById('acquisition_type').addEventListener('change', function() {
    const acquisitionType = this.value;
    const purchasedFields = document.querySelectorAll('.purchased-fields');
    const donatedFields = document.querySelectorAll('.donated-fields');
    const paymentMethod = document.getElementById('payment_method');

    if (acquisitionType === 'purchased') {
        // Show purchased fields, hide donated fields
        purchasedFields.forEach(field => field.style.display = 'block');
        donatedFields.forEach(field => field.style.display = 'none');
        // Make payment method required for purchased assets
        paymentMethod.required = true;
    } else if (acquisitionType === 'donated') {
        // Hide purchased fields, show donated fields
        purchasedFields.forEach(field => field.style.display = 'none');
        donatedFields.forEach(field => field.style.display = 'block');
        // Set payment method to donated and make it not required
        paymentMethod.value = 'donated';
        paymentMethod.required = false;
    } else {
        // Hide all conditional fields if no type selected
        purchasedFields.forEach(field => field.style.display = 'none');
        donatedFields.forEach(field => field.style.display = 'none');
        paymentMethod.required = false;
    }
});

// Initialize fields as readonly (default values from category)
toggleCategoryFields(true);
</script>

<?php include 'includes/accounting_footer.php'; ?>
